/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "GMCStation.h"
#include "PachubeClient.h"
#include "SuperTweetClient.h"

int main() {
    Ether ether = Ether::create("config.txt");
    ether.setup();
    GMCounter gmc = GMCounter::create(p17, p22, p23, "config.txt"); // gmc, buzzer, led
    Location location = Location::create("config.txt");
    NTPClient ntpClient = NTPClient::create("config.txt");
    TextLCD lcd(p24, p26, p27, p28, p29, p30); // rs, e, d4-d7
    PachubeClient datastream = PachubeClient::create("config.txt");
    SuperTweetClient supertweet = SuperTweetClient::create("config.txt");
    GMCStation station(1234, gmc, location, ntpClient, lcd);
    Timer timer;

    station.display("   GMCStation", "    ver 1.0");
    wait(5);

    if (ether.isActive()) {
        char ipAddress[17];
        char *ip = ether.getIpAddress();
        sprintf(ipAddress, "%d.%d.%d.%d", ip[0], ip[1], ip[2], ip[3]);
        station.display(ipAddress, "Port = 1234");
    } else {
        station.display("Network error");
        error("network error\n");
    }

    wait(5);
    
    if (!ntpClient.setTime()) {
        station.display("No RTC server", "available");
        wait(5);
    }
    
    char message[17] = {};
    while (true) {
        timer.reset();
        timer.start();
        station.handleClient();

        time_t seconds = time(NULL) + 9 * 3600;
        char timestamp[16];
        strftime(timestamp, sizeof(timestamp), "%m/%d %X", localtime(&seconds));
        float cpm = gmc.getAverageCPM();
        float uSv = gmc.getAverageRadiation();

        // update pachube every minute
        if (seconds % 60 == 0) {
            datastream.add(0, uSv);
            datastream.add(1, cpm);
            bool result = datastream.update();
            printf("Pachube datastream update %s\n", result ? "succeeded." : "failed.");
            sprintf(message, "Pachube %s", result ? "updated" : "failed");
        }

        // tweet every hour
        if (seconds % 3600 == 2) {
            bool result = supertweet.tweet("", cpm, uSv);
            printf("Tweet CPM & uSv %s\n", result ? "succeeded." : "failed.");
            sprintf(message, "Tweet %s", result ? "succeeded" : "failed");
        }

        // adjust RTC every 0.5 hours (for my injured mbed)
        if (seconds % 1800 == 4) {
            bool result = ntpClient.setTime();
            printf("RTC %s\n", result ? "adjusted." : "not adjusted.");
            sprintf(message, "RTC %s", result ? "adjusted" : "not adjusted");
        }
        
        if (!message[0]) // if no message written
            sprintf(message, "%3.1f, %4.3fuSv", cpm, uSv);
        
        station.display(timestamp, message);
        printf("%s: %3.1f, %4.3fuSv\n", timestamp, cpm, uSv);

        float duration = 1 - timer.read();
        if (duration > 0) wait(duration);
        
        if (seconds % 2 == 1) // clear message every other second
            message[0] = 0;
    }
}
