/*
Copyright (c) 2011, Senio Networks, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#ifndef GMCSTATION_H
#define GMCSTATION_H

#include "Ether.h"
#include "Station.h"
#include "GMCounter.h"
#include "Location.h"
#include "NTPClient.h"
#include "TextLCD.h"

class GMCStation : public Station {
public:
    /**
     * Creates a GMCStation server
     *
     * @param port GMCStation server prot number
     * @param gmc reference to a GMCounter instance
     * @param location reference to a Location instance
     * @param ntpClient reference to an NTPClient instance
     * @param lcd reference to a TextLCD instance
     */
    GMCStation(int port, GMCounter& gmc, Location& location, NTPClient ntpClient, TextLCD& lcd)
            : Station(port), gmc(gmc), location(location), ntpClient(ntpClient), lcd(lcd) {
        setBuffer(buf, sizeof(buf));
        addHandler("GetCPM", this, &GMCStation::getCPM);
        addHandler("GetAverageCPM", this, &GMCStation::getAverageCPM);
        addHandler("GetRadiation", this, &GMCStation::getRadiation);
        addHandler("GetAverageRadiation", this, &GMCStation::getAverageRadiation);
        addHandler("ToggleBuzzer", this, &GMCStation::toggleBuzzer);
        addHandler("ToggleLED", this, &GMCStation::toggleLED);
        addHandler("GetLongitude", this, &GMCStation::getLongitude);
        addHandler("GetLatitude", this, &GMCStation::getLatitude);
        addHandler("GetElevation", this, &GMCStation::getElevation);
        addHandler("GetTime", this, &GMCStation::getTime);
        addHandler("SetTime", this, &GMCStation::setTime);
        addHandler("AdjustTime", this, &GMCStation::adjustTime);
    }

    /**
     * displays message on the attached LCD
     *
     * @param message1 1st line of the message buffer
     * @param message2 2nd line of the message buffer
     */
    void display(char *message1, char *message2 = 0) {
        lcd.cls();
        lcd.printf("%s", message1);
        if (message2) {
            lcd.locate(0, 1); //column, row
            lcd.printf("%s", message2);
        }
    }

private:
    GMCounter& gmc;
    Location& location;
    NTPClient& ntpClient;
    TextLCD& lcd;
    char buf[128];

    void getCPM() {
        sprintf(buf, "GetCPM = %d\n", gmc.getCPM());
    }

    void getAverageCPM() {
        sprintf(buf, "GetAverageCPM = %3.1f\n", gmc.getAverageCPM());
    }
    void getRadiation() {
        sprintf(buf, "GetRadiation = %f\n", gmc.getRadiation());
    }

    void getAverageRadiation() {
        sprintf(buf, "GetAverageRadiation = %f\n", gmc.getAverageRadiation());
    }

    void toggleBuzzer() {
        bool enabled = !gmc.getBuzzer();
        gmc.setBuzzer(enabled);
        sprintf(buf, "ToggleBuzzer = %s\n", enabled ? "ON" : "OFF");
    }

    void toggleLED() {
        bool enabled = !gmc.getLED();
        gmc.setLED(enabled);
        sprintf(buf, "ToggleLED = %s\n", enabled ? "ON" : "OFF");
    }

    void getLongitude() {
        sprintf(buf, "GetLongitude = %10.7f\n", location.getLongitude());
    }

    void getLatitude() {
        sprintf(buf, "GetLatitude = %10.7f\n", location.getLatitude());
    }

    void getElevation() {
        sprintf(buf, "GetElevation = %.1f\n", location.getElevation());
    }

    void getTime() {
        time_t seconds = time(NULL) + 9 * 3600;
        strftime(buf, sizeof(buf), "GetTime = %F %X\n", localtime(&seconds));
    }

    void setTime() {
        struct tm t = {};
        sscanf(buf, "SetTime %d-%d-%d %d:%d:%d", &t.tm_year, &t.tm_mon, &t.tm_mday, &t.tm_hour, &t.tm_min, &t.tm_sec);
        t.tm_year -= 1900;
        t.tm_mon -= 1;
        time_t seconds = mktime(&t) - 9 * 3600;
        set_time(seconds);
        strftime(buf, sizeof(buf), "SetTime = %F %X\n", localtime(&seconds));
    }

    void adjustTime() {
        if (ntpClient.setTime()) {
            time_t seconds = time(NULL) + 9 * 3600;
            strftime(buf, sizeof(buf), "AdjustTime = %F %X\n", localtime(&seconds));
        } else {
            sprintf(buf, "AdjustTime failed\n");
        }
    }
};

#endif