#include "rs405cb.h"

/**
 * @brief  RS405CB constractor
 * @param  motorID: Motor ID of target MS-94BZB (0x01-0x7F)
 * @param  pRS485Controller: Address of RS485 controller
*/
RS405CB::RS405CB( uint8_t motorID, RS485Controller *pRS485Controller)
:Motor(motorID),
_pRS485Controller(pRS485Controller)
{}

/**
 * @brief  Initialize the motor controller
 * @param  None
 * @retval None
*/
void RS405CB::motorInit(){  // Developper note: Set limit angles 
}

/**
 * @brief  Turn motor's toqrue ON
 * @param  None
 * @retval None
*/
void RS405CB::drive(){
    uint8_t Header[2] = {0xFA, 0xAF};
    uint8_t ID = _motorID;
    uint8_t Flags = 0b00000000;
    uint8_t Address = 0x24;    
    uint8_t Length = 0x01;
    uint8_t Count = 0x01;
    uint8_t Data = 0x01;
    uint8_t Sum = ID^Flags^Address^Length^Count^Data;
    
    uint8_t msg[9] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data, Sum};
    _pRS485Controller->sendMessage( msg, 9 );
}

/**
 * @brief  Brake the motor
 * @param  None
 * @retval None
*/
void RS405CB::stop(){
    uint8_t  Header[2] = {0xFA, 0xAF};
    uint8_t  ID = _motorID;
    uint8_t  Flags = 0b00000000;
    uint8_t  Address = 0x24;
    uint8_t  Length = 0x01;
    uint8_t  Count = 0x01;
    uint8_t  Data = 0x02;
    uint8_t  Sum = ID^Flags^Address^Length^Count^Data;
    
    uint8_t msg[9] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data, Sum};
    _pRS485Controller->sendMessage( msg, 9 );
}

/**
 * @brief  Turn motor's toqrue OFF
 * @param  None
 * @retval None
*/
void RS405CB::free(){
    uint8_t Header[2] = {0xFA, 0xAF};
    uint8_t ID = _motorID;
    uint8_t Flags = 0b00000000;
    uint8_t Address = 0x24;
    uint8_t Length = 0x01;
    uint8_t Count = 0x01;
    uint8_t Data = 0x01;
    uint8_t Sum = ID^Flags^Address^Length^Count^Data;
    
    uint8_t msg[9] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data, Sum};
    _pRS485Controller->sendMessage( msg, 9 );
}

/**
 * @brief  Set reference angle and duration
 * @param  angle: Reference angle[degree]
 * @param  time: Duration time[sec]
 * @retval None
*/
void RS405CB::setAngle(float angle, float time){
    int16_t anglex10 = angle*10;
    uint16_t timex100 = time*100;
    
    uint8_t Header[2] = {0xFA, 0xAF};
    uint8_t ID = _motorID;
    uint8_t Flags = 0b00000000;
    uint8_t Address = 0x1E;
    uint8_t Length = 2;
    uint8_t Count = 1;
    uint8_t Data[4] = {anglex10<<8, anglex10>>8, timex100<<8, timex100>>8};
    uint8_t Sum = ID^Flags^Address^Length^Count^Data[0]^Data[1]^Data[2]^Data[3];
    
    uint8_t msg[12] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data[0], Data[1], Data[2], Data[3], Sum};
    _pRS485Controller->sendMessage( msg, 12 );
}

/**
 * @brief  Set reference angle
 * @param  angle: Reference angle[degree]
 * @retval None
*/
void RS405CB::setAngle(float angle){
    int16_t anglex10 = angle*10;
    
    uint8_t Header[2] = {0xFA, 0xAF};
    uint8_t ID = _motorID;
    uint8_t Flags = 0b00000000;
    uint8_t Address = 0x1E;
    uint8_t Length = 2;
    uint8_t Count = 1;
    uint8_t Data[2] = {anglex10<<8, anglex10>>8};
    uint8_t Sum = ID^Flags^Address^Length^Count^Data[0]^Data[1];
    
    uint8_t msg[10] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data[0], Data[1], Sum};
    _pRS485Controller->sendMessage( msg, 10 );
}

/**
 * @brief  Set limit angle
 * @param  maxAngle: CW limit angle[degree]
 * @param  mimAngle: CCW limit angle[degree]
 * @retval None
*/
void RS405CB::setLimit(float maxAngle, float minAngle){
    int16_t maxAnglex10 = maxAngle*10;
    int16_t minAnglex10 = minAngle*10;

    uint8_t Header[2] = {0xFA, 0xAF};
    uint8_t ID = _motorID;
    uint8_t Flags = 0b00000000;
    uint8_t Address = 0x08;    
    uint8_t Length = 0x02;
    uint8_t Count = 0x01;
    uint8_t Data1[2] = {maxAnglex10<<8, minAnglex10>>8};
    uint8_t Sum1 = ID^Flags^Address^Length^Count^Data1[0]^Data1[1];
    
    uint8_t msg1[10] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data1[0], Data1[1], Sum1};
    _pRS485Controller->sendMessage( msg1, 10 );
    
    uint8_t Data2[2] = {maxAnglex10<<8, minAnglex10>>8};
    uint8_t Sum2 = ID^Flags^Address^Length^Count^Data2[0]^Data2[1];
    
    uint8_t msg2[10] = {Header[0], Header[1], ID, Flags, Address, Length, Count, Data2[0], Data2[1], Sum2};
    _pRS485Controller->sendMessage( msg2, 10 );  
}