#include "plc_controller.h"
#include "XNucleoPLC01A1.h"

/**
 * @brief  Initialize PLC
 * @param  None
*/
PLCController::PLCController()
:_spi(X_NUCLEO_PLC01A1_PIN_SPI_MOSI, X_NUCLEO_PLC01A1_PIN_SPI_MISO, X_NUCLEO_PLC01A1_PIN_SPI_SCLK),
_plc(X_NUCLEO_PLC01A1_PIN_SPI_CS1, X_NUCLEO_PLC01A1_PIN_SPI_CS2, X_NUCLEO_PLC01A1_PIN_OUT_EN, _spi)
{
    _inputArray[0] = _inputArray[1] = _outputArray[0] = _outputArray[1] = 0x00;
    /* Set spi configuration. */
    _spi.format(X_NUCLEO_PLC01A1_PIN_SPI_BITS, 0);
    _spi.frequency(1E6);
}

/**
 * @brief  Set output signal
 * @param  pinNum: Target pin number
 * @param  state: Pin state HIGH:1 or LOW:0
 * @retval None
*/
void PLCController::output(uint8_t pinNum, bool state){
    if( (PINMIN <= pinNum) || (pinNum <= PINMAX) ){
        /* Update HIGH or LOW pins */
        if(state){
            _outputArray[1] |= (0b00000001 << (pinNum-1));
        }else{
            _outputArray[1] &= ~(0b00000001 << (pinNum-1));
        }
    }
    /* Parity bits calculation */
    _plc.output_parity_bits(_outputArray);
    /* Send output information to solid state relay */
    _plc.plc_output().ssrelay_set_output(_outputArray);
}
    
/**
 * @brief  Get input signal
 * @param  pinNum: Target pin number
 * @retval Pin state HIGH:1 or LOW:0
*/
bool PLCController::input(uint8_t pinNum){
    _plc.plc_input().set_read_status(1);
    /* Wait until the read status is TRUE */
    while(~_plc.plc_input().get_read_status());
    _plc.plc_input().set_read_status(0);
    
    if( (PINMIN <= pinNum) || (pinNum <= PINMAX) ){
        /* Receive input data from Digital Input Termination Device */
        _plc.plc_input().dig_inp_array_get_input(_inputArray);
        return (bool)( _inputArray[1] & (0b00000001 << (pinNum-1)) );
    }else{
        return 0;
    }
}