#include "wheel.h"

/**
 * @brief  Constractor of Wheel class
 * @param  pCanController: Address of CAN controller
 * @param  pPlcController: Address of PLC controller
*/
Wheel::Wheel(CANController *pCanController, PLCController *pPlcController )
:_leftWheel( lWheelID, pCanController ), _rightWheel( rWheelID, pCanController ),
_brake(brakePin, pPlcController),
_brakeState(0)
{}

/**
 * @brief  Initialize wheels
 * @param  None
 * @retval None
*/
void Wheel::init(){
    _leftWheel.motorInit();
    _rightWheel.motorInit();
}

/**
 * @brief  Drive wheels
 * @param  None
 * @retval None
*/
void Wheel::drive(){
    if( _brakeState == 1 ){
        _brake = 0;
        _brakeState = 0;
        wait(0.5);
    }
    _leftWheel.drive();
    _rightWheel.drive();
}

/**
 * @brief  Stop wheels
 * @param  None
 * @retval None
*/
void Wheel::stop(){
    _leftWheel.stop();
    _rightWheel.stop();
}

/**
 * @brief  Sets angular velocities of wheels
 * @param  lVelocity: Anguglar velocity of the left wheel [rpm]
 * @param  rVelocity: Anguglar velocity of the right wheel  [rpm]
 * @retval None
*/
void Wheel::setVelocity( float lVelocity, float rVelocity ){
    _leftWheel.setAngularVelocity( lVelocity );
    _rightWheel.setAngularVelocity( rVelocity );
}

/**
 * @brief  Stop wheels and Turn on the wheels' brake
 * @param  None
 * @retval None
*/
void Wheel::brake(){
    Wheel::stop();
    wait(0.1);
    _brake = 1;
    _brakeState = 1;
}

float Wheel::getBatteryVoltage(){
    return _leftWheel.getInputVoltage()*0.1;
}