#include "mbed.h"

#define OV7670_I2C_ADDR 0x42

/*
 * Basic window sizes.  These probably belong somewhere more globally
 * useful.
 */
#define VGA_WIDTH       640
#define VGA_HEIGHT      480
#define QVGA_WIDTH      320
#define QVGA_HEIGHT     240
#define CIF_WIDTH       352
#define CIF_HEIGHT      288
#define QCIF_WIDTH      176
#define QCIF_HEIGHT     144

/* Register */
#define REG_GAIN        0x00    /* Gain lower 8 bits (rest in vref) */
#define REG_BLUE        0x01    /* blue gain */
#define REG_RED         0x02    /* red gain */
#define REG_VREF        0x03    /* Pieces of GAIN, VSTART, VSTOP */
#define REG_COM1        0x04    /* Control 1 */
#define  COM1_CCIR656     0x40  /* CCIR656 enable */
#define REG_BAVE        0x05    /* U/B Average level */
#define REG_GbAVE       0x06    /* Y/Gb Average level */
#define REG_AECHH       0x07    /* AEC MS 5 bits */
#define REG_RAVE        0x08    /* V/R Average level */
#define REG_COM2        0x09    /* Control 2 */
#define  COM2_SSLEEP      0x10  /* Soft sleep mode */
#define REG_PID         0x0a    /* Product ID MSB */
#define REG_VER         0x0b    /* Product ID LSB */
#define REG_COM3        0x0c    /* Control 3 */
#define  COM3_SWAP        0x40    /* Byte swap */
#define  COM3_SCALEEN     0x08    /* Enable scaling */
#define  COM3_DCWEN       0x04    /* Enable downsamp/crop/window */
#define REG_COM4        0x0d    /* Control 4 */
#define REG_COM5        0x0e    /* All "reserved" */
#define REG_COM6        0x0f    /* Control 6 */
#define REG_AECH        0x10    /* More bits of AEC value */
#define REG_CLKRC       0x11    /* Clocl control */
#define   CLK_EXT         0x40    /* Use external clock directly */
#define   CLK_SCALE       0x3f    /* Mask for internal clock scale */
#define REG_COM7        0x12    /* Control 7 */
#define   COM7_RESET      0x80    /* Register reset */
#define   COM7_FMT_MASK   0x38
#define   COM7_FMT_VGA    0x00
#define   COM7_FMT_CIF    0x20    /* CIF format */
#define   COM7_FMT_QVGA   0x10    /* QVGA format */
#define   COM7_FMT_QCIF   0x08    /* QCIF format */
#define   COM7_RGB        0x04    /* bits 0 and 2 - RGB format */
#define   COM7_YUV        0x00    /* YUV */
#define   COM7_BAYER      0x01    /* Bayer format */
#define   COM7_PBAYER     0x05    /* "Processed bayer" */
#define REG_COM8        0x13    /* Control 8 */
#define   COM8_FASTAEC    0x80    /* Enable fast AGC/AEC */
#define   COM8_AECSTEP    0x40    /* Unlimited AEC step size */
#define   COM8_BFILT      0x20    /* Band filter enable */
#define   COM8_AGC        0x04    /* Auto gain enable */
#define   COM8_AWB        0x02    /* White balance enable */
#define   COM8_AEC        0x01    /* Auto exposure enable */
#define REG_COM9        0x14    /* Control 9  - gain ceiling */
#define REG_COM10       0x15    /* Control 10 */
#define   COM10_HSYNC     0x40    /* HSYNC instead of HREF */
#define   COM10_PCLK_HB   0x20    /* Suppress PCLK on horiz blank */
#define   COM10_HREF_REV  0x08    /* Reverse HREF */
#define   COM10_VS_LEAD   0x04    /* VSYNC on clock leading edge */
#define   COM10_VS_NEG    0x02    /* VSYNC negative */
#define   COM10_HS_NEG    0x01    /* HSYNC negative */
#define REG_HSTART      0x17    /* Horiz start high bits */
#define REG_HSTOP       0x18    /* Horiz stop high bits */
#define REG_VSTART      0x19    /* Vert start high bits */
#define REG_VSTOP       0x1a    /* Vert stop high bits */
#define REG_PSHFT       0x1b    /* Pixel delay after HREF */
#define REG_MIDH        0x1c    /* Manuf. ID high */
#define REG_MIDL        0x1d    /* Manuf. ID low */
#define REG_MVFP        0x1e    /* Mirror / vflip */
#define   MVFP_MIRROR     0x20    /* Mirror image */
#define   MVFP_FLIP       0x10    /* Vertical flip */
 
#define REG_AEW         0x24    /* AGC upper limit */
#define REG_AEB         0x25    /* AGC lower limit */
#define REG_VPT         0x26    /* AGC/AEC fast mode op region */
#define REG_HSYST       0x30    /* HSYNC rising edge delay */
#define REG_HSYEN       0x31    /* HSYNC falling edge delay */
#define REG_HREF        0x32    /* HREF pieces */
#define REG_TSLB        0x3a    /* lots of stuff */
#define   TSLB_YLAST      0x04    /* UYVY or VYUY - see com13 */
#define REG_COM11       0x3b    /* Control 11 */
#define   COM11_NIGHT     0x80    /* NIght mode enable */
#define   COM11_NMFR      0x60    /* Two bit NM frame rate */
#define   COM11_HZAUTO    0x10    /* Auto detect 50/60 Hz */
#define   COM11_50HZ      0x08    /* Manual 50Hz select */
#define   COM11_EXP       0x02
#define REG_COM12       0x3c    /* Control 12 */
#define   COM12_HREF      0x80    /* HREF always */
#define REG_COM13       0x3d    /* Control 13 */
#define   COM13_GAMMA     0x80    /* Gamma enable */
#define   COM13_UVSAT     0x40    /* UV saturation auto adjustment */
#define   COM13_UVSWAP    0x01    /* V before U - w/TSLB */
#define REG_COM14       0x3e    /* Control 14 */
#define   COM14_DCWEN     0x10    /* DCW/PCLK-scale enable */
#define REG_EDGE        0x3f    /* Edge enhancement factor */
#define REG_COM15       0x40    /* Control 15 */
#define   COM15_R10F0     0x00    /* Data range 10 to F0 */
#define   COM15_R01FE     0x80    /*            01 to FE */
#define   COM15_R00FF     0xc0    /*            00 to FF */
#define   COM15_RGB565    0x10    /* RGB565 output */
#define   COM15_RGB555    0x30    /* RGB555 output */
#define REG_COM16       0x41    /* Control 16 */
#define   COM16_AWBGAIN   0x08    /* AWB gain enable */
#define REG_COM17       0x42    /* Control 17 */
#define   COM17_AECWIN    0xc0    /* AEC window - must match COM4 */
#define   COM17_CBAR      0x08    /* DSP Color bar */

/*
 * This matrix defines how the colors are generated, must be
 * tweaked to adjust hue and saturation.
 *
 * Order: v-red, v-green, v-blue, u-red, u-green, u-blue
 *
 * They are nine-bit signed quantities, with the sign bit
 * stored in 0x58.  Sign for v-red is bit 0, and up from there.
 */
#define REG_CMATRIX_BASE 0x4f
#define   CMATRIX_LEN 6
#define REG_CMATRIX_SIGN 0x58


#define REG_BRIGHT      0x55    /* Brightness */
#define REG_CONTRAS     0x56    /* Contrast control */
 
#define REG_GFIX        0x69    /* Fix gain control */

#define REG_REG76       0x76    /* OV's name */
#define   R76_BLKPCOR     0x80    /* Black pixel correction enable */
#define   R76_WHTPCOR     0x40    /* White pixel correction enable */
 
#define REG_RGB444      0x8c    /* RGB 444 control */
#define   R444_ENABLE     0x02    /* Turn on RGB444, overrides 5x5 */
#define   R444_RGBX       0x01    /* Empty nibble at end */
 
#define REG_HAECC1      0x9f    /* Hist AEC/AGC control 1 */
#define REG_HAECC2      0xa0    /* Hist AEC/AGC control 2 */

#define REG_BD50MAX     0xa5    /* 50hz banding step limit */
#define REG_HAECC3      0xa6    /* Hist AEC/AGC control 3 */
#define REG_HAECC4      0xa7    /* Hist AEC/AGC control 4 */
#define REG_HAECC5      0xa8    /* Hist AEC/AGC control 5 */
#define REG_HAECC6      0xa9    /* Hist AEC/AGC control 6 */
#define REG_HAECC7      0xaa    /* Hist AEC/AGC control 7 */
#define REG_BD60MAX     0xab    /* 60hz banding step limit */

struct regval_list {
        unsigned char reg_num;
        unsigned char value;
};

static struct regval_list ov7670_default_regs[] = {
        { REG_COM7, COM7_RESET },
/*
 * Clock scale: 3 = 15fps
 *              2 = 20fps
 *              1 = 30fps
 */
        { REG_CLKRC, 1 },     /* OV: clock scale (30 fps), internal clk= external clk/(1+1) */
        { REG_TSLB,  0x04 },    /* OV */
        { REG_COM7, 0 },        /* VGA,YUV */
        /*
         * Set the hardware window.  These values from OV don't entirely
         * make sense - hstop is less than hstart.  But they work...
         */
        { REG_HSTART, 0x13 },   { REG_HSTOP, 0x01 },
        { REG_HREF, 0xb6 },     { REG_VSTART, 0x02 },
        { REG_VSTOP, 0x7a },    { REG_VREF, 0x0a },

        { REG_COM3, 0 },        { REG_COM14, 0 },
        /* Mystery scaling numbers */
        { 0x70, 0x3a }, //Bit[7]:Test Pattern
                        //Bit[6:0]:Horizontal scale factor
        { 0x71, 0x35 }, //Bit[7]:Test Pattern (/w 0x70)
                        //Bit[6:0]:Vertical scale factor
        { 0x72, 0x11 }, //DCW control
                        //Bit[7] Vertical average calculation option
                        // 0:Vertical truncation
                        // 1:Vertical rounding
                        //Bit[6] Vertical down sampling option
                        // 0:Vertical truncation
                        // 1:Vertical rounding
                        //Bit[5:4] Vertical down sampling rate
                        // 00:No down sampling, 01: by2, 10:by4, 11:by8
                        //Bit[3] Horizontal averagecalculation option
                        // 0: Horizontal truncation
                        // 1: Horizontal rounding
                        //Bit[2] Horizontal down sampling option
                        // 0: Horizontal truncation
                        // 1: Horizontal rounding
                        //Bit[1:0] Horizontal down sampling rate
                        // 00: No down sampling, 01: by2, 10:by4, 11:by8 
        { 0x73, 0xf0 }, //Bit[7:4] Reserved (Original : 0b1111)
                        //Bit[3] Bypass clock divider for DSP scale control
                        // 0: Enable clock divider, 1:Bypass clock divider
                        // Bit[2:0] Clock divider control for DSP scale control
                        // 000: Divided by 1, 001:Divided by 2, 010:Divided by4
                        // 100: Divided by 8, 101:Divided by 16, 110-111:Not allowed
        { 0xa2, 0x02 }, //Pixel Clock Delay
                        //Bit[7]: Reserved (original:0)
                        //Bit[6:0] Scaling output delay (original :0x02)
        { REG_COM10, 0x20 },//Common Control 10 (original :0x00)
                           //Bit[7] Reserved
                           //Bit[6] HREF changed to HSYNC
                           //Bit[5] PCLK output option (0: Free running PCLK, 1: PCLK does not toggle during horizontal blank)
                           //Bit[4] PCLK reverse
                           //Bit[3] HREF reverse
                           //Bit[2] VSYNC option (0: VSYNC changes on falling edge of PCLK, 1:rising edge)
                           //Bit[1] VSYNC negative
                           //Bit[0] HSYNC negative

        /* Gamma curve values */
        { 0x7a, 0x20 },         { 0x7b, 0x10 },
        { 0x7c, 0x1e },         { 0x7d, 0x35 },
        { 0x7e, 0x5a },         { 0x7f, 0x69 },
        { 0x80, 0x76 },         { 0x81, 0x80 },
        { 0x82, 0x88 },         { 0x83, 0x8f },
        { 0x84, 0x96 },         { 0x85, 0xa3 },
        { 0x86, 0xaf },         { 0x87, 0xc4 },
        { 0x88, 0xd7 },         { 0x89, 0xe8 },

        /* AGC and AEC parameters.  Note we start by disabling those features,
           then turn them only after tweaking the values. */
        { REG_COM8, COM8_FASTAEC | COM8_AECSTEP | COM8_BFILT },
        // Common control 8 (for AGC, AEC Off)
        { REG_GAIN, 0 },
        { REG_AECH, 0x04 }, //AEC[9:2]
        { 0x07, 0x00 }, //Bit[7:6] Reserved
                        //Bit[5:0] AEC[15:10]
        { 0x04, 0x00 }, //Common control 1
                        //Bit[7] Reserved
                        //Bit[6] CCIR format (0:Disable, 1:Enable)
                        //Bit[5:2] Reserved
                        //Bit[1:0] AEC[1:0]
        { REG_COM4, 0x40 }, /* magic reserved bit */
                            //Bit[7:6] Reserved (Original 0b01)
                            //Bit[5:4] Average option(must be same value as COM17[7:6])(Original 0b00)
                            //Bit[3:0] Reserved (Original 0b0000)
        { REG_COM9, 0x18 }, /* 4x gain + magic rsvd bit */
        { REG_BD50MAX, 0x05 },  { REG_BD60MAX, 0x07 },
        { REG_AEW, 0x95 }, //AGC/AEC - Stable Operating Region(Upper limit)
        { REG_AEB, 0x33 }, //AGC/AEC - Stable Operating Region(Lower limit)
        { REG_VPT, 0xe3 }, //AGC/AEC Fast mode Operating Region
                           //Bit[7:4]: High nibble of upper limit of fast mode control zone
                           //Bit[3:0]: High nibble of lower limit of fast mode control zone 
        { REG_HAECC1, 0x78 },// Histgram based AEC/AGC Control 1 (Original : 0x78, Spec init:0xC0)
        { REG_HAECC2, 0x68 },// Histgram based AEC/AGC Control 2 (Original : 0x68, Spec init:0x90)
        { 0xa1, 0x03 }, /* magic */
        { REG_HAECC3, 0xd8 },// Histgram based AEC/AGC Control 3 (Original : 0xd8, Spec init:0xF0)
        { REG_HAECC4, 0xd8 },// Histgram based AEC/AGC Control 4 (Original : 0xd8, Spec init:0xC1)
        { REG_HAECC5, 0xf0 },// Histgram based AEC/AGC Control 5 (Original : 0xf0, Spec init:0xf0)
        { REG_HAECC6, 0x90 },// Histgram based AEC/AGC Control 6 (Original : 0x90, Spec init:0xC1)
        { REG_HAECC7, 0x94 },// Histgram based AEC/AGC Control 7 (Original : 0x94, Spec init:0x14)
                             //Bit[7]: AEC algorithm selection(0:Average-based, 1:Histogram-based)
                             //Bit[6:0] Reserved
        { REG_COM8, COM8_FASTAEC|COM8_AECSTEP|COM8_BFILT|COM8_AGC|COM8_AEC },
        //COM8_FASTAEC    0x80    /* Enable fast AGC/AEC */
        //COM8_AECSTEP    0x40    /* Unlimited AEC step size */
        //COM8_BFILT      0x20    /* Band filter enable */
        //COM8_AGC        0x04    /* Auto gain enable */
        //COM8_AWB        0x02    /* White balance enable */
        //COM8_AEC        0x01    /* Auto exposure enable */
        /* Almost all of these are magic "reserved" values.  */
        { REG_COM5, 0x61 }, //Bit[7:0] Reserved
        { REG_COM6, 0x4b }, //Bit[7]: Output optical black line output 0:Disable,1:Enable(original:0)
                            //Bit[6:2] Reserved (0b10010)
                            //Bit[1] Reset all timing when format changes 0:No reset,1:Reset (original:1)
                            //Bit[0] Reserved (1)
        { 0x16, 0x02 },     //Reserved
        { REG_MVFP, 0x03 }, //Bit[7:6] Reserved (original 0b00)
                            //Bit[5] Mirror  0:Normal, 1:Mirror (original 0)
                            //Bit[4] VFlip   0:Normal, 1:Vertically flip (origianl 0)
                            //Bit[3] Reserved (original 0)
                            //Bit[2] Black sun enable (original 1)
                            //Bit[1:0] Reserved (original 0b11)
        { 0x21, 0x02 },         { 0x22, 0x91 },
        { 0x29, 0x07 },         { 0x33, 0x0b },
        { 0x35, 0x0b },         { 0x37, 0x1d },
        { 0x38, 0x71 },         { 0x39, 0x2a },
        { REG_COM12, 0x68 },    { 0x4d, 0x40 },
        { 0x4e, 0x20 },         { REG_GFIX, 0 },
        { 0x6b, 0xca }, //Bit[7:6] PLL control Original:11(x8) 
        { 0x74, 0x10 }, //Bit[4]: Digital gain contrl: 1:by VREF[7:6],0:by REG74[1:0]
        { 0x8d, 0x4f },         { 0x8e, 0 },
        { 0x8f, 0 },            { 0x90, 0 },
        { 0x91, 0 },            { 0x96, 0 },
        { 0x9a, 0 },            { 0xb0, 0x84 },
        { 0xb1, 0x0c },         { 0xb2, 0x0e },
        { 0xb3, 0x82 },         { 0xb8, 0x0a },

        /* More reserved magic, some of which tweaks white balance */
        { 0x43, 0x0a },         { 0x44, 0xf0 },
        { 0x45, 0x34 },         { 0x46, 0x58 },
        { 0x47, 0x28 },         { 0x48, 0x3a },
        { 0x59, 0x88 },         { 0x5a, 0x88 },
        { 0x5b, 0x44 },         { 0x5c, 0x67 },
        { 0x5d, 0x49 },         { 0x5e, 0x0e },
        { 0x6c, 0x0a },         { 0x6d, 0x55 },
        { 0x6e, 0x11 },         { 0x6f, 0x9f }, /* "9e for advance AWB" */
        { 0x6a, 0x40 },         { REG_BLUE, 0x40 },
        { REG_RED, 0x60 },
        //{ REG_COM8, COM8_FASTAEC|COM8_AECSTEP|COM8_BFILT|COM8_AGC|COM8_AEC|COM8_AWB },
        { REG_COM8, COM8_FASTAEC|COM8_AECSTEP|COM8_BFILT|COM8_AGC|COM8_AWB },
        /* Matrix coefficients */ //default value
        { 0x4f, 0x80 },         { 0x50, 0x80 },
        { 0x51, 0 },            { 0x52, 0x22 },
        { 0x53, 0x5e },         { 0x54, 0x80 },
        { 0x58, 0x9e },

        { REG_COM16, COM16_AWBGAIN },
        { REG_EDGE, 0x00 }, //Bit[7:5]: Reserved
                         //Bit[4:0]: Edge enchandement factor (original: 0x00)
        { 0x75, 0x05 }, //Bit[7:5] : reserved, Bit[4:0] : Edge Enhancement lower limit (original:0x05)
        { 0x76, 0xff }, //Bit[7]:Black correction enable 0:Disable, 1:Enable(original:1)
                        //Bit[6]:White pixel correction enable: 0:Disable, 1:Enable(original:1)
                        //Bit[5]:Reserved (original:0b10)
                        //Bit[4:0]: Edge Enhancement higher limit(original:0x01)
        { 0x4c, 0 },    //De-noise Setength
        { 0x77, 0x00 }, //De-noise offset (original 0x01)
        { REG_COM13, 0xc3 }, //Common Control 13
                             //Bit[7] Gamma enable
                             //Bit[6] UV saturation level
                             //Bit[5:1] Reserved
                             //Bit[0] UV swap 
        { 0x4b, 0x09 },      //Bit[7:1] Reserved
                             //Bit[0] UV average enable
        { 0xc9, 0x60 },      //Saturation Control
                             //Bit[7:4] UV saturation control min
                             //Bit[3:0] UV saturation control result      
        { REG_COM16, 0x38 }, //Common control 16 (Original 0b00111000)
                             //Bit[7:6] Reserved (Original 0b00)
                             //Bit[5] Enable edge enhancement threshold auto-adjustment for YUV output (Original 1)
                             //Bit[4] De-noise threshold auto-adjustment (0:Disable, 1:Enable)(Original 1)
                             //Bit[3] AWB gain enable (Original 1)
                             //Bit[2] Reserved
                             //Bit[1] Color matrix coefficient double option (0: Original, 1:Double)
                             //Bit[0] Reserved (Original 0)
        { 0x56, 0x40 },     //Martix Coefficient Sign for Coefficient 5 to 0
                            //Bit[7] Auto contrast center enable
                            //Bit[6] Reserved
                            //Bit[5:0] Matrix coefficient sign (0:Plus, 1:Minus)
        { 0x34, 0x11 },     //Array Reference Control Bit[7:0] Reserved
        { REG_COM11, COM11_EXP|COM11_HZAUTO }, //Common control 11
                                               //Bit[7] Night mode (0:disable, 1:Enable)
                                               //Bit[6:5] Minimum frame rate of night mode
                                               //00:same, 01:1/2, 10:1/4, 11:1/8
                                               //Bit[4] D56_Auto (0:Disable, 1:Enable)
                                               //Bit[3] Banding filter value select
                                               // 0: Select BD60ST, 1:Select BD50ST
                                               //Bit[2] Reserved
                                               //Bit[1] Exprosure timing can be less than limit of banding filter 
                                               //Bit[0] Reserved
        { 0xa4, 0x88 },         { 0x96, 0 },
        { 0x97, 0x30 },         { 0x98, 0x20 },
        { 0x99, 0x30 },         { 0x9a, 0x84 },
        { 0x9b, 0x29 },         { 0x9c, 0x03 },
        { 0x9d, 0x4c },         { 0x9e, 0x3f },
        { 0x78, 0x04 },

        /* Extra-weird stuff.  Some sort of multiplexor register */
        { 0x79, 0x01 },         { 0xc8, 0xf0 },
        { 0x79, 0x0f },         { 0xc8, 0x00 },
        { 0x79, 0x10 },         { 0xc8, 0x7e },
        { 0x79, 0x0a },         { 0xc8, 0x80 },
        { 0x79, 0x0b },         { 0xc8, 0x01 },
        { 0x79, 0x0c },         { 0xc8, 0x0f },
        { 0x79, 0x0d },         { 0xc8, 0x20 },
        { 0x79, 0x09 },         { 0xc8, 0x80 },
        { 0x79, 0x02 },         { 0xc8, 0xc0 },
        { 0x79, 0x03 },         { 0xc8, 0x40 },
        { 0x79, 0x05 },         { 0xc8, 0x30 },
        { 0x79, 0x26 },
        
        /* Added register value */
        //{ 0x30, 0x00 }, //REG_HSYSET (HSYNC Rising Edge Delay : 08 -> 00
        //{ 0x2A, 0x00 },  // Dummy Pixel insert MSB
        //{ 0x2B, 0x00 },  // Dummy Pixel insert MSB
        { 0x15, 0x05 }, // Common Control 10 : Bit[4] 1:PCLK negative
        { 0x3A, 0x05 }, //Bit[7:6]: Reserved (origina:0b00)
                        //Bit[5]: Negative Image 0:Normal, 1:Negative (original:0)
                        //Bit[4]: UV output value 0:Normal, 1:User define (original:0)
                        //Bit[3]: Output sequence (original:1)
                        //Bit[2:1]:reserved
                        //Bit[0]: Auto output window

        { 0xff, 0xff }, /* END MARKER */
};

static struct regval_list ov7670_fmt_rgb565[] = {
        { REG_COM7, COM7_FMT_QVGA|COM7_RGB }, /* Selects RGB mode */
        { REG_RGB444, 0 },      /* No RGB444 please */
        { REG_COM1, 0x0 },
        { REG_COM15,  0xc0 | COM15_RGB565 },
        { REG_COM9, 0x28 },     /* 16x gain ceiling; 0x8 is reserved bit */
        { 0x4f, 0xb3 },         /* "matrix coefficient 1" */
        { 0x50, 0xb3 },         /* "matrix coefficient 2" */
        { 0x51, 0    },         /* vb */
        { 0x52, 0x3d },         /* "matrix coefficient 4" */
        { 0x53, 0xa7 },         /* "matrix coefficient 5" */
        { 0x54, 0xe4 },         /* "matrix coefficient 6" */
        { REG_COM13, COM13_GAMMA|COM13_UVSAT },
        { 0xff, 0xff },
};

// QVGA setting
const int width = QVGA_WIDTH;
const int height= QVGA_HEIGHT;
const int com7_bit= COM7_FMT_QVGA;
const int hstart= 164;          /* Empirically determined */
const int hstop=  20; //Original code: 20
const int vstart=  14;
const int vstop= 494;

