/*

    CC3000 OSC message receiver

    Sparkfun CC3000 WiFi Shield meets mbed!
       
    This program supports the following mbed boards with Sparkfun CC3000 Sheild.
    '#define' switches board.(refer to source code)
    (1)FRDM-KL25Z (KL25Z)
    (2)ST Nucleo F401RE (STM32F401)
    (3)ST Nucleo F030R8 (STM32F030)
    (4)LPCXpresso1549 (LPC1549)
    (5)Seeduino-Arch-Pro (ARCH_PRO)

 Please push RESET button on CC3000 WiFi Shield to start the program.

  reference:  
    https://www.sparkfun.com/products/12071 for CC300 Shield
    http://opensoundcontrol.org/introduction-osc for OSC(Open Sound Control)

  date: 2014/8/31
  written by: xshige
  
console out sample:
----------------------------------
C3000 OSC receiver demo.
IP address: 192.168.0.2

Wait for packet...
incomming OSC msg:
/1 ,
Received packet from: 192.168.0.23
-------------------

Wait for packet...
incomming OSC msg:
/1/fader3 ,f 0.565401
Received packet from: 192.168.0.23
-------------------

Wait for packet...
incomming OSC msg:
/1/fader3 ,f 0.565401
Received packet from: 192.168.0.23
-------------------

Wait for packet...
incomming OSC msg:
/4 ,
Received packet from: 192.168.0.23
-------------------

Wait for packet...
incomming OSC msg:
/3 ,
Received packet from: 192.168.0.23
-------------------

Wait for packet...
incomming OSC msg:
/3/xy ,ff 0.604869 0.505618
Received packet from: 192.168.0.23
-------------------

Wait for packet...
----------------------------------  
*/

#include "mbed.h"
#include "cc3000.h"

// define board you like (KL25Z, LPC1549, STM32F401, STM32F030 ...)
//#define KL25Z
#define STM32F401
//#define STM32F030
//#define LPC1549
//#define ARCH_PRO

// define SSID, PASSWORD you like
#define SSID "ssid"
#define PASSWORD "password"

#include "UDPSocket.h"

using namespace mbed_cc3000;

/* cc3000 module declaration specific for user's board. */
#if defined(KL25Z)
// for KL25Z
cc3000 wifi(PTD4, PTC9, PTD0, SPI(PTD2, PTD3, PTD1), SSID, PASSWORD, WPA2, false);
Serial pc(USBTX, USBRX);
#endif
#if defined(STM32F401)
// for Nucleo STM32F401
cc3000 wifi(PA_10, PA_8, PB_6, SPI(PA_7, PA_6, PA_5), SSID, PASSWORD, WPA2, false);
// for Nucleo STM32F401
Serial pc(SERIAL_TX, SERIAL_RX);
#endif
#if defined(STM32F030)
// for Nucleo STM32F030
cc3000 wifi(PA_10, PA_8, PB_6, SPI(PA_7, PA_6, PA_5), SSID, PASSWORD, WPA2, false);
// for Nucleo STM32F030
Serial pc(SERIAL_TX, SERIAL_RX);
#endif
#if defined(LPC1549)
// for LPC1549
cc3000 wifi(P0_29, P0_0, P0_27, SPI(P0_28, P0_12, P0_16), SSID, PASSWORD, WPA2, false);
Serial pc(USBTX, USBRX);
#endif
#if defined(ARCH_PRO)
// for Seeed Studio Arch Pro
cc3000 wifi(P0_4, P2_5, P0_6, SPI(P0_9, P0_8, P0_7), SSID, PASSWORD, WPA2, false);
Serial pc(USBTX, USBRX);
#endif

// OSC related
#include "OSCmsgCodec.h"

#define INCOMMING_PORT 8000

#define PACKET_SIZE 512
#define RECBUF_SIZE 256
#define PERIOD_BUZZER 10

union OSCarg msg[10];
char recbuf[RECBUF_SIZE],buff[PACKET_SIZE],packet[PACKET_SIZE];

int main() {
    pc.baud(115200);

    printf("\n\nCC3000 OSC receiver demo. \n");
    wifi.init();
    if (wifi.connect() == -1) {
        printf("Failed to connect. Please verify connection details and try again. \n");
    } else {
        printf("IP address: %s \n", wifi.getIPAddress());
    }
    
    UDPSocket sock;
    sock.bind(INCOMMING_PORT);
    
    Endpoint receiver;

    while (true) {
        printf("\nWait for packet...\n");
        int n = sock.receiveFrom(receiver, recbuf, sizeof(recbuf));
        if (n > 0) {
            decOSCmsg(recbuf, msg);
            pc.printf("incomming OSC msg:\n%s %s", msg[0].address, msg[1].typeTag);
            for(int m=0; m < (strlen(msg[1].typeTag)-1); m++) {
                if (msg[1].typeTag[m+1]=='f') pc.printf(" %f",msg[m+2].f);
                if (msg[1].typeTag[m+1]=='i') pc.printf(" %d",msg[m+2].i);                
            }
            pc.printf("\n");
            printf("Received packet from: %s \n", receiver.get_address());
            printf("-------------------\n");
        };
    } // while
}
