/**
 * The following task checks the digital inputs on pin11 and pin12
 * it also checks for "errors" and save the error code to the state.
 * If pin12 is 1, it schedules the counting task.
 *
 * Author: Jacob Baungard Hansen
 */

#include "check_task.h"

/**
 *
 * @param starting_offset       when the task should but run first
 * @param frequency_ms          the frequency in ms of how often the task should run
 */
CheckTask::CheckTask(int starting_offset, int frequency_ms,
                                                                     State * state, TaskManager * tm)
                                    : Task(starting_offset, frequency_ms) {
                                        
    this->state = state;
    this->tm = tm;
    this->counting_task = new CountingTask(0, 1800);
    this->counting_scheduled = 0;
    this->digital_in_1 = new DigitalIn(p11);
    this->digital_in_2 = new DigitalIn(p12);
    
}

/*
 * Simple destructor
 */
CheckTask::~CheckTask() {
    delete this->counting_task;
    delete this->digital_in_1;
    delete this->digital_in_2;
}

/** 
 * The function that is run at the specified frequency.
 */
void CheckTask::action() {
    
    // get the digital values
    int switch_1 = digital_in_1->read();
    int switch_2 = digital_in_2->read();
    
    // save to the state
    this->state->set_digital_1(switch_1);
    this->state->set_digital_2(switch_2);

    // check possible error codes
    if (switch_1 == 1 && (this->state->get_avg_analog_1() > this->state->get_avg_analog_2() ) ) {
        this->state->set_error(3);
    } else {
        this->state->set_error(0);
    }
    
    // schedule or deschedule counting task as appropriate
    if (switch_2 == 1 && this->counting_scheduled == 0) {
        // schedule binary LED task
        this->counting_scheduled = 1;
        // set the binary LED task next run
        int next_run = this->get_next_run_ms()+300;
        this->counting_task->set_next_run_ms(next_run);
        // add task to task manager
        this->tm->add_task(this->counting_task);
    } else if (switch_2 == 0) {
        // delete task from task manager
        this->tm->remove_task(this->counting_task);
        this->counting_scheduled = 0;
    }

}