#include "MbedAgent.h"
#include "watchdog.h"
#include "logging.h"

MbedAgent::MbedAgent(DeviceInfo& deviceInfo):
    client(), tpl(), _bootstrap(deviceInfo), _integration(client, tpl, deviceInfo),
    lcdThirdLineBlank(true), signal(deviceInfo), temp(), poti(), gps(),
    acc(), sock(), pool(), _operationSupport(tpl, pool)
{
    reporters[0] = &ConfigSync::inst();
    reporters[1] = &temp;
    reporters[2] = &signal;
    reporters[3] = &gps;
    reporters[4] = &acc;
    reporters[5] = &poti;
}

bool MbedAgent::init()
{
    bool flag = true;

    // Insert measurement Acceleration
    // USAGE: 106,<DEVICE/ID>,<X>,<Y>,<Z>
    if (!tpl.add("10,106,POST,/measurement/measurements,application/vnd.com.nsn.cumulocity.measurement+json,application/vnd.com.nsn.cumulocity.measurement+json,%%,NOW UNSIGNED NUMBER NUMBER NUMBER,\"{\"\"time\"\":\"\"%%\"\",\"\"source\"\":{\"\"id\"\":\"\"%%\"\"},\"\"type\"\":\"\"c8y_MotionMeasurement\"\",\"\"c8y_MotionMeasurement\"\":{\"\"x\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"m/s^2\"\"},\"\"y\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"m/s^2\"\"},\"\"z\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"m/s^2\"\"}}}\"\r\n"))
        flag = false;
    
    // Insert measurement Potentiometer
    // USAGE: 107,<DEVICE/ID>,<ANALOG1>,<ANALOG2>
    if (!tpl.add("10,107,POST,/measurement/measurements,application/vnd.com.nsn.cumulocity.measurement+json,application/vnd.com.nsn.cumulocity.measurement+json,%%,NOW UNSIGNED NUMBER NUMBER,\"{\"\"time\"\":\"\"%%\"\",\"\"source\"\":{\"\"id\"\":\"\"%%\"\"},\"\"type\"\":\"\"c8y_AnalogMeasurement\"\",\"\"c8y_AnalogMeasurement\"\":{\"\"A1\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"A\"\"},\"\"A2\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"A\"\"}}}\"\r\n"))
        flag = false;
    
    // Update device position
    // USAGE: 108,<DEVICE/ID>,<ALTITUDE>,<LATITUDE>,<LONGITUDE>
    if (!tpl.add("10,108,PUT,/inventory/managedObjects/%%,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,UNSIGNED NUMBER NUMBER NUMBER,\"{\"\"c8y_Position\"\":{\"\"alt\"\":%%,\"\"lat\"\":%%,\"\"lng\"\":%%},\"\"c8y_MotionTracking\"\":{\"\"active\"\":true}}\"\r\n"))
        flag = false;

    // Insert measurement Location
    // USAGE: 109,<DEVICE/ID>,<ALTITUDE>,<LATITUDE>,<LONGITUDE>
    if (!tpl.add("10,109,POST,/event/events,application/vnd.com.nsn.cumulocity.event+json,application/vnd.com.nsn.cumulocity.event+json,%%,NOW UNSIGNED NUMBER NUMBER NUMBER,\"{\"\"time\"\":\"\"%%\"\",\"\"source\"\":{\"\"id\"\":\"\"%%\"\"},\"\"type\"\":\"\"c8y_LocationUpdate\"\",\"\"text\"\":\"\"Mbed location update\"\",\"\"c8y_Position\"\":{\"\"alt\"\":%%,\"\"lat\"\":%%,\"\"lng\"\":%%}}\"\r\n"))
        flag = false;

    // Insert measurement Signal Quality
    // USAGE: 104,<DEVICE/ID>,<RSSI>,<BER>
    if (!tpl.add("10,104,POST,/measurement/measurements,application/vnd.com.nsn.cumulocity.measurement+json,application/vnd.com.nsn.cumulocity.measurement+json,%%,NOW UNSIGNED NUMBER UNSIGNED,\"{\"\"time\"\":\"\"%%\"\",\"\"source\"\":{\"\"id\"\":\"\"%%\"\"},\"\"type\"\":\"\"c8y_SignalStrength\"\",\"\"c8y_SignalStrength\"\":{\"\"rssi\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"dBm\"\"},\"\"ber\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"%\"\"}}}\"\r\n"))
        flag = false;
        
    // Insert measurement Temperature
    // USAGE: 105,<DEVICE/ID>,<TEMPERATURE>
    if (!tpl.add("10,105,POST,/measurement/measurements,application/vnd.com.nsn.cumulocity.measurement+json,application/vnd.com.nsn.cumulocity.measurement+json,%%,NOW UNSIGNED NUMBER,\"{\"\"time\"\":\"\"%%\"\",\"\"source\"\":{\"\"id\"\":\"\"%%\"\"},\"\"type\"\":\"\"c8y_TemperatureMeasurement\"\",\"\"c8y_TemperatureMeasurement\"\":{\"\"T\"\":{\"\"value\"\":%%,\"\"unit\"\":\"\"C\"\"}}}\"\r\n"))
        flag = false;

    // Update Configuration
    // Usage: 130,<DEVICE/ID>,<CONFIG/STRING>,<RESPONSIBILITY>
    if (!tpl.add("10,130,PUT,/inventory/managedObjects/%%,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,UNSIGNED STRING NUMBER,\"{\"\"c8y_Configuration\"\":{\"\"config\"\":\"\"%%\"\"},\"\"c8y_RequiredAvailability\"\":{ \"\"responseInterval\"\":%%}}\"\r\n"))
        flag = false;

    if (!_integration.init()) {
        LCDDisplay::inst().setLines("Integrate init fail");
        flag = false;
    }
    if (!_operationSupport.init()) {
        LCDDisplay::inst().setLines("Operation init fail");
        flag = false;
    }
    for (size_t i = 0; i < N; ++i) {
        if (!reporters[i]->init()) {
            aError("Init %s", reporters[i]->name());
        }
    }
   return flag;
}

int MbedAgent::run()
{
    // device bootstrapping process
    if (!_bootstrap.bootstrap()) {
        LCDDisplay::inst().setLines("Bootstrap Failure");
        return -1;
    }

    Thread::wait(2000);
    LCDDisplay::inst().setLines("Connect to Cloud", srHost);
    setX_ID(UBLOX_SMARTREST_VERSION);
    if (!_integration.integrate()) {
        LCDDisplay::inst().setLines("Integrate failure");
        return -2;
    }
    setX_ID(client.getIdentifier());
    aInfo("Set X-ID: %s\n", srX_ID);

    return 0;
}

void MbedAgent::loop()
{
    ReportThread reportThread(pool);
    _operationSupport.executePendingOperations();
    PollThread pollThread(pool);
    pollThread.setChannel(deviceID);

    Watchdog wdt;
    wdt.kick(300.0);    // set a 60.0 seconds watchdog
    while (true) {
        int l = 0;
        bool flag = false;
        for (size_t i = 0; i < N; ++i) {
//            if (reporters[i] == &ConfigSync::inst()) {
            int l2 = reporters[i]->read(buf2+l, sizeof(buf2)-l, status, DISPLAY_LEN);
            if (l2 < 0) {
                flag |= reporters[i] == &signal;
            } else if (l2 > 0) { // Refresh LCD display needed
                l += l2;
                LCDDisplay::inst().setThirdLine(status);
                lcdThirdLineBlank = false;
                Thread::wait(400);
            } else if (!lcdThirdLineBlank) { // Clear LCD display needed
                LCDDisplay::inst().setThirdLine("");
                lcdThirdLineBlank = true;
                Thread::wait(100);
            }
//            }
        }
        if (l) {
            l = snprintf(buf, sizeof(buf), fmtSmartRest, "/s", l, buf2);
            sock.setBlocking(3000);
            l = sock.sendOnly(buf, l);
            if (l < 0)
                aWarning("%s\n", status);
        }
        if (!flag)
            wdt.kick();    // reset watchdog timer
    }
}