#include <stdio.h>
#include "DeviceIntegration.h"
#include "ComposedRecord.h"
#include "CharValue.h"
#include "IntegerValue.h"
#include "SmartRestConf.h"
#include "logging.h"

bool DeviceIntegration::init()
{
    if (_init)
        return false;
    
    // get device by identity
    // Usage: 100,<SERIAL/NR>
    if (!_tpl.add("10,100,GET,/identity/externalIds/c8y_Serial/%%,,application/vnd.com.nsn.cumulocity.externalId+json,%%,STRING,\r\n"))
        return false;

    // get device id from identity
    // Response: 200,<DEVICE/ID>
    if (!_tpl.add("11,200,\"$.managedObject\",,\"$.id\"\r\n"))
        return false;

    // Create device
    // Usage: 101,<SERIAL/NR>
    if (!_tpl.add("10,101,POST,/inventory/managedObjects,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,STRING,\"{\"\"name\"\":\"\"Mbed Test Device\"\",\"\"type\"\":\"\"com_ublox_C027_REV-A\"\",\"\"c8y_Hardware\"\":{\"\"revision\"\":\"\"1\"\",\"\"model\"\":\"\"Ublox C027\"\",\"\"serialNumber\"\":\"\"%%\"\"},\"\"c8y_SupportedOperations\"\":[\"\"c8y_Relay\"\",\"\"c8y_Configuration\"\",\"\"c8y_Message\"\"],\"\"c8y_IsDevice\"\":{},\"\"com_cumulocity_model_Agent\"\":{}}\"\r\n"))
        return false;

    // Get device id
    // Response: 201,<DEVICE/ID>
    if (!_tpl.add("11,201,,\"$.c8y_IsDevice\",\"$.id\"\r\n"))
        return false;

    // Insert global ID
    // Usage: 102,<DEVICE/ID>,<SERIAL/NR>
    if (!_tpl.add("10,102,POST,/identity/globalIds/%%/externalIds,application/vnd.com.nsn.cumulocity.externalId+json,application/vnd.com.nsn.cumulocity.externalId+json,%%,UNSIGNED STRING,\"{\"\"type\"\":\"\"c8y_Serial\"\",\"\"externalId\"\":\"\"%%\"\"}\"\r\n"))
        return false;

    // Update IMEI, CellId and iccid
    // Usage: 103,<DEVICE/ID>,<IMEI>,<CELL/ID>,<ICCID>
    if (!_tpl.add("10,103,PUT,/inventory/managedObjects/%%,application/vnd.com.nsn.cumulocity.managedObject+json,application/vnd.com.nsn.cumulocity.managedObject+json,%%,UNSIGNED STRING STRING STRING,\"{\"\"c8y_Mobile\"\":{\"\"imei\"\":\"\"%%\"\",\"\"cellId\"\":\"\"%%\"\",\"\"iccid\"\":\"\"%%\"\"}}\"\r\n"))
        return false;

    _init = true;
    return true;
}

bool DeviceIntegration::integrate()
{
    if (deviceID != 0)
        return false;

    // template bootstrapping process
    if (_client.bootstrap(_tpl) != SMARTREST_SUCCESS) {
        aError("Template bootstrap.\n");
        return false;
    }

    if ((!deviceExisting()) && ((!createDevice()) || (!addGlobalIdentifier())))
        return false;

    if (!updateDevice())
        return false;

    return true;
}

bool DeviceIntegration::deviceExisting()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(100);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        aWarning("Send failed.\r\n");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        aError("No device found.\n");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() == 0) {
        return false;
    }

    if (received.value(0).integerValue() == 50) {
        return false;
    }
    
    if (received.value(0).integerValue() != 200) {
        aError("Unknown msg %d.\n", received.value(0).integerValue());
        return false;
    }
    setDeviceID(received.value(2).integerValue());
    
    return true;
}

bool DeviceIntegration::createDevice()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(101);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        _client.stop();
        return 0;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        aError("Create device.\n");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        aError("%d/3 values received.\n", received.values());
        return false;
    }
    
    if (received.value(0).integerValue() != 201) {
        aError("Unknown msg %d.\n", received.value(0).integerValue());
        return false;
    }

    setDeviceID(received.value(2).integerValue());
    return true;
}

bool DeviceIntegration::addGlobalIdentifier()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(102);
    IntegerValue devId(deviceID);
    CharValue imei(_deviceInfo.imei());
    if ((!record.add(msgId)) || (!record.add(devId)) || (!record.add(imei)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        aWarning("Add X-ID failed.\n");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        aError("Receive X-ID failed.\n");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        aError("%d/3 values received.\n", received.values());
        return false;
    }
    
    if (received.value(0).integerValue() != 200) {
        aError("Unknown msg %d\n", received.value(0).integerValue());
        return false;
    }

    return true;
}

bool DeviceIntegration::updateDevice()
{
    ComposedRecord record;
    ParsedRecord received;

    IntegerValue msgId(103);
    IntegerValue devId(deviceID);
    CharValue imei(_deviceInfo.imei());
    CharValue cellId(_deviceInfo.cellId());
    CharValue iccid(_deviceInfo.iccid());
    if ((!record.add(msgId)) || (!record.add(devId)) || (!record.add(imei)) || (!record.add(cellId)) || (!record.add(iccid)))
        return false;

    if (_client.send(record) != SMARTREST_SUCCESS) {
        aError("Send.\n");
        _client.stop();
        return false;
    }

    if (_client.receive(received) != SMARTREST_SUCCESS) {
        aError("Receive.\n");
        _client.stop();
        return false;
    }
    _client.stop();

    if (received.values() != 3) {
        aError("%d/3 values received.\n", received.values());
        return false;
    }

    if (received.value(0).integerValue() != 201) {
        aError("Unknown msg %d\n", received.value(0).integerValue());
        return false;
    }

    return true;
}
