/*
 * SmartRest.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <string.h>
#include "SmartRest.h"
#include "SmartRestConf.h"

/*-------------------------------------------------------------------------*/
#ifdef SMARTREST_TRANSACTIONAL
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::request(const DataGenerator& generator, const char *overrideIdentifier)
{
	uint8_t res = send(generator, overrideIdentifier);
	stop();
	return res;
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::request(const DataGenerator& generator, Aggregator& aggregator, const char *overrideIdentifier)
{
	if (!aggregator.managed())
		return SMARTREST_INTERNAL_ERROR;

	uint8_t res = send(generator, overrideIdentifier);
	if (res != SMARTREST_SUCCESS) {
		stop();
		return res;
	}

	ParsedRecord recvd;
	while ((res = receive(recvd)) == SMARTREST_SUCCESS) {
		if (!aggregator.add(recvd))
			return SMARTREST_INTERNAL_ERROR;
	}

	stop();
	if (res == SMARTREST_END_OF_RESPONSE)
		return SMARTREST_SUCCESS;
	return res;
}
/*-------------------------------------------------------------------------*/
#endif  // SMARTREST_TRANSACTIONAL
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::bootstrap(const DataGenerator& generator)
{
	int8_t ret = beginRequest(NULL, NULL);
	if (ret != SMARTREST_SUCCESS)
		return ret;
	ret = awaitResponse();
	if (ret != SMARTREST_SUCCESS)
		return ret;

	ParsedRecord record;
	ret = receive(record);
	if (ret != SMARTREST_SUCCESS)
		return ret;
	if (!record) {
		return SMARTREST_INTERNAL_ERROR;
	}
	stop();

	if (setMoGid(record)) {
		return SMARTREST_SUCCESS;
	}

	if (record.value(0).integerValue() != 40)
		return SMARTREST_INTERNAL_ERROR;

	ret = send(generator, NULL);
	if (ret != SMARTREST_SUCCESS)
		return ret;
	ret = receive(record);
	if (ret != SMARTREST_SUCCESS)
		return ret;
	stop();

	if (!setMoGid(record))
		return SMARTREST_INTERNAL_ERROR;
	else
		return SMARTREST_SUCCESS;
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::send(const DataGenerator& generator, const char *overrideIdentifier)
{
	uint8_t res = beginRequest(NULL, overrideIdentifier);
	if (res != SMARTREST_SUCCESS)
		return res;

	_client.sendData(generator);
	return awaitResponse();
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::sendAndClose(const DataGenerator& generator, const char *overrideIdentifier)
{
	uint8_t res = beginRequest(NULL, overrideIdentifier);
	if (res != SMARTREST_SUCCESS)
		return res;

	_client.sendData(generator);
	if (_client.endRequest() == CLIENT_OK) {
		return SMARTREST_SUCCESS;
	} else {
		return SMARTREST_INTERNAL_ERROR;
	}
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::stream(const char *uri, const Record& record, const char *overrideIdentifier)
{
	uint8_t res = beginRequest(uri, overrideIdentifier);
	if (res != SMARTREST_SUCCESS)
		return res;

	_client.sendData(record);
	return awaitResponse();
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::receive(ParsedRecord& record)
{
	if (_source == NULL)
		return SMARTREST_INTERNAL_ERROR;

	uint8_t res = _parser.readFrom(*_source, record);
	switch (res)
	{
	case PARSER_SUCCESS: 		 return SMARTREST_SUCCESS;
	case PARSER_END_OF_RESPONSE: return SMARTREST_END_OF_RESPONSE;
	case PARSER_TIMEOUT_ERROR:	 return SMARTREST_TIMEOUT_ERROR;
	default:					 return SMARTREST_INTERNAL_ERROR;
	}
}
/*-------------------------------------------------------------------------*/
void SmartRest::stop()
{
	_source = NULL;
	_client.stop();
}
/*-------------------------------------------------------------------------*/
const char * SmartRest::getIdentifier() const
{
	if (*_mogid)
		return _mogid;
	else
		return NULL;
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::beginRequest(const char *uri, const char *overrideIdentifier)
{
	int res;
    if (uri) {
        res = _client.beginStream(uri);
    } else {
        res = _client.beginRequest();
    }

	if (res == CLIENT_CONNECTION_ERROR)	{
		return SMARTREST_CONNECTION_FAILED;
	} else if (res != CLIENT_OK) {
		return SMARTREST_INTERNAL_ERROR;
	}
    if (overrideIdentifier != NULL) {
        if (_client.sendIdentifier(overrideIdentifier) != CLIENT_OK)
            return SMARTREST_INTERNAL_ERROR;
    } else {
        if (_client.sendIdentifier(srX_ID) != CLIENT_OK)
            return SMARTREST_INTERNAL_ERROR;
    }
    return SMARTREST_SUCCESS;
}
/*-------------------------------------------------------------------------*/
uint8_t SmartRest::awaitResponse()
{
	if ((_client.endRequest() != CLIENT_OK) ||
		(_client.awaitResponse() != CLIENT_OK)) {
		return SMARTREST_INTERNAL_ERROR;
	}
	_source = &_client.receiveData();
	return SMARTREST_SUCCESS;
}
/*-------------------------------------------------------------------------*/
bool SmartRest::setMoGid(ParsedRecord& record)
{
	const char *mogid;

	*_mogid = 0;
	if ((record.values() < 2) || (record.value(0).integerValue() != 20))
		return false;

	if ((record.value(1).valueType() != VALUE_INTEGER) ||
		(record.value(1).integerValue() <= 0))
		return false;

	mogid = record.rawValue(1);
	if (strlen(mogid)+1 > sizeof(_mogid))
		return false;

	strcpy(_mogid, mogid);

	return true;
}
/*-------------------------------------------------------------------------*/