/*
 * ParsedValue.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef PARSEDVALUE_H
#define PARSEDVALUE_H

#include "config.h"
#include "Value.h"
#include "CharValue.h"

#ifndef SMARTREST_PARSED_FIXED_SIZE
#define SMARTREST_PARSED_FIXED_SIZE 16
#endif

/**
 * Provides access to a parsed value by implementing the value interface.
 * 
 * Example:
 * @code
 * // given a filled parsed record
 * ParsedRecord record;
 * 
 * // get first value
 * const Value& val = record.value(0);
 * switch (val.valueType()) {
 * case VALUE_NULL:
 *     printf("Null value.\n");
 *     break;
 * case VALUE_CHARACTER:
 *     printf("Character value: %s\n", val.characterValue());
 *     break;
 * case VALUE_INTEGER:
 *     printf("Integer value: %d\n", val.integerValue());
 *     break;
 * case VALUE_FLOAT:
 *     printf("Float value: %lf\n", val.floatValue());
 *     break;
 * }
 * @encode
 */
class ParsedValue : public Value
{
	public:
		ParsedValue(const char*, bool = false);

		virtual uint8_t valueType() const;
		virtual long integerValue() const;
		virtual double floatValue() const;
		virtual const char * characterValue() const;

		virtual size_t write(AbstractDataSink&) const;
		virtual size_t length() const;
		virtual Value* copy() const;

	private:
		void extractValue();

	private:
		CharValue _value;
		uint8_t _type;
		double _float;
		long _integer;
};

#endif
