/*
 * Aggregator.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "Aggregator.h"
#include <stdlib.h>

/*-------------------------------------------------------------------------*/
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
Aggregator::Aggregator(size_t capacity, bool growing, bool managed) :
	_initial(capacity),
	_growing(growing),
	_managed(managed)
{
	_capacity = 0;
	_list = NULL;
	_length = 0;
}
#else
Aggregator::Aggregator(bool managed) :
	_managed(managed)
{
	_length = 0;
}
#endif
/*-------------------------------------------------------------------------*/
Aggregator::~Aggregator()
{
	if (_managed)
	{
		for (size_t n = 0; n < _length; n++)
			delete _list[n];
	}
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	if (_list != NULL)
		free(_list);
#endif
}
/*-------------------------------------------------------------------------*/
bool Aggregator::add(const Record& record)
{
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	if (_length == _capacity)
	{
		size_t capacity;

		if ((!_growing) && (_capacity != 0))
			return false;

		if (_capacity == 0)
			capacity = _initial;
		else
			capacity = _capacity + SMARTREST_AGGREGATOR_MEMORY_INCREMENT;

		const Record **list = (const Record**)realloc(_list,
				capacity*sizeof(Record*));
		if (list == NULL)
			return false;
		_list = list;
		_capacity = capacity;
	}
#else
	if (_length == SMARTREST_AGGREGATOR_FIXED_SIZE)
		return false;
#endif

	if (_managed)
	{
		Record *copy = record.copy();
		if (copy == NULL)
			return false;
		_list[_length++] = copy;
	}
	else
	{
		_list[_length++] = &record;
	}
	return true;
}
/*-------------------------------------------------------------------------*/
const Record& Aggregator::get(size_t index) const
{
	return *_list[index];
}
/*-------------------------------------------------------------------------*/
void Aggregator::clear()
{
	if (_managed)
	{
		for (size_t n = 0; n < _length; n++)
			delete _list[n];
	}
	_length = 0;

#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	if (_list != NULL)
		free(_list);
	_list = NULL;
	_capacity = 0;
#endif
}
/*-------------------------------------------------------------------------*/
size_t Aggregator::length() const
{
	return _length;
}
/*-------------------------------------------------------------------------*/
bool Aggregator::full() const
{
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	return (_growing) ? false : (_length == _capacity);
#else
	return (_length == SMARTREST_AGGREGATOR_FIXED_SIZE);
#endif
}
/*-------------------------------------------------------------------------*/
size_t Aggregator::capacity() const
{
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	return (_growing) ? 0 : _capacity;
#else
	return SMARTREST_AGGREGATOR_FIXED_SIZE;
#endif
}
/*-------------------------------------------------------------------------*/
bool Aggregator::managed() const
{
	return _managed;
}
/*-------------------------------------------------------------------------*/
size_t Aggregator::writeTo(AbstractDataSink& sink) const
{
	size_t len = 0;
	for (size_t n = 0; n < _length; n++)
		len += _list[n]->writeTo(sink);
	return len;
}
/*-------------------------------------------------------------------------*/
size_t Aggregator::writtenLength() const
{
	size_t len = 0;
	for (size_t n = 0; n < _length; n++)
		len += _list[n]->writtenLength();
	return len;
}
/*-------------------------------------------------------------------------*/
Aggregator* Aggregator::copy() const
{
#ifndef SMARTREST_AGGREGATOR_FIXED_SIZE
	Aggregator *copy = new Aggregator(_length, _growing, true);
#else
	Aggregator *copy = new Aggregator(true);
#endif
	for (size_t n = 0; n < _length; n++)
		copy->add(*_list[n]);
	return copy;
}
/*-------------------------------------------------------------------------*/
