/*
 * FloatValue.cpp
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "FloatValue.h"
#include <math.h>
#include <stdio.h>


/*-------------------------------------------------------------------------*/
FloatValue::FloatValue(double number, uint8_t digits, bool zflag)
{
	_negative = (number < 0.0);
	_zflag = zflag;
	if (_negative)
		_number = -number;
	else
		_number = number;
	if (digits == SMARTREST_FLOATVALUE_DETECTPREC)
		_digits = detectPrecision(_number);
	else
		_digits = digits;
}
/*-------------------------------------------------------------------------*/
uint8_t FloatValue::valueType() const
{
	return VALUE_FLOAT;
}
/*-------------------------------------------------------------------------*/
long FloatValue::integerValue() const
{
	return 0L;
}
/*-------------------------------------------------------------------------*/
double FloatValue::floatValue() const
{
	return (_negative) ? -_number : _number;
}
/*-------------------------------------------------------------------------*/
const char * FloatValue::characterValue() const
{
	return 0;
}
/*-------------------------------------------------------------------------*/
size_t FloatValue::write(AbstractDataSink& sink) const
{
	size_t n = 0;
	double number, rounding;
	uint8_t i;
	unsigned long int_part;
	double remainder;

	if (isnan(_number))
		return sink.write("nan");

	if (_negative)
		n += sink.write('-');

	if (isinf(_number))
	{
		n += sink.write("inf");
		return n;
	}

	number = _number;
	rounding = 0.5;
	for (i = 0; i < _digits; ++i)
		rounding /= 10.0;
	number += rounding;

	int_part = (unsigned long)number;
	remainder = number - (double)int_part;

	if ((_zflag) || (int_part))
		n += sink.write(int_part);

	if (_digits == 0)
		return n;
	n += sink.write("."); 

	uint8_t digits = _digits;
	while (digits-- > 0)
	{
		remainder *= 10.0;
		unsigned long l = (unsigned long)remainder;
		n += sink.write(l);
		remainder -= l; 
	} 

	return n;
}
/*-------------------------------------------------------------------------*/
size_t FloatValue::length() const
{
	size_t l = 0;
	double number, rounding;
	unsigned long n;
	uint8_t i;

	if (isnan(_number))
		return 3;

	if (_negative)
		l++;

	if (isinf(_number))
	{
		l += 3;
		return l;
	}

	number = _number;
	rounding = 0.5;
	for (i = 0; i < _digits; ++i)
		rounding /= 10.0;
	number += rounding;

	n = (unsigned long)number;
	if ((_zflag) || (n))
	{
		do
		{
			n /= 10;
			l++;
		} while(n);
	}

	if (_digits > 0)
		l += 1 + _digits;

	return l;
}
/*-------------------------------------------------------------------------*/
Value* FloatValue::copy() const
{
	double number;

	number = (_negative) ? -_number : _number;
	return new FloatValue(number, _digits, _zflag);
}
/*-------------------------------------------------------------------------*/
uint8_t FloatValue::detectPrecision(double n)
{
	uint8_t prec = 0, count = 0;
	int8_t d;

	if ((isinf(n)) || (isnan(n)))
		return 0;

	n -= (long)n;

	while ((prec < 6) && (count < 3))
	{
		n *= 10;
		d = (uint8_t)n;

		if (count == 0)
		{
			if ((d <= 0) || (d >= 9))
				count++;
		}
		else
		{
			count++;
		}
		prec++;
		n -= d;
	}

	return --prec;
}
/*-------------------------------------------------------------------------*/
