/*
 * AbstractClient.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file AbstractClient.h
 * An abstraction layer for the SmartREST http client.
 */

#ifndef ABSTRACTCLIENT_H
#define ABSTRACTCLIENT_H

#include "config.h"
#include "DataGenerator.h"
#include "AbstractDataSource.h"
#include "AbstractDataSink.h"

/** Return value indicating that the operation has been successful. */
#define CLIENT_OK 0
/** Return value indicating that there has been a connection error. */
#define CLIENT_CONNECTION_ERROR 1
/** Return value indicating that an internal error occurred. */
#define CLIENT_INTERNAL_ERROR 2

/*
 * An abstraction layer for the SmartREST http client.
 * 
 * All methods have to be called in the following order:
 * - setAuthorization()
 * - beginRequest() or beginStream()
 * - sendIdentifier() (optional)
 * - sendData() (optional if in request mode)
 * - endRequest()
 * - awaitResponse()
 * - receiveData() (optional)
 * - stop()
 * 
 * Additionally, stop() can be called at any time to reset the state of
 * this instance.
 * A return value of any function unequal to CLIENT_OK must follow a
 * call to stop() before starting another request.
 */
class AbstractClient
{
	public:
		virtual ~AbstractClient() { };

		/**
		 * Sets authorization username and password. This method can only be
		 * called when no request is processing. Values set using this method
		 * must be kept in memory until either new values are set or the object
		 * is being destroyed.
		 * @param username the username used for further request authorization
		 * @param password the password used for further request authorization
		 */
//		virtual uint8_t setAuthorization(const char*, const char*) = 0;

		/**
		 * Begins a new request. This method has to be the first call for
		 * any request. A connection to the server will be established
		 * and the start of a HTTP post request will be sent over the
		 * connection including Host and Authorization headers.
		 */
		virtual uint8_t beginRequest() = 0;

        /**
         * Begins a new stream. This method has to be the first call for
		 * any request. A connection to the server will be established
		 * and the start of a HTTP post request will be sent over the
		 * connection including Host and Authorization headers.
         */
        virtual uint8_t beginStream(const char*) = 0;

		/**
		 * Sends the X-Id device identifier header.
		 * @param identifier the identifier to be sent. The value has to remain
		 *                   valid for the entire request. If null or empty, no
		 *                   identifier is sent.
		 */
		virtual uint8_t sendIdentifier(const char*) = 0;

		/**
		 * Sends POST data over the connection. Before the data is sent,
		 * a Content-Length header is also sent over the connection. if the
		 * estimated length does not match the actual conten length,
		 * unexpected behavior will occur.
		 * @param generator the data generator for the data to be sent
		 */
		virtual uint8_t sendData(const DataGenerator&) = 0;

		/**
		 * Finishes the request. In case no data has been sent over the
		 * connection, this method will send twice carriage-return new-line.
		 */
		virtual uint8_t endRequest() = 0;

		/**
		 * Blocks until the start of a response has been received.
		 */
		virtual uint8_t awaitResponse() = 0;

		/**
		 * Returns a data source for reading data.
		 * When no data can be read for whatever reason,
		 * a data source with an error state may be returned.
		 * @return the data source to read from
		 */
		virtual AbstractDataSource& receiveData() = 0;

		/**
		 * Resets the connection. This method can be called at any time
		 * and resets the state of this instance.
		 */
		virtual void stop() = 0;
};

#endif
