/*
 * Parser.h
 *
 * Created on: Nov 1, 2013
 * * Authors: Vincent Wochnik <v.wochnik@gmail.com>
 *
 * Copyright (c) 2013 Cumulocity GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file Parser.h
 * A CSV parser utilizing an AbstractDataSource instance for reading from
 * a connection.
 */

#ifndef PARSER_H
#define PARSER_H

#include "config.h"
#include <stddef.h>
#include <stdint.h>
#include "ParsedRecord.h"
#include "AbstractDataSource.h"

#ifndef SMARTREST_PARSER_BUFFER_SIZE
#define SMARTREST_PARSER_BUFFER_SIZE 81
#endif

/** Return value indicating that no error occurred. */
#define PARSER_SUCCESS 0
/** Return value indicating that a transmission timeout occurred during
 * the transmission of a record. */
#define PARSER_TIMEOUT_ERROR 1
/** Return value indicating the end of response. */
#define PARSER_END_OF_RESPONSE 2
/** Return value indicating a parse error. */
#define PARSER_PARSE_ERROR 3
/** Return value indicating an internal error. */
#define PARSER_INTERNAL_ERROR 4

/*
 * A record parser reading from a data source.
 * 
 * Example:
 * @code
 * // given a concrete data sorce
 * AbstractDataSource source;
 * 
 * Parser parser;
 * ParsedRecord record;
 * uint8_t ret;
 * 
 * while ((ret = parser.readFrom(source, record)) == PARSER_SUCCESS) {
 *     // read record
 * }

 * // error handling
 * switch (ret) {
 * case PARSER_END_OF_RESPONSE:
 *     // no error; end of response
 *     break;
 * case PARSER_TIMEOUT_ERROR:
 *     // timeout error
 *     break;
 * default:
 *     // parse error
 *     break;
 * }
 * @encode
 */
class Parser
{
	public:
		/*
		 * Creates a new Parser instance.
		 */
		Parser();

		/*
		 * Parses a row from a stream.
		 * @param source the source to read from
		 * @param row the object written to
		 * @return a non-zero value if and only if an error occured.
		 */
		uint8_t readFrom(AbstractDataSource&,ParsedRecord&);

	private:
		void parse(char);
		void append(char);
		void close();
		void reset();

	private:
		char _buffer[SMARTREST_PARSER_BUFFER_SIZE];
		uint8_t _state;
		char *_ptr;
		size_t _count, _length, _trailing;
};

#endif
