#include "Button.h"
#include "mbed.h"


/******************************************************************************/
// constructor of instance button
//      - PinName:  Name of pin of MBed LPC1768 (e.g. p9, p10,...)
//      - PinMode:  PullUp, PullDown, PullNone, OpenDrain
//      - time:     defines the time after the longPress-Flag is set.
//
/******************************************************************************/
Button::Button(PinName pBtn, PinMode  pBtnMode, float time)
{
    fTime = time;
    pin = new DigitalIn(pBtn, pBtnMode);

    if(fTime > 0)
        timer = new Timeout;

    init();

}


/******************************************************************************/
// initialisation
//      - bNewState:        current state of button
//      - bOldState:        previous state of button
//      - bIsPressed:       flag is set if button is pressed
//      - bLongPresed:      flag is set, after timer is zero (bIsPressed = true)
//      - bWasPressed:      after button is not pressed anymore
//
/******************************************************************************/
void Button::init()
{
    bNewState   = false;
    bOldState   = false;    
    
    bIsPressed  = false;
    bLongPress  = false;
    bWasPressed = false;


    bBtnNewAction = false;

    iNumCycle = 0;
    if(fTime > 0)
        timer->detach();


    return;
}


/******************************************************************************/
// query the Button with debouncing
//      - bNewState:        current state of button
//      - bOldState:        previous state of button
//      - bBtnNewAction:    flag, if an action is occur
//      - iNumCycle:        cycle counter of same state
//      - NUNCYCLES:        max. value of cycle counter
//      - *iNumBtn:         pointer to variable, which counts the number of pressed buttons
//
/******************************************************************************/
void Button::polling(uint8_t *iNumBtn)
{
    bOldState   = bNewState;                                                    // bNewState gets the becomes the bOldstate
    bNewState   = (bool)pin->read();                                            // update bNewState
    bWasPressed = false;                                                        // reset the flag was pressed

    
    if(!bBtnNewAction and (bOldState != bNewState))                             // if there is a action (changing signal) on the button (btn), then rise the flag bBtnNewAction
        bBtnNewAction = true;
    

    if(bBtnNewAction) {                                                         // if flag ist true (new action on the button)

        if(bOldState == bNewState) {                                            // if current state equals previous state
            if(iNumCycle <= NUMCYCLE) {                                         // count the number of cycle unless the max. value is reached
                iNumCycle++;

            } else if(iNumCycle > NUMCYCLE) {                                   // else if the max. value is reached
                iNumCycle = 0;                                                  // reset the counter, there is no action any more expected
                bBtnNewAction = false;                                          // there is nomore new action.

                if(bNewState == PRESSED) {                                      // btn is pressed, then
                    bIsPressed = true;                                          // set this flag true
                    
                    if(iNumBtn != NULL)
                        (*iNumBtn)++;
                    
                    if(fTime > 0)                                                // if btn has a timer, 
                        timer->attach(callback(this, &Button::setLongPressFlag), fTime);
                        

                } else if(bNewState == NOT_PRESSED) {                           // if btn is not pressed, then
                    if(bIsPressed == true)                                      // if btn was pressed, then
                        bWasPressed = true;                                     // set this flag true
                    
                    bIsPressed = false;                                         // set this flag false
                    
                    if(fTime > 0)                                               // if btn has a timer, 
                        timer->detach();
                        bLongPress = false;
                        
                    if(iNumBtn != NULL && *iNumBtn > 0)                         // if there is a pointer to a counter-Variable and the value is greater than 0
                        (*iNumBtn)--;                                           // decrease iNumBtn by one 
                }

            }
        } else {
            iNumCycle = 0;
        }
    }

    return;
}


/******************************************************************************/
//
/******************************************************************************/
bool Button::isPressed()
{
    if(bIsPressed)  return true;
    else            return false;
}


/******************************************************************************/
//
/******************************************************************************/
bool Button::isLongPress()
{
    if(bLongPress)  return true;
    else            return false;

}


/******************************************************************************/
//
/******************************************************************************/
bool Button::wasPressed()
{
    if(bWasPressed) return true;
    else            return false;

}


/******************************************************************************/
//
/******************************************************************************/
void Button::setLongPressFlag(){
    bLongPress = true;
    timer->detach();
}


