/* mbed Repetitive Interrupt Timer Library
 * Copyright (c) 2011 wvd_vegt
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef __RIT_h
#define __RIT_h

#include "mbed.h"

#define SYSCLK 96000000L
#define ONEMHZ  1000000L
#define RIT_FREQ 96*1000L
#define RIT_DIV (SYSCLK/RIT_FREQ)-1

/** Repetitive Interrupt Timer (RIT) class.
* 
* Sample Code:
* 
* DigitalOut rit_led(LED3); 
* 
* //Our function isr. 
* void RIT_IRQHandler(void) { 
*   rit_led=!rit_led; // Flash a Led 
* } 
* 
* RIT rit(1000); //Set interval to 1000 ms or 1 second. 
* 
* rit.append(RIT_IRQHandler); //Append our own function ISR. 
* rit.enable(); 
* 
* wait(10); //wait 10 seconds (and watch the led flashing). 
* 
* rit.disable(); 
* rit.unappend(); 
*/
class RIT {
public:
    /** Constructor.
     *
     * @parm ms the number of milliseconds between two interrupts.
     */
    RIT(uint32_t ms);

    /** Setup timing in ms.
     *
     * @parm ms the number of milliseconds between two interrupts.
     */
    void setup_ms(uint32_t ms);

    /** Setup timing in us.
     *
     * @parm us the number of microseconds between two interrupts.
     */
    void setup_us(uint32_t us);

    /** Attach custom interrupt handler replacing default.
     *
     * @parm fptr the new interrupt handler.
     */
    void attach(void(*fptr)(void));

    /** Restore default interrupt handler.
     *
     */
    void detach(void);

    /** Append function isr to global interrupt handler.
     *
     * @parm fptr the function isr to be called from within the default interrupt handler.
     */
    void append(void(*fptr)(void));

    /** Remove function isr from global interrupt handler.
     *
     */
    void unappend();

    /** Enable RIT & Interrupt
     *
     */
    void enable(void);

    /** Disable RIT & Interrupt.
     *
     */
    void disable(void);

    /** Power up RIT.
     *
     */
    void power_enable(void);

    /** Power down RIT
     *
     * Note: Powering down might take some more effect when entering Deep Sleep,
     * accoring to errata.lpc1768.pdf.
     */
    void power_disable(void);

private:

    /** There can only be one.
     *
     */
    static RIT *instance;

    /** Storage for an appended function isr.
     *
     */
    void(*_rit_g_isr)(void);

    /** The default (instance) isr.
     *
     * Note: this isr calls the static one so there is only be a single one.
     */
    void ritisr(void);

    /** The actual (static) default isr.
     *
     */
    static void _ritisr(void);

    /** Select the Peripheral Clock (PCLK_RIT) for the RIT
     *  to be equal to the CCLK.
     *
     */
    void select_clk(void);
};

#endif