#include "mbed.h"
#include "i2c_mbed_fpga.h"

#define COMM_STATE_I2C_IDLE (1u)
#define COMM_STATE_I2C_APPLICATION (2u)
#define COMM_STATE_DIRECT_IDLE (4u)
#define COMM_STATE_DIRECT_APPLICATION (6u)

void i2c_mbed_fpga::wait_for_idle_state(void)
{
    int comm_state = 0;
    do {
        uint16_t tmp;
        this->i2c_word_read_simple(I2C_STATE, &tmp);
        comm_state = (tmp & I2C_STATE_COMM_STATE_MASK) >> I2C_STATE_COMM_STATE_OFFSET;
        wait_us(500);
    } while((comm_state != COMM_STATE_I2C_IDLE)&&(comm_state != COMM_STATE_DIRECT_IDLE));
}

void i2c_mbed_fpga::wait_for_application_state(void)
{
    int comm_state = 0;
    do {
        uint16_t tmp;
        this->i2c_word_read_simple(I2C_STATE, &tmp);
        comm_state = (tmp & I2C_STATE_COMM_STATE_MASK) >> I2C_STATE_COMM_STATE_OFFSET;
        wait_us(500);
    } while((comm_state != COMM_STATE_I2C_APPLICATION)&&(comm_state != COMM_STATE_DIRECT_APPLICATION));
}



/** i2c read from slave DUT using the command interpreter feature
*  retun 0 on success, otherwise fails
*
*  @param[in]   address         16-bit address to read from
*  @param[out]  return value    Pointer to the value read at address
*  @retval      0   Read failed
*  @retval      1   Read successful
*/
int i2c_mbed_fpga::i2c_word_read_interpreter(uint16_t address, uint16_t* return_value){
    uint16_t read_value;
    int busy;
    int ack = 0;
    while(i2c_check_busy_flag()) {
        /* Wait for the busy flag to clear */
    }
    ack += this->i2c_word_write_simple(I2C_CMD_INTERPRETER_ADDRESS, (address | 0x1)); /* Write the address with the LSB set to 1 to indicate a read operation */
    wait_us(100); 
    /* Check if the command is finished by polling the LSB of CMD_INTERPRETER_ADDRESS */
    do {
        ack += this->i2c_word_read_simple(I2C_CMD_INTERPRETER_ADDRESS, &read_value);
        if((read_value & 0x1) == 1u) {
            busy = true;
        } else {
            busy = false;
        }
        wait_us(100); 
    }
    while(busy == true);
    
    /* We know the read command has finished. Read back the value */
    ack += this->i2c_word_read_simple(I2C_CMD_INTERPRETER_DATA, &read_value);
    *return_value = read_value;
    return (ack == 0) ? 0 : 1; 
}


bool i2c_mbed_fpga::i2c_check_busy_flag(void){
    bool retval = true;
    uint16_t tmp;
    this->i2c_word_read_simple(I2C_STATUS, &tmp);
    if((tmp & I2C_STATUS_COMMAND_BUSY_MASK) == 0u) {
        retval = false;
    }
    return retval;
}

/** i2c read from slave DUT
*  retun 0 on success, otherwise fails
*
*  @param i2c_master specifies the i2c interface
*  @param word is 4byte, first 2bytes as addr, the rest 2bytes to store data
*/
int i2c_mbed_fpga::i2c_word_read(char *word){
    int ack = 0;
    ack = i2c_master.write(I2C_SLAVE_ADDR, word, 1, true);  //restart
    ack += i2c_master.read(I2C_SLAVE_ADDR, word+1, 2, false); //stop bit
    wait_us(100); 
    return (ack == 0) ? 0 : 1; 
}

/** i2c read from slave DUT
*  retun 0 on success, otherwise fails
*
*  @param i2c_master specifies the i2c interface
*  @param word is 4byte, first 2bytes as addr, the rest 2bytes to store data
*/
int i2c_mbed_fpga::i2c_word_read_simple(char address, uint16_t* return_value){
    int ack = 0;
    char value[2];
    ack = i2c_master.write(I2C_SLAVE_ADDR, &address, 1, true);  //restart
    ack += i2c_master.read(I2C_SLAVE_ADDR, value, 2, false); //stop bit
    *return_value = (int)value[1] | ((int)value[0] << 8);
    wait_us(100); 
    return (ack == 0) ? 0 : 1; 
}

/** i2c write to slave DUT
*   ==> one time write, not read back check
*
*  @param i2c_master specifies the i2c interface
*  @param word is considered as 4byte char data
*/
int i2c_mbed_fpga::i2c_word_write(char *word){
    int ack = 0;
    ack = i2c_master.write(I2C_SLAVE_ADDR, word, 3, false);
    return ack; 
}

/** i2c read from slave DUT
*  retun 0 on success, otherwise fails
*
*  @param i2c_master specifies the i2c interface
*  @param word is 4byte, first 2bytes as addr, the rest 2bytes to store data
*/
int i2c_mbed_fpga::i2c_word_write_simple(char address, uint16_t value){   
    int ack = 0;
    char i2cMessage[3];
    *(i2cMessage+0) = (char)(address)& 0xff;
    *(i2cMessage+1) = (char)((value >> 8u) & 0xff);
    *(i2cMessage+2) = (char)(value & 0xff);  
    ack = i2c_word_write(i2cMessage);
    wait_us(100); 
    return ack;
}

int i2c_mbed_fpga::i2c_word_write_interpreter(uint16_t address, uint16_t value){
    uint16_t read_value;
    int busy;
    int ack = 0;
    while(i2c_check_busy_flag()) {
        /* Wait for the busy flag to clear */
    }
    ack += this->i2c_word_write_simple(I2C_CMD_INTERPRETER_DATA, value);
    ack += this->i2c_word_write_simple(I2C_CMD_INTERPRETER_ADDRESS, (address & 0xFFFE)); /* Write the address with the LSB set to 0 to indicate a write operation */
    /* Check if the command is finished by polling the LSB of CMD_INTERPRETER_ADDRESS */
    do {
        ack += this->i2c_word_read_simple(I2C_CMD_INTERPRETER_ADDRESS, &read_value);
        if((read_value & 0x1) == 0u) {
            busy = true;
        } else {
            busy = false;
        }
        wait_us(100); 
    }
    while(busy == true);
    return (ack == 0) ? 0 : 1; 
}



/** i2c enter key to open I2C window (for old releases)
*/
//int i2c_mbed_fpga::i2c_window_open(){
//    char i2cMessage[3];
//    *(i2cMessage+0) = (char)(I2C_CUST_ID3)& 0xff;
//    *(i2cMessage+1) = (char)(0xD0)& 0xff;
//    *(i2cMessage+2) = (char)(0xD0)& 0xff;  
//    return i2c_word_write(i2cMessage);
//}

/** i2c enter key to Start the motor (for old releases)
*/
//int i2c_mbed_fpga::i2c_motor_start(){
//    char i2cMessage[3];
//    *(i2cMessage+0) = (char)(I2C_CUST_ID3)& 0xff;
//    *(i2cMessage+1) = (char)(0xCA)& 0xff;
//    *(i2cMessage+2) = (char)(0xFE)& 0xff;  
//    return i2c_word_write(i2cMessage);
//}


/** i2c enter key to open I2C configuration mode entry*/
int i2c_mbed_fpga::i2c_config_mode_entry(){
    return i2c_word_write_simple(I2C_COMMAND_CONTROL, 0x1DEA);
}

/** i2c enter MLX key to open I2C MLX configuration mode entry*/
int i2c_mbed_fpga::i2c_mlx_mode_entry(){
    int ack = 0;
    ack += i2c_word_write_simple(I2C_COMMAND_KEY, 0x65A9);
    ack += i2c_word_write_simple(I2C_COMMAND_CONTROL, 0x354B);  
    return ack;
}

/** i2c enter key to enter HTOL mode*/
int i2c_mbed_fpga::i2c_htol_mode_entry(){
    int ack = 0;
    ack += i2c_word_write_simple(I2C_COMMAND_KEY, 0x2BAD);
    ack += i2c_word_write_simple(I2C_COMMAND_CONTROL, 0xD1E0);
    return ack;
}

/** i2c ram start up flag set to skip OTP copy */
int i2c_mbed_fpga::i2c_skip_app_copy(){
    return i2c_word_write_simple(I2C_STARTUP_FLAGS_1, 0x0500);
}

/** i2c soft reset */
int i2c_mbed_fpga::i2c_soft_reset(){
    return i2c_word_write_simple(I2C_COMMAND_CONTROL, 0xC1A0);
}

/** i2c to set the 50k PWM*/
int i2c_mbed_fpga::i2c_set_50k_pwm(unsigned int pwm50k){
    nv_gen_ctrl_val &= ~NV_PWM_50K_MASK;
    nv_gen_ctrl_val |= pwm50k << NV_PWM_50K_OFFSET;
    return i2c_word_write_simple(I2C_GEN_CTRL, nv_gen_ctrl_val);
}

/** i2c to set the Postion Pulse width*/
int i2c_mbed_fpga::i2c_set_position_pulse_width(unsigned int mantisaa_2b, unsigned int exponent_3b){
    nv_positin_val &= ~NV_POSITION_PULSE_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 2) | mantisaa_2b) << NV_POSITION_PULSE_TIME_OFFSET;     
    return i2c_word_write_simple(I2C_POSITION, nv_positin_val);
}

/** i2c to set the Postion Flat width */
int i2c_mbed_fpga::i2c_set_position_flat(unsigned int mantisaa_2b, unsigned int exponent_3b){
    nv_positin2_val &= ~NV_POSITION_FLAT_TIME_MASK;
    nv_positin2_val |= ((exponent_3b << 2) | mantisaa_2b) << NV_POSITION_FLAT_TIME_OFFSET;      
    return i2c_word_write_simple(I2C_POSITION2, nv_positin2_val);
}

/** i2c to set the Postion Pulse duty cycle */
int i2c_mbed_fpga::i2c_set_position_duty(unsigned int duty_2b){
    nv_positin_val &= ~NV_POSITION_DUTY_MASK;
    nv_positin_val |= duty_2b << NV_POSITION_DUTY_OFFSET;     
    return i2c_word_write_simple(I2C_POSITION, nv_positin_val);
}

/** i2c to enable the Postion Pulse majority volting */
int i2c_mbed_fpga::i2c_set_position_maj_vote(unsigned int maj_1b){
    nv_positin_val &= ~NV_POSI_MAJO_VOTE_MASK;
    nv_positin_val |= maj_1b << NV_POSI_MAJO_VOTE_OFFSET;     
    return i2c_word_write_simple(I2C_POSITION, nv_positin_val);
}

/** i2c to set the anti-cogging rotation direction */
int i2c_mbed_fpga::i2c_set_position_anti_cog(unsigned int cog_1b){
    nv_positin_val &= ~NV_ANTI_COG_MASK;
    nv_positin_val |= cog_1b << NV_ANTI_COG_OFFSET;     
    return i2c_word_write_simple(I2C_POSITION, nv_positin_val);
}


/** i2c to set the Start Up Pulse width (pulse train) */
int i2c_mbed_fpga::i2c_set_start_up_pulse_width(unsigned int mantisaa_3b, unsigned int exponent_3b){
    nv_start_up_val &= ~NV_START_UP_TIME_MASK;
    nv_start_up_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_START_UP_TIME_OFFSET;     
    return i2c_word_write_simple(I2C_START_UP, nv_start_up_val);
}

/** i2c to set the Start Up Flat width */
int i2c_mbed_fpga::i2c_set_start_up_flat(unsigned int mantisaa_3b, unsigned int exponent_3b){
    nv_positin2_val &= ~NV_START_UP_FLAT_TIME_MASK;
    nv_positin2_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_START_UP_FLAT_TIME_OFFSET;    
    return i2c_word_write_simple(I2C_POSITION2, nv_positin2_val);
}

/** i2c to set the Start up Pulse duty cycle (pulse train) */
int i2c_mbed_fpga::i2c_set_start_up_duty(unsigned int duty_2b){
    nv_start_up_val &= ~NV_START_DUTY_MASK;
    nv_start_up_val |= duty_2b << NV_START_DUTY_OFFSET;
    return i2c_word_write_simple(I2C_START_UP, nv_start_up_val);
}

/** i2c to set the Start up commutation number of EHPs (pulse train) */
int i2c_mbed_fpga::i2c_set_start_up_num_comm(unsigned int comm){
    nv_start_up_val &= ~NV_COMM_START_NUM_MASK;
    nv_start_up_val |= comm << NV_COMM_START_NUM_OFFSET;     
    return i2c_word_write_simple(I2C_START_UP, nv_start_up_val);
}

/** i2c to set the Soft Start Up (pulse train) */
int i2c_mbed_fpga::i2c_set_soft_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b, unsigned int step_size, unsigned int num_steps){
    int ack = 0;
    nv_start_up_val &= ~NV_SOFT_START_MASK;
    nv_start_up_val |= enbale << NV_SOFT_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    nv_start_up_val &= ~NV_SOFT_STEP_SIZE_MASK;
    nv_start_up_val |= step_size << NV_SOFT_STEP_SIZE_OFFSET;
    nv_wind_brake_val &= ~NV_SOFT_NUM_STEP_MASK;
    nv_wind_brake_val |= num_steps << NV_SOFT_NUM_STEP_OFFSET; 
    ack += i2c_word_write_simple(I2C_POSITION, nv_positin_val);     
    ack += i2c_word_write_simple(I2C_START_UP, nv_start_up_val);   
    ack += i2c_word_write_simple(I2C_WIND_BRAKE, nv_wind_brake_val);
    
    return ack; 
}

/** i2c to set the High Torque Start Up (pulse train */
int i2c_mbed_fpga::i2c_set_high_torque_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b){
    int ack = 0;
    nv_start_up_val &= ~NV_LONG_START_MASK;
    nv_start_up_val |= enbale << NV_LONG_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    ack += i2c_word_write_simple(I2C_POSITION, nv_positin_val);        
    ack += i2c_word_write_simple(I2C_START_UP, nv_start_up_val);
        
    return ack; 
}

/** i2c to set the Single Pulse Start Up (pulse train) */
int i2c_mbed_fpga::i2c_set_single_pulse_start_up(unsigned int enbale, unsigned int mantisaa_3b, unsigned int exponent_3b){
    int ack = 0;
    nv_start_up_val &= ~NV_SINGLE_PULSE_START_MASK;
    nv_start_up_val |= enbale << NV_SINGLE_PULSE_START_OFFSET;
    nv_positin_val &= ~NV_FIRST_NON_FLAT_TIME_MASK;
    nv_positin_val |= ((exponent_3b << 3) | mantisaa_3b) << NV_FIRST_NON_FLAT_TIME_OFFSET;
    ack += i2c_word_write_simple(I2C_POSITION, nv_positin_val);   
    ack += i2c_word_write_simple(I2C_START_UP, nv_start_up_val);
        
    return ack; 
}

/** i2c to set the rough regulation gain */
int i2c_mbed_fpga::i2c_force_rough_reg(unsigned int reg){
    int ack = 0;
    nv_gen_ctrl_val &= ~NV_ROUGH_REG_MASK;
    nv_gen_ctrl_val |= reg << NV_ROUGH_REG_OFFSET;
    ack += i2c_word_write_simple(I2C_GEN_CTRL, nv_gen_ctrl_val);
    
    return ack; 
}

/** i2c to set the rough regulation gain */
int i2c_mbed_fpga::i2c_set_rough_gain(unsigned int rough_gain){
    int ack = 0;
    nv_gen_ctrl_val &= ~NV_ROUGH_GAIN_MASK;
    nv_gen_ctrl_val |= rough_gain << NV_ROUGH_GAIN_OFFSET;
    ack += i2c_word_write_simple(I2C_GEN_CTRL, nv_gen_ctrl_val);
    
    return ack; 
}

/** i2c to set the ehp regulation gain */
int i2c_mbed_fpga::i2c_set_ehp_reg_gain(unsigned int ehp_gain){
    int ack = 0;
    nv_gen_ctrl_val &= ~NV_EHP_REG_GAIN_MASK;
    nv_gen_ctrl_val |= ehp_gain << NV_EHP_REG_GAIN_OFFSET;
    ack += i2c_word_write_simple(I2C_GEN_CTRL, nv_gen_ctrl_val);
    
    return ack; 
}

/** i2c to set the ehp regulation gain */
int i2c_mbed_fpga::i2c_set_fall_time_blank(unsigned int blank_time){
    int ack = 0;
    nv_dig_config_val &= ~NV_FLAT_BLANK_MASK;
    nv_dig_config_val |= blank_time << NV_FLAT_BLANK_OFFSET;
    ack += i2c_word_write_simple(I2C_DIGITAL_CFG, nv_dig_config_val);
    
    return ack; 
}

/** i2c to set the current threshold for I_didt */
int i2c_mbed_fpga::i2c_set_comm_i_thres(unsigned int i_thr_low, unsigned int i_thr_high){
    int ack = 0;
    nv_i_zc_th_low_val &= ~NV_I_ZC_TH_LOW_MASK;
    nv_i_zc_th_low_val |= i_thr_low << NV_I_ZC_TH_LOW_OFFSET;
    nv_i_zc_th_high_val &= ~NV_I_ZC_TH_HIGH_MASK;
    nv_i_zc_th_high_val |= i_thr_high << NV_I_ZC_TH_HIGH_OFFSET;
    ack += i2c_word_write_simple(I2C_I_ZC_TH_LOW_REG, nv_i_zc_th_low_val);
    ack += i2c_word_write_simple(I2C_I_ZC_TH_HIGH_REG, nv_i_zc_th_high_val);
    return ack; 
}

/** i2c to set the di current threshold for didt */
int i2c_mbed_fpga::i2c_set_comm_di_thres(unsigned int di_1st, unsigned int di_2nd){
    int ack = 0;
    nv_di_th_1st_val &= ~NV_DI_TH_1ST_MASK;
    nv_di_th_1st_val |= di_1st << NV_DI_TH_1ST_OFFSET;
    nv_di_th_2nd_val &= ~NV_DI_TH_2ND_MASK;
    nv_di_th_2nd_val |= di_2nd << NV_DI_TH_2ND_OFFSET;
    ack += i2c_word_write_simple(I2C_DI_TH_1ST_REG, nv_di_th_1st_val);
    ack += i2c_word_write_simple(I2C_DI_TH_2ND_REG, nv_di_th_2nd_val);
    return ack; 
}

/** i2c to clean the I2C controller settins */
int i2c_mbed_fpga::i2c_clear_spd_ctrl(){
    int ack = 0;
    nv_spd_control_1_val = 0;
    nv_spd_control_2_val = 0;
    ack += i2c_word_write_simple(I2C_SPD_CTRL_1, nv_spd_control_1_val);
    ack += i2c_word_write_simple(I2C_SPD_CTRL_2, nv_spd_control_2_val);
    
    return ack; 
}

/** i2c to set the I2C speed input mode */
int i2c_mbed_fpga::i2c_set_input_mode(unsigned int mode){
    int ack = 0;
    nv_application_cfg_val &= ~NV_INPUT_MODE_CFG_MASK;
    nv_application_cfg_val |= mode << NV_INPUT_MODE_CFG_OFFSET;
    ack += i2c_word_write_simple(I2C_APPLICATION_CFG, nv_application_cfg_val);    
    return ack; 
}

/** i2c to set the open loop mode */
int i2c_mbed_fpga::i2c_set_loop_mode(unsigned int openloop){
    int ack = 0;
    nv_spd_control_1_val &= ~NV_SPD_LOOP_MODE_MASK;
    nv_spd_control_1_val |= openloop << NV_SPD_LOOP_MODE_OFFSET;
    ack += i2c_word_write_simple(I2C_SPD_CTRL_1, nv_spd_control_1_val);
    
    return ack; 
}

/** i2c to set the Single Pulse Start Up (pulse train) */
int i2c_mbed_fpga::i2c_set_open_loop_duty(unsigned int duty){
    int ack = 0;
    ram_open_duty_val = duty;
    ack += i2c_word_write_simple(I2C_SPEED_DUTY, ram_open_duty_val);
        
    return ack; 
}

/** i2c to set the speed curve type */
int i2c_mbed_fpga::i2c_set_curve_type(unsigned int curvetype){
    int ack = 0;
    nv_spd_control_1_val &= ~NV_CURVE_MODE_MASK;
    nv_spd_control_1_val |= curvetype << NV_CURVE_MODE_OFFSET;
    ack += i2c_word_write_simple(I2C_SPD_CTRL_1, nv_spd_control_1_val);
    
    return ack; 
}

/** i2c to set the open dc ini */
int i2c_mbed_fpga::i2c_set_dc_ini(unsigned int ini){
    int ack = 0;
    nv_spd_control_2_val &= ~NV_DC_OPENLOOP_INI_MASK;
    nv_spd_control_2_val |= ini << NV_DC_OPENLOOP_INI_OFFSET;
    ack += i2c_word_write_simple(I2C_SPD_CTRL_2, nv_spd_control_2_val);
    
    return ack; 
}

/** i2c to set the open dc slew rate */
int i2c_mbed_fpga::i2c_set_dc_sr(unsigned int sr){
    int ack = 0;
    nv_spd_control_2_val &= ~NV_DC_OPENLOOP_SR_MASK;
    nv_spd_control_2_val |= sr << NV_DC_OPENLOOP_SR_OFFSET;
    ack += i2c_word_write_simple(I2C_SPD_CTRL_2, nv_spd_control_2_val);
    
    return ack; 
}

/** i2c to set the target CLIM during start up */
int i2c_mbed_fpga::i2c_set_clim_start_up(unsigned int clim){
    int ack = 0;
    nv_clim_user_1_val &= ~NV_TARGET_CLIM_USER_PULSES_MASK;
    nv_clim_user_1_val |= clim << NV_TARGET_CLIM_USER_PULSES_OFFSET;
    ack += i2c_word_write_simple(I2C_CLIM_USER_1, nv_clim_user_1_val);
    
    return ack; 
}

/** i2c to set the target CLIM during brake */
int i2c_mbed_fpga::i2c_set_clim_brake(unsigned int clim){
    int ack = 0;
    nv_clim_user_0_val &= ~NV_TARGET_CLIM_USER_BRAKE_MASK;
    nv_clim_user_0_val |= clim << NV_TARGET_CLIM_USER_BRAKE_OFFSET;
    ack += i2c_word_write_simple(I2C_CLIM_USER_0, nv_clim_user_0_val);
    
    return ack; 
}

/** i2c to set the target CLIM during run time */
int i2c_mbed_fpga::i2c_set_clim_run_time(unsigned int clim){
    int ack = 0;
    nv_clim_user_0_val &= ~NV_TARGET_CLIM_USER_MOTOR_MASK;
    nv_clim_user_0_val |= clim << NV_TARGET_CLIM_USER_MOTOR_OFFSET;
    ack += i2c_word_write_simple(I2C_CLIM_USER_0, nv_clim_user_0_val);
    
    return ack; 
}


/** i2c to enbale test debug mode */
int i2c_mbed_fpga::i2c_enable_pules_debug_mode(unsigned int enable){
    int ack = 0;
    ram_debug_ctrl_val &= ~0x0001;
    ram_debug_ctrl_val |= enable;
    ack += i2c_word_write_simple(0xEA, 1);
    
    return ack; 
}

