#include <SPI.h>

#include "qeihw.h"
#include "USBHID.h"
#include "hid_command.h"


/* Type Defines                     */
typedef union {
    uint32_t uint32;
    uint8_t uint8[4];
} ui32_ui8;

#ifdef TARGET_LPC1768
/// SPI hardware
static SPI spi(p5, p6, p7); // mosi, miso, sclk
/// SPI chip select
DigitalOut cs_n(p8); // chip select, active low

/// GPI hardware
DigitalIn channel_y(p16); // Channel Y connected to GPI pin
DigitalIn channel_di2(p17); // Channel DI2 connected to GPI pin
DigitalIn channel_di3(p18); // Channel DI3 connected to GPI pin

DigitalIn channel_z(P1_24); // Channel Z, enable GPI
#elif defined(TARGET_LPC1549)
/// SPI hardware
static SPI spi(P0_12, P0_12, P0_16); // mosi, miso, sclk
/// SPI chip select
DigitalOut cs_n(P0_27); // chip select, active low

/// GPI hardware
DigitalIn channel_y(P1_3); // Channel Y connected to GPI pin
DigitalIn channel_di2(P0_16); // Channel DI2 connected to GPI pin
DigitalIn channel_di3(P0_10); // Channel DI3 connected to GPI pin

DigitalIn channel_z(P0_9); // Channel Z, enable GPI
#endif

/// QEI hardware
static QEIHW qeihw (
        0, //_dirinv Direction invert. 
           // When = 1, complements the QEICONF register DIR bit
        0, //_sigmode    Signal mode. 
           // When = 0, PhA and PhB are quadrature inputs. 
           // When = 1, PhA is direction and PhB is clock
        0, //_capmode    Capture mode. 
           // When = 0, count PhA edges only (2X mode). 
           // When = 1, count PhB edges also (4X mode).
        0  //_invinx Invert index. 
           // When = 1, inverts the sense of the index signal
    );

/**
 * Setup SPI transmission bits per frame, polarity and phase
 */
void setup_spi()
{
    spi.format(8, 1); // 8 bits, POL = 0, PHA = 1
}

/** Reads the command byte from the HID_REPORT and calls the corresponding
 * function to process the command.
 *
 * @param *in input HID_REPORT pointer
 * @param *out output HID_REPORT pointer
 */
void command(HID_REPORT* in, HID_REPORT* out)
{
    switch(in->data[0])
    {
        case COMMAND_SYSTEM:
            command_system(in, out);
            break;
        case COMMAND_SPI:
            command_spi(in, out);
            break;
        case COMMAND_QUADRATURE_ENCODER:
            command_qei(in, out);
            break;
        case COMMAND_READ_CHANNELS:
            command_read_channels(in, out);
        default:
            break;
    }
}

/** Process a HID_REPORT for a system command
 *
 * @param *in input HID_REPORT pointer
 * @param *out output HID_REPORT pointer
 */
void command_system(HID_REPORT* in, HID_REPORT* out)
{
    switch(in->data[1])
    {
        case SYSTEM_VERSION:
            out->data[0] = COMMAND_SYSTEM;
            out->data[1] = SYSTEM_VERSION;
            out->data[2] = PROTOCOL_VERSION_MAJOR;
            out->data[3] = PROTOCOL_VERSION_MINOR;    
            break;
        default:
            break;
    }
}

/** Process a HID_REPORT for a SPI command
 *
 * @param *in input HID_REPORT pointer
 * @param *out output HID_REPORT pointer
 */
void command_spi(HID_REPORT* in, HID_REPORT* out)
{
    int number_of_bytes;
    int i;
    
    out->data[0] = COMMAND_SPI;
    out->data[1] = in->data[1];
    
    number_of_bytes = in->data[1];
    
    cs_n = 0;
    
    for(i=0; i<number_of_bytes; i++)
    {
        out->data[2+i] = spi.write(in->data[2+i]);
    }
    
    cs_n = 1;
}

/** Process a HID_REPORT for a quadrature command
 *  
 *  @param *in input HID_REPORT pointer
 *  @param *out output HID_REPORT pointer
 */
void command_qei(HID_REPORT* in, HID_REPORT* out)
{
    ui32_ui8 data; 
    
    out->data[0] = COMMAND_QUADRATURE_ENCODER;
    out->data[1] = in->data[1];
    
    switch(in->data[1])
    {
        case QEI_SETUP:
            qeihw.SetVelocityTimerReload_us(QEI_VELOCITY_TIMER_US);
            qeihw.SetMaxPosition(QEI_MAX_POSITION);
            //qeihw.Reset(QEI_RESET_POSOnIDX);
            break;
        case QEI_POSITION:
            data.uint32 = qeihw.GetPosition();
            out->data[2] = data.uint8[0];
            out->data[3] = data.uint8[1];
            out->data[4] = data.uint8[2];
            out->data[5] = data.uint8[3];
            break;
        case QEI_VELOCITY:
            data.uint32 = qeihw.GetVelocityCap();
            out->data[2] = data.uint8[0];
            out->data[3] = data.uint8[1];
            out->data[4] = data.uint8[2];
            out->data[5] = data.uint8[3];
            break;
        case QEI_POSITION_RESET:
            qeihw.Reset(QEI_RESET_POS);
            break;
    }
    
}

/**
 * Process input for a read channel command
 *
 * @param *in input HID_REPORT pointer
 * @param *out output HID_REPORT pointer
 */
void command_read_channels(HID_REPORT* in, HID_REPORT* out)
{
    out->data[0] = COMMAND_READ_CHANNELS;
    out->data[1] = in->data[1];

    switch(in->data[1])
    {
        case READ_INPUT:
            out->data[2] = channel_y; // Y
            out->data[3] = channel_di2; // DI2
            out->data[4] = channel_di3; // DI3
            out->data[5] = channel_z; // Z 
            break;
    }
}

    
