/******************************************************************//**
* @file max14724.h
*
* @author Wilson Tang
*
* @version 1.0
*
* Started: 22APR15
*
* Updated: 
*
* @brief Header file for MAX14724 class
* 
* http://www.maximintegrated.com/max14724
*
***********************************************************************
*
* @copyright 
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#ifndef MAX14724_H
#define MAX14724_H


#include "mbed.h"


/******************************************************************//**
* Max14724 Class
**********************************************************************/
class Max14724 : public I2C
{
    uint8_t w_adrs, r_adrs;
    
    public:
    
        /**
        * max14724_i2c_adrs_t - enumerated MAX14724 I2C Addresses
        */
        typedef enum
        {
            MAX14724_I2C_ADRS0 = 0x74,
            MAX14724_I2C_ADRS1
        }max14724_i2c_adrs_t;
        
        
        /**
        * max14724_regs_t - enumerated MAX14724 register addresses
        */
        typedef enum
        {
            DIR0,         //Switches 8A–1A direct read/write access
            DIR1,         //Switches 8B–1B direct read/write access
            DIR2,         //Switches 8C–1C direct read/write access
            DIR3,         //Switches 8D–1D direct read/write access
            SHDW0 = 0x10, //Switches 8A–1A shadow read/write access
            SHDW1,        //Switches 8B–1B shadow read/write access
            SHDW2,        //Switches 8C–1C shadow read/write access
            SHDW3,        //Switches 8D–1D shadow read/write access
            CMD0,         //Set mux A and B command (reads 0x00)
            CMD1          //Set mux C and D command (reads 0x00)
        }max14724_regs_t;
        
        
        /**
        * max14724_cmds_t - enumerated MAX14724 commands
        */
        typedef enum
        {
            ENABLE_SW01,     //enables sw1 on bank only
            ENABLE_SW02,     //enables sw2 on bank only
            ENABLE_SW03,     //enables sw3 on bank only
            ENABLE_SW04,     //enables sw4 on bank only
            ENABLE_SW05,     //enables sw5 on bank only
            ENABLE_SW06,     //enables sw6 on bank only
            ENABLE_SW07,     //enables sw7 on bank only
            ENABLE_SW08,     //enables sw8 on bank only
            DISABLE_BANK,    //opens all switches on bank
            COPY_SHADOW,     //copies both shadow registers for bank
            NO_CHANGE
        }max14724_cmds_t;
        
        
        /**
        * max14724_sw_t - enumerated MAX14724 switch bitmasks
        */
        typedef enum
        {
            SW01 = (1 << 0),
            SW02 = (1 << 1),
            SW03 = (1 << 2),
            SW04 = (1 << 3),
            SW05 = (1 << 4),
            SW06 = (1 << 5),
            SW07 = (1 << 6),
            SW08 = (1 << 7)
        }max14724_sw_t;
        
        
        /**********************************************************//**
        * Constructor for Max14724 Class
        *
        * On Entry:
        *     @param[in] sda - sda pin of I2C bus
        *     @param[in] scl - scl pin of I2C bus
        *     @param[in] i2c_adrs - 7-bit slave address of MAX14724
        *
        * On Exit:
        *    @return none
        *
        * Example:
        * @code
        * 
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0); 
        *
        * @endcode
        **************************************************************/
        Max14724(PinName sda, PinName scl, max14724_i2c_adrs_t i2c_adrs);
        

        /******************************************************************//**
        * Writes given commands to CMD_A, CMD_B, CMD_C, and CMD_D
        *
        * On Entry:
        *     @param[in] cmdA - command for CMD_A
        *     @param[in] cmdB - command for CMD_B
        *     @param[in] cmdC - command for CMD_C
        *     @param[in] cmdD - command for CMD_D
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        *
        * rtn_val = mux.wrt_cmd_registers(DISABLE_BANK, DISABLE_BANK, DISABLE_BANK, DISABLE_BANK);
        * 
        * @endcode
        **********************************************************************/
        uint16_t wrt_cmd_registers(max14724_cmds_t cmdA, max14724_cmds_t cmdB, max14724_cmds_t cmdC, max14724_cmds_t cmdD);


        /******************************************************************//**
        * Writes bankA, bankB, bankC, and bankD to coresponding shadow registers
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *     @param[in] bankC - binary representation of switch states
        *     @param[in] bankD - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0); 
        *
        * uint16_t bankA = (SW03 | SW02); //example only
        * uint16_t bankB = (SW05 | SW04);
        * uint16_t bankC = SW07;
        * uint16_t bankD = SW08;
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.wrt_shadow_registers(bankA, bankB, bankC, bankD);
        *
        * @endcode
        **********************************************************************/
        uint16_t wrt_shadow_registers(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD);


        /******************************************************************//**
        * Writes bankA, bankB, bankC, and bankD to coresponding direct access registers 
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *     @param[in] bankC - binary representation of switch states
        *     @param[in] bankD - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code  
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);   
        *
        * uint16_t bankA = (SW12 | SW02); //example only
        * uint16_t bankB = (SW11 | SW01);
        * uint16_t bankC = SW07;
        * uint16_t bankD = SW08;
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.wrt_dir_registers(bankA, bankB, bankC, bankD);
        *
        * @endcode
        **********************************************************************/
        uint16_t wrt_dir_registers(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD);


        /******************************************************************//**
        * Writes bankA, bankB, bankC, and bankD to coresponding shadow register and then 
        * issues copy command for both banks
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *     @param[in] bankC - binary representation of switch states
        *     @param[in] bankD - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code  
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);   
        *
        * uint16_t bankA = (SW12 | SW02); //example only
        * uint16_t bankB = (SW11 | SW01);
        * uint16_t bankC = SW07;
        * uint16_t bankD = SW08;
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.set_switches(bankA, bankB, bankC, bankD);
        *
        * @endcode
        **********************************************************************/
        uint16_t set_switches(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD);


        /**********************************************************//**
        * Reads data from direct access registers starting at DIR0 and 
        * stores it in byte array pointed at by 'data'
        *
        * On Entry:
        *     @param[in] data - pointer to byte array for storing data
        *
        * On Exit:
        *     @param[out] data - data buffer now contains data read 
        *                        from dir registers
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        * uint8_t data[4];
        * 
        * //read direct access registers
        * rtn_val = mux.rd_dir_registers(data);
        *
        * @endcode
        **********************************************************************/
        uint16_t rd_dir_registers(uint8_t* data);


        /**********************************************************//**
        * Reads data from shadow registers starting at SHDW0 and stores 
        * it in byte array pointed at by 'data'
        *
        * On Entry:
        *     @param[in] data - pointer to byte array for storing data
        *
        * On Exit:
        *     @param[out] data - data buffer now contains data read 
        *                        from shadow registers
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14724 mux(D14, D15, MAX14724_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        * uint8_t data[4];
        * 
        * //read shadow registers
        * rtn_val = mux.rd_shadow_registers(data);
        *
        * @endcode
        **************************************************************/      
        uint16_t rd_shadow_registers(uint8_t* data);  
};
                                  
                                          
#endif /* MAX14724_H*/
