/******************************************************************//**
* @file max14724.cpp
*
* @author Wilson Tang
*
* @version 1.0
*
* Started: 22APR15
*
* Updated:
*
* @brief Source file for MAX14724 class
*
* http://www.maximintegrated.com/max14724
*
***********************************************************************
*
* @copyright
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "max14724.h"


/**********************************************************//**
* Constructor for Max14724 Class
*
* On Entry:
*     @param[in] sda - sda pin of I2C bus
*     @param[in] scl - scl pin of I2C bus
*     @param[in] i2c_adrs - 7-bit slave address of MAX14724
*
* On Exit:
*    @return none
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* @endcode
**************************************************************/
Max14724::Max14724(PinName sda, PinName scl, max14724_i2c_adrs_t i2c_adrs) :
    I2C(sda, scl)
{
    w_adrs = (i2c_adrs << 1);
    r_adrs = (w_adrs | 0x01);
}


/******************************************************************//**
* Writes given commands to CMD_A, CMD_B, CMD_C, and CMD_D
*
* On Entry:
*     @param[in] cmdA - command for CMD_A
*     @param[in] cmdB - command for CMD_B
*     @param[in] cmdC - command for CMD_C
*     @param[in] cmdD - command for CMD_D
*
* On Exit:
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint16_t rtn_val;
*
* rtn_val = mux.wrt_cmd_registers(DISABLE_BANK, DISABLE_BANK, DISABLE_BANK, DISABLE_BANK);
*
* @endcode
**********************************************************************/
uint16_t Max14724::wrt_cmd_registers(max14724_cmds_t cmdA, max14724_cmds_t cmdB, max14724_cmds_t cmdC, max14724_cmds_t cmdD)
{
    uint8_t data[3];
    uint8_t data_length = 0;
    uint16_t rtn_val = 1;

    //build packet
    data[data_length++] = CMD0;
    data[data_length++] = (cmdB & 0x0F) << 4 | (cmdA & 0x0F);
    data[data_length++] = (cmdD & 0x0F) << 4 | (cmdC & 0x0F);

    rtn_val = write(w_adrs,(const char*) data, data_length);

    return(rtn_val);
}


/******************************************************************//**
* Writes bankA, bankB, bankC, and bankD to coresponding shadow registers
*
* On Entry:
*     @param[in] bankA - binary representation of switch states
*     @param[in] bankB - binary representation of switch states
*     @param[in] bankC - binary representation of switch states
*     @param[in] bankD - binary representation of switch states
*
* On Exit:
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint16_t bankA = (SW03 | SW02); //example only
* uint16_t bankB = (SW05 | SW04);
* uint16_t bankC = SW07;
* uint16_t bankD = SW08;
* uint16_t rtn_val;
*
* //wite shadow registers
* rtn_val = mux.wrt_shadow_registers(bankA, bankB, bankC, bankD);
*
* @endcode
**********************************************************************/
uint16_t Max14724::wrt_shadow_registers(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD)
{
    uint8_t data[5];
    uint8_t data_length = 0;
    uint16_t rtn_val = 1;

    data[data_length++] = SHDW0;
    data[data_length++] = (bankA & 0x00FF);
    data[data_length++] = (bankB & 0x00FF);
    data[data_length++] = (bankC & 0x00FF);
    data[data_length++] = (bankD & 0x00FF);

    rtn_val = write(w_adrs,(const char*) data, data_length);

    return(rtn_val);
}


/******************************************************************//**
* Writes bankA, bankB, bankC, and bankD to coresponding shadow registers
*
* On Entry:
*     @param[in] bankA - binary representation of switch states
*     @param[in] bankB - binary representation of switch states
*     @param[in] bankC - binary representation of switch states
*     @param[in] bankD - binary representation of switch states
*
* On Exit:
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint8_t bankA = (SW03 | SW02); //example only
* uint8_t bankB = (SW05 | SW04);
* uint8_t bankC = SW07;
* uint8_t bankD = SW08;
* uint16_t rtn_val;
*
* //wite shadow registers
* rtn_val = mux.wrt_shadow_registers(bankA, bankB, bankC, bankD);
*
* @endcode
**********************************************************************/
uint16_t Max14724::wrt_dir_registers(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD)
{
    uint8_t data[5];
    uint8_t data_length = 0;
    uint16_t rtn_val = 1;

    data[data_length++] = DIR0;
    data[data_length++] = (bankA & 0x00FF);
    data[data_length++] = (bankB & 0x00FF);
    data[data_length++] = (bankC & 0x00FF);
    data[data_length++] = (bankD & 0x00FF);

    rtn_val = write(w_adrs,(const char*) data, data_length);

    return(rtn_val);
}


/******************************************************************//**
* Writes bankA, bankB, bankC, and bankD to coresponding shadow register and then
* issues copy command for both banks
*
* On Entry:
*     @param[in] bankA - binary representation of switch states
*     @param[in] bankB - binary representation of switch states
*     @param[in] bankC - binary representation of switch states
*     @param[in] bankD - binary representation of switch states
*
* On Exit:
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint16_t bankA = (SW12 | SW02); //example only
* uint16_t bankB = (SW11 | SW01);
* uint16_t bankC = SW07;
* uint16_t bankD = SW08;
* uint16_t rtn_val;
*
* //wite shadow registers
* rtn_val = mux.set_switches(bankA, bankB, bankC, bankD);
*
* @endcode
**********************************************************************/
uint16_t Max14724::set_switches(uint8_t bankA, uint8_t bankB, uint8_t bankC, uint8_t bankD)
{
    uint8_t data[7];
    uint8_t data_length = 0;
    uint16_t rtn_val = 1;

    data[data_length++] = SHDW0;
    data[data_length++] = (bankA & 0x00FF);
    data[data_length++] = (bankB & 0x00FF);
    data[data_length++] = (bankC & 0x00FF);
    data[data_length++] = (bankD & 0x00FF);
    data[data_length++] = ((COPY_SHADOW << 4) | COPY_SHADOW);
    data[data_length++] = ((COPY_SHADOW << 4) | COPY_SHADOW);

    rtn_val = write(w_adrs,(const char*) data, data_length);

    return(rtn_val);
}


/**********************************************************//**
* Reads data from direct access registers starting at DIR0 and
* stores it in byte array pointed at by 'data'
*
* On Entry:
*     @param[in] data - pointer to byte array for storing data
*
* On Exit:
*     @param[out] data - data buffer now contains data read
*                        from dir registers
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint16_t rtn_val;
* uint8_t data[4];
*
* //read direct access registers
* rtn_val = mux.rd_dir_registers(data);
*
* @endcode
**********************************************************************/
uint16_t Max14724::rd_dir_registers(uint8_t* data)
{
    uint16_t rtn_val = 1;

    data[0] = DIR0;

    rtn_val = write(w_adrs,(const char*) data, 1);

    if(!rtn_val) {
        rtn_val = read(r_adrs,(char*) data, 4);
    }

    return(rtn_val);
}


/**********************************************************//**
* Reads data from shadow registers starting at SHDW0 and stores
* it in byte array pointed at by 'data'
*
* On Entry:
*     @param[in] data - pointer to byte array for storing data
*
* On Exit:
*     @param[out] data - data buffer now contains data read
*                        from shadow registers
*     @return return value = 0 on success, non-0 on failure
*
* Example:
* @code
*
* //declare mux object
* Max14724 mux(D14, D15, MAX14724_I2C_ADRS0);
*
* uint16_t rtn_val;
* uint8_t data[4];
*
* //read shadow registers
* rtn_val = mux.rd_shadow_registers(data);
*
* @endcode
**************************************************************/
uint16_t Max14724::rd_shadow_registers(uint8_t* data)
{
    uint16_t rtn_val = 1;

    data[0] = SHDW0;

    rtn_val = write(w_adrs,(const char*) data, 1);

    if(!rtn_val) {
        rtn_val = read(r_adrs,(char*) data, 4);
    }

    return(rtn_val);
}
