/*-----------------------------------------------------------
 * F746_BUTTON Library v1.0
 * Copyright (c) 2018 Wynand Steenberg
 * s216875730@mandela.ac.za
 *
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *-----------------------------------------------------------
 */

#ifndef F746_BUTTON_HPP
#define F746_BUTTON_HPP

#include "mbed.h"
#include <string>
#include "TS_DISCO_F746NG.h"
#include "LCD_DISCO_F746NG.h"
#include "F746GUI.hpp"

/** A Class library for using Buttons on the DISCO-F746NG Development board.  The class
 * uses the existing BSP class created by Team ST.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "F746_BUTTON.hpp"
 * 
 * TS_DISCO_F746NG ts_;
 * LCD_DISCO_F746NG lcd_;
 * 
 * int main()
 * {
 *     lcd_.Clear(LCD_COLOR_WHITE);                                        // Set LCD Background colour
 *        
 *     Button btn1(lcd_, ts_, 20, 50, 80, 40,           
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 1, "BUTTON1", Font12);  // Define btn1 button            
 *     btn1.Render();                                                         // Draw btn1 button
 *     
 *     Button test(lcd_, ts_, 120, 50, 80, 40,
 *                  LCD_COLOR_DARKBLUE, LCD_COLOR_GREEN, 1, "TEST", Font12);  // Define test button                   
 *     test.Render();                                                         // Draw test button
 *     
 *     Button show(lcd_, ts_, 220, 50, 80, 40,            
 *                  LCD_COLOR_BROWN, LCD_COLOR_GRAY, 1,  "SHOW", Font12);     // Define hide button            
 *     show.Hide();  
 *     
 *     Button hide(lcd_, ts_, 220, 50, 80, 40,            
 *                  LCD_COLOR_BROWN, LCD_COLOR_GRAY, 1,  "HIDE", Font12);     // Define hide button             
 *     hide.Render();                                                         // Draw hide button
 *     
 *     Button button3D(lcd_, ts_, 320, 50, 120, 40,             
 *                  LCD_COLOR_BLUE, LCD_COLOR_YELLOW, 2, "3D BUTTON", Font12);    // Define button3D button            
 *     button3D.Render();                                                         // Draw 3Dbutton button
 * 
 *     lcd_.SetTextColor(LCD_COLOR_BLACK);                                                         // Set Text colour to Black
 *     lcd_.SetFont(&Font12);                                                                      // Font size 12
 *     lcd_.DisplayStringAt(5, 5, (uint8_t *)"Button / Slider driver for DISCO_F746", LEFT_MODE);  // Display main header text
 *     lcd_.DisplayStringAt(5, 20, (uint8_t *)"Button example", LEFT_MODE);  // Display secondary header text
 *     lcd_.DisplayStringAt(5, 135, (uint8_t *)"Slider example", LEFT_MODE);  // Display secondary header text
 * 
 *     while (true)                                                    // Main program loop
 *     {
 *     
 *         if (btn1.Press())                                         // Check if btn1 button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font12);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"Button1 pressed", LEFT_MODE);
 *             wait(0.5); 
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"                                        ", LEFT_MODE);
 *         }                                                         // End btn1 button instructions
 *         
 *         if (test.Press())                                         // Check if test button was touched and run instructions if true
 *         {
 *             lcd_.SetFont(&Font12);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"TEST pressed", LEFT_MODE);
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"                                        ", LEFT_MODE);
 *             
 *         }                                                         // End test button instructions
 *         
 *         if (hide.Press())                                         // Check if hide button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font12);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"Hide pressed - Hide other buttons", LEFT_MODE);
 *             btn1.Hide();                                            // Hide btn1 button
 *             test.Hide();                                            // Hide test button
 *             hide.Hide();  
 *             show.Render();
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"                                        ", LEFT_MODE);
 *         }                                                           // End hide button instructions
 *         
 *          if (show.Press())                                        // Check if hide button was touched and run instructions if true
 *         {  
 *             lcd_.SetFont(&Font12);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"UNHIDE pressed - Restore other buttons", LEFT_MODE);
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"                                        ", LEFT_MODE);
 *             show.Hide();                                            // Hide show button before drawing new button in same place
 *             hide.Render();                                            // Draw hide button after hiding button in same position
 *             btn1.Render();                                            // Draw btn1 button
 *             test.Render();                                            // Draw test button
 *            
 *         }         
 *         
 *         if (button3D.Press())                                     // Check if 3Dbutton button was touched and run instructions if true
 *         {    
 *             lcd_.SetFont(&Font12);
 *             lcd_.SetTextColor(LCD_COLOR_BLACK);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"3D Button pressed", LEFT_MODE);
 *             wait(0.5);
 *             lcd_.DisplayStringAt(5, 110, (uint8_t *)"                                           ", LEFT_MODE);
 *         }                                                           // End 3D button instructions
 *  
 *         wait(0.02f);   
 *     }                                                               // End Main program loop
 * }                                                                   // End Main program 
 * @endcode
 */

    class Button
    {
    public:
    
        //! Constructor
        Button(LCD_DISCO_F746NG &lcd, TS_DISCO_F746NG &ts,
               uint16_t x, uint16_t y, uint16_t width, uint16_t height,
               uint32_t buttonColour, uint32_t changeColour,
               int style, const string label = "", sFONT &fonts = Font12,
               uint32_t textColour = LCD_COLOR_WHITE)
               : lcd_(lcd), ts_(ts), ButtonX_(x), ButtonY_(y), ButtonW_(width), ButtonH_(height),
                 BUTTON_COLOUR_(buttonColour), CHANGE_COLOUR_(changeColour),
                 STYLE_(style), LABEL_(label), FONTS_(&fonts), FONT_WIDTH_(fonts.Width),
                 FONT_HEIGHT_(fonts.Height), active_(true)
                 { Render(); }

        /**
          * @brief  Draw Button.
          *         
          */
        void Render();
        
        /**
          * @brief  Draw 3D Button.
          *         
          */
        void Render3D();
        
        /**
          * @brief  Change colour of Button.
          *         
          */
        void Change();

        /**
          * @brief  Change colour of 3D Button.
          *         
          */
        void Change3D();
        
        /**
          * @brief  Hide Button.
          *         
          */
        void Hide();

        /**
          * @brief  Check touch detected on Button.
          *         
          */
        bool Press();
        
        /**
          * @brief  Check if touch is on Button.
          *         
          */      
        bool ButtonBoundaryCheck();
        
        /**
          * @brief  Check previous state of Button.
          *         
          */
        static TS_StateTypeDef GottenState()
        {   return state_; }

        /**
          * @brief  Set or reset multi-touch.
          *         
          */
        static void SetMultiTouch(bool tf) { multiTouch = tf; }


    private:

        LCD_DISCO_F746NG &lcd_;         // LCD
        TS_DISCO_F746NG &ts_;           // Touch
        
        const uint16_t ButtonX_, ButtonY_, ButtonW_, ButtonH_;
        const uint32_t BUTTON_COLOUR_;  // Original colour
        const uint32_t CHANGE_COLOUR_;  // Colour to change to color
        const int STYLE_;               // Button style
        const string LABEL_;            // Button label
        sFONT *const FONTS_;            // Button Style
        const uint16_t FONT_WIDTH_;     
        const uint16_t FONT_HEIGHT_;
        bool active_;                   // Button active boolean
        
        static TS_StateTypeDef state_;
        static bool multiTouch;

        /**
          * @brief  Disallow copy constructor and assignment operator.
          *         
          */ 
        Button(const Button&);
        Button& operator=(const Button&);
        

    };
#endif  // F746_BUTTON_HPP
