/*
 * File: SensorNode/SensorNode.cpp
 * Author: William Jessup Salisbury
 * Company: Tufts Hybrid Racing Team
 * Copyright: CC BY-NC-SA 3.0
 * Date: 1/12/2012
 */

#include "mbed.h"
#include "CANProtocol.h"
#include "SensorNode.h"

SensorNode::SensorNode() : _leftWheel(LW_PIN), _rightWheel(RW_PIN), _can(CAN_RX, CAN_TX), _console(USBTX, USBRX), _leftTicks(0), _rightTicks(0), _leftRevolutions(0), _rightRevolutions(0), _syncID(0)  {
    /* InterruptIn initialization */
    _leftWheel.mode(PullUp);
    _leftWheel.fall(this, &SensorNode::leftTick);
    _rightWheel.mode(PullUp);
    _rightWheel.fall(this, &SensorNode::rightTick);

    /* CAN initialization */
    _can.frequency(CAN_FREQUENCY);
    _can.attach(this, &SensorNode::canReceive);

    /* Ticker initialization */
    //_statusTicker.attach(this, &SensorNode::canSend, tickerTimeout);

    /* Hello, World! */
    _console.printf("%s\r\n", "SensorNode instantiated");
}

SensorNode::~SensorNode() {
    _statusTicker.detach();
}

void SensorNode::leftTick() {
    if ( (++_leftTicks % ticksPerRevolution) == 0 ) {
        _leftTicks = 0;
        ++_leftRevolutions;
    }
}

void SensorNode::rightTick() {
    if ( (++_rightTicks % ticksPerRevolution) == 0 ) {
        _rightTicks = 0;
        ++_rightRevolutions;
    }
}

void SensorNode::canSend() {
    CANMessage msg;
    msg.id = CAN_STATUS;
    msg.len = 8;
    msg.data[0] = _syncID;
    msg.data[1] = 0xE;
    msg.data[2] = 0xA;
    msg.data[3] = 0xD;
    msg.data[4] = 0xB;
    msg.data[5] = 0xE;
    msg.data[6] = 0xE;
    msg.data[7] = 0xF;
    if (_can.write(msg)) {
        _console.printf("%s\r\n", "Message sent.");
    } else {
        _console.printf("%s\r\n", "Message send failure.");
    }
}

void SensorNode::canReceive() {
    CANMessage msg;

    if (_can.read(msg)) {
        switch (msg.id) {
            case CAN_RESET:
                _console.printf("%s\r\n", "CAN_RESET Message received.");
                Reset();
                break;

            case CAN_SYNC:
                _console.printf("%s\r\n", "CAN_SYNC Message received.");
                _syncID = msg.data[0];
                canSend();
                break;

            default:
                _console.printf("%s\r\n", "Message decode failure.");
                break;
        }
    } else {
        _console.printf("%s\r\n", "Message recieve failure.");
    }
}

void SensorNode::Reset() {
    _leftTicks = 0;
    _rightTicks = 0;
    _leftRevolutions = 0;
    _rightRevolutions = 0;
    _can.reset();
}