/* ePy-Lite BLE usage example
 * Copyright (c) 2021 Richlink Technology
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "USBSerial.h"
#include "epy_lite_io.h"

#if MBED_MAJOR_VERSION < 6
#error "Mbed OS version requires 6 or later."
#endif

#include "SPIFBlockDevice.h"

// Disable Write Protection and Hold functions
DigitalOut spif_wp(SPIFLASH_WP, 1);
DigitalOut spif_hold(SPIFLASH_HOLD, 1);

/*
BlockDevice *bd = new SPIFBlockDevice(MBED_CONF_SPIF_DRIVER_SPI_MOSI,
                                      MBED_CONF_SPIF_DRIVER_SPI_MISO,
                                      MBED_CONF_SPIF_DRIVER_SPI_CLK,
                                      MBED_CONF_SPIF_DRIVER_SPI_CS);
*/
BlockDevice *bd = new SPIFBlockDevice(SPIFLASH_MOSI,
                                      SPIFLASH_MISO,
                                      SPIFLASH_CLK,
                                      SPIFLASH_CS);

// This example uses LittleFileSystem as the default file system
#include "LittleFileSystem.h"
LittleFileSystem fs("fs");

// Uncomment the following two lines and comment the previous two to use FAT file system.
// #include "FATFileSystem.h"
// FATFileSystem fs("fs");


USBSerial *_console;

BufferedSerial *_ble_serial;
ATCmdParser *_parser;

Ticker flipper;
DigitalOut _led(LED_Y, 0);
DigitalOut _ledr(LED_R, 0);
DigitalOut _ledb(LED_B, 0);

void flip_led()
{
    _led = !_led;
}

void ble_recv()
{
    unsigned char c;

    while(_ble_serial->readable()) {
        _ledb = !_ledb;
        _ble_serial->read(&c, 1);
        _console->write(&c, 1);
    }
}

void console_recv()
{
    static bool pt_mode = false;
    unsigned char c;

    while(_console->readable()) {
        _ledr = !_ledr;
        _console->read(&c, 1);
        if (pt_mode == true)
        {
            if (c == '\r')
            {
                _console->write("\r\n", 2); // echo to console
                _ble_serial->write("\r\n", 2);
            }
            else if (c == 0x03) // Ctrl-C
            {
                pt_mode = false;
                _console->printf("\r\nExit pass-through Mode\r\n");
            }
            else
            {
                _console->write(&c, 1);     // echo to console
                _ble_serial->write(&c, 1);
            }
        }
        else {
            if (c == '?')
            {
                _console->printf("\r\n ? : print this menu");
                _console->printf("\r\n p : enter pass-through mode. Press CTRL-C to exit the mode");
                _console->printf("\r\n v : send AT+VERSION");
                _console->printf("\r\n a : send AT+ADDR=?");
                _console->printf("\r\n r : send AT+RESET");
                _console->printf("\r\n d : send AT+MODE_DATA");
                _console->printf("\r\n c : send !CCMD@");
                _console->printf("\r\n");
            }
            else if (c == 'p')
            {
                pt_mode = true;
                _console->printf("\r\nEnter pass-through mode...\r\n");
            }
            else if (c == 'v')
            {
                //_ble_serial->write("AT+VERSION\r\n", 12);
                _ble_serial->write("AT+VERSION", 10);
                ThisThread::sleep_for(2ms);
                _ble_serial->write("\r\n", 2);
            }
            else if (c == 'a')
            {
                _ble_serial->write("AT+ADDR=?\r\n", 11);
            }
            else if (c == 'r')
            {
                _ble_serial->write("AT+RESET\r\n", 10);
            }
            else if (c == 'd')
            {
                _ble_serial->write("AT+MODE_DATA\r\n", 14);
            }
            else if (c == 'c')
            {
                _ble_serial->write("!CCMD@", 6);
            }
        }
    }
}

int main()
{
    flipper.attach(&flip_led, 1s);

    _console = new USBSerial(true);
    _ble_serial = new BufferedSerial(BLE_UART_TX, BLE_UART_RX, BLE_DEFAULT_BAUD_RATE);
    _parser = new ATCmdParser(_ble_serial, "\r\n");
    //_parser->debug_on( 0 );
    //_parser->set_delimiter( "\r\n" );

#if 1
    // Try SPI Flash

    bd->init();
    _console->printf("spif size: %llu\r\n",         bd->size());
    _console->printf("spif read size: %llu\r\n",    bd->get_read_size());
    _console->printf("spif program size: %llu\r\n", bd->get_program_size());
    _console->printf("spif erase size: %llu\r\n",   bd->get_erase_size());    
    _console->printf("Mounting the filesystem... ");
    int err = fs.mount(bd);
    _console->printf("%s\r\n", (err ? "Fail :(" : "OK"));
    if (err) {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        _console->printf("No filesystem found, formatting... ");
        err = fs.reformat(bd);
        _console->printf("%s\r\n", (err ? "Fail :(" : "OK"));
        if (err) {
            //error("error: %s (%d)\r\n", strerror(-err), err);
             _console->printf("error: %s (%d)\r\n", strerror(-err), err);
        }
    }

#endif

#if 1
//    _console->attach(&console_recv);
//    _ble_serial->attach(&ble_recv, Serial::RxIrq);
    _console->printf("\r\nStart ...");
    _console->printf("\r\nPress ? to show menu ...\r\n");

    ThisThread::sleep_for(1s);

    //_ble_serial->write("AT+ADDR=?\r\n", 11);
    //ThisThread::sleep_for(1s);
    //_ble_serial->write("AT+VERSION\r\n", 12);

    while(1)
    {
        console_recv();
        ble_recv();    
    }
#endif

    _console->printf("\r\nATCmdParser on ePy-Lite example\r\n");
    ThisThread::sleep_for(2s);

#if 0
    if (_parser->recv("READY OK"))
    {
        _console->printf("\r\nModule is ready!");
    }
#endif

#if 0
    // Assume BLE module is in DATA mode, switch to CMD mode
    if (_parser->send("!CCMD@") && _parser->recv("OK"))
    {
        _console->printf("\r\nSwitch to CMD mode success.");
    }
    else {
        _console->printf("\r\nSwitch to Cmd mode timeout.");
        // Assume already in CMD mode
    }
#endif

    //Now get the FW version number of BLE module by sending an AT command 
    _console->printf("\r\nATCmdParser: Retrieving FW version");

#if 0
    while(true)
    {
        char s[256];
        if(_parser->read(s, 256) < 0)
            break;

        _console->printf("%s\r\n", s);
    }
    _parser->write("AT+VERSION\r\n",12);
#endif

    _parser->write("AT+VERSION\r\n",12);
    
    int ver_major, ver_minor, ver_patch;
    if(_parser->recv("VERSION %d.%d.%d OK", &ver_major, &ver_minor, &ver_patch) /* && _parser->recv("OK") */) {
        _console->printf("\r\nATCmdParser: FW version: %d.%d.%d", ver_major, ver_minor, ver_patch);
        _console->printf("\r\nATCmdParser: Retrieving FW version success");
    } else { 
        _console->printf("\r\nATCmdParser: Retrieving FW version failed");
        return -1;
    }

    _parser->write("AT+ADDR=?\r\n",11);

#if 1
    if (_parser->recv("MAC_ADD"))
    {
        char mac_addr[20];
        _parser->read(mac_addr, 14);
        _console->printf("\r\nMAC Addr is %s\r\n", mac_addr);

    }
#else
    long int mac_addr;
    if(_parser->recv("MAC_ADDR %lx OK", &mac_addr) /* && _parser->recv("OK") */)
    {
        _console->printf("\r\nATCmdParser: MAC Address : %lx", mac_addr);
        _console->printf("\r\nATCmdParser: Retrieving MAC Address success");
    } else { 
        _console->printf("\r\nATCmdParser: Retrieving FW version failed");
        return -1;
    }
#endif

    _console->printf("\r\nDone\r\n");
}
