/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@author Louis Wray
@date   14th April 2015
*/

#ifndef MAIN_H
#define MAIN_H
#include "mbed.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"
#include "DebounceIn.h"
#define DIRECTION_TOLERANCE 0.05

/**
@namespace LocalFileSystem
@brief defines a local filesystem
*/
LocalFileSystem local("local"); // create local filesystem

/**
@namespace button
@brief represents button input pin from joystick
*/
DebounceIn button(p17);

/**
@namespace buzzer
@brief represents buzzer output pin from pwm
*/
PwmOut PWM1(p25);

/**
@namespace xPot
@brief analog input from potentiometer reading x position
*/
AnalogIn xPot(p15);

/**
@namespace yPot
@brief analog input from potentiometer reading y position
*/
AnalogIn yPot(p16);

/**
@namespace randomPin
@brief analog input to seed random function
*/
AnalogIn randomPin(p19);

/**
@namespace serial
@brief serial connection established
*/
Serial serial(USBTX,USBRX);

/**
@namespace myled
@brief GPIO for on board mbed LED1
*/
DigitalOut myled(LED1);

/**
@namespace lcd
@brief creates object for display.
@brief shows pins used and there configuration
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26); // pin settings

/**
@namespace buttonS
@brief creates interrupt in for shooting function.
*/
DebounceIn buttonS(p22); // create interrupt button shoot

/**
@namespace buttonP
@brief creates interrupt in for other functions.
*/
DebounceIn buttonP(p23); // create interrupt button

/**
@brief class created for user component ship.
*code will access the components of this ship class to control user component on the screen
*/
class Ship
{
public:
    int xs; /**< ships integer x coordinate */
    int ys; /**< ships integer y coordinate */

    /**
    *set the values for the ship, used for drawing and moving etc
    */
    void SetValues(int,int);

    /**
    *function to draw ships shape
    @param xs - ships integer x coordinate
    @param ys - ships integer y coordinate
    */
    void shape() {
        lcd.drawLine(xs,ys,xs-5,ys+6,1);
        lcd.drawLine(xs,ys,xs+5,ys+6,1);
        lcd.drawLine(xs-5,ys+6,xs+5,ys+6,1);
        lcd.setPixel(xs-3,ys+7);
        lcd.setPixel(xs+3,ys+7);
        // lcd.refresh();
    }

    /**
    *function for moving the ship to the left
    */
    void ShipLeft() {
        xs = xs-1;
        clearShapeL();
        shape();
    }

    /**
    *function for moving the ship to the right
    */
    void ShipRight() {
        xs = xs+1;
        clearShapeR();
        shape();
    }

    /**
    *function for clearing the previous shape when moved to the left
    */
    void clearShapeL() {
        lcd.drawLine(xs+1,ys,xs-4,ys+6,0);
        lcd.drawLine(xs+1,ys,xs+6,ys+6,0);
        lcd.drawLine(xs-4,ys+6,xs+6,ys+6,0);
        lcd.clearPixel(xs-2,ys+7);
        lcd.clearPixel(xs+4,ys+7);
    }

    /**
    *function for clearing the previous shape when moved to the right
    */
    void clearShapeR() {
        lcd.drawLine(xs-1,ys,xs-6,ys+6,0);
        lcd.drawLine(xs-1,ys,xs+4,ys+6,0);
        lcd.drawLine(xs-6,ys+6,xs+4,ys+6,0);
        lcd.clearPixel(xs-4,ys+7);
        lcd.clearPixel(xs+2,ys+7);
    }
} ship;

/** function which sets the ships xs and ys coordinates, used in the main program as an initialisation
@param X - integer value to set xs
@param Y - integer value to set ys
*/
void Ship::SetValues(int X,int Y)
{
    xs = X;
    ys = Y;
}

/**
@brief class created for bullet fired.
*code will access the components of this Bullet class to control the bullet
*/
class Bullet
{
public:
    int x; /**< bullets integer x coordinate */
    int y; /**< bullets integer y coordinate */

    /**
    *set the values for the bullet, used when first fired for other commands to work off
    */
    void SetValues(int,int);

    /**
    *function to draw bullets shape
    @param x - bullets integer x coordinate
    @param y - bullets integer y coordinate
    */
    void createBullet() {
        lcd.setPixel(x,y);
        lcd.setPixel(x+1,y);
        lcd.setPixel(x-1,y);
        lcd.setPixel(x,y-1);
        lcd.refresh();
    }

    /**
    *function to erase previous bullets when moving forward
    *always followed by create bullet so no need for lcd refresh when drawing
    */
    void erasePrevBullet() {
        lcd.clearPixel(x,y+1);
        lcd.clearPixel(x+1,y+1);
        lcd.clearPixel(x-1,y+1);
    }
    
    /**
    *bool function, which determines whether the bullet has collided with any enemies returns true or false
    */
    bool collision() {
        if(lcd.getPixel(x+1,y-1)||lcd.getPixel(x-1,y-1)||lcd.getPixel(x,y-2)) {
            clearBullet();
            return true;
        } else {return false;}
    }
    /**
    *bool function, which determines whether the bullet has reached the edge of the screen returns true or false
    */
    bool offScreen() {
        if(y<4) {
            clearBullet();
            return true;
        } else {
            return false;
        }
    }
    /**
    *function to clear bullets current position
    */
    void clearBullet() {
        lcd.clearPixel(x,y);
        lcd.clearPixel(x+1,y);
        lcd.clearPixel(x-1,y);
        lcd.clearPixel(x,y-1);
        lcd.refresh();
    }
} bullet;

/** function which sets the bullets x and y coordinates, used in the shooting program to initialise where the bullet was fired
@param Xb - integer value to set x
@param Yb - integer value to set y
*/
void Bullet::SetValues(int Xb,int Yb)
{
    x = Xb;
    y = Yb;
}

/**
@brief class created for Asteroids.
*code will access the components of this Asteroid class to control the Asteroid
*/
class Asteroid
{
public:
    int xa; /**< asteroids integer x coordinate */
    int ya; /**< asteroids integer y coordinate */
    int nscore; /**< an integer value score that increments every time the player has destroyed an asteroid */

    /**
    *set the x value for the asteroid, used at intervals called by a ticker
    */
    void SetAsteroidValue(int,int);

    /**
    *function to draw and asteroid according to given coordinates
    */
    void createAsteroid() {
        lcd.drawCircle(xa,ya,3,0,1);
        lcd.refresh();
    }

    /**
    *function to clear previous asteroid when moving forward(forward being from the aseroids point of view, looking at
    *the screen would be downwards)
    */
    void erasePrevAsteroid() {
        lcd.drawCircle(xa,ya-1,3,0,0);
        lcd.refresh();
    }

    /**
    *function to eliminate asteroid, this will be called when a bullet has made a collision with the asteroid
    */
    void destroyAsteroid() {
        lcd.drawCircle(xa,ya,3,0,0);
        nscore = nscore+1;
        bullet.clearBullet();
        lcd.refresh();
    }

    /**
    *boolean function which returns true or false, determining whether the asteroid has reached the bottom of the screen or not
    */
    bool armageddon() {
        if(ya+4 == 47) {
            return true;
        } else {
            return false;
        }
    }
} asteroid;

/** function which sets the asteroids x position (y posisition is set to top of the screen)
@param XA - integer value to set xa
@param YA - integer value to set ya
*/
void Asteroid::SetAsteroidValue(int XA, int YA)
{
    xa = XA;
    ya = YA;
}

/**
*function prototype to caibrate the joystick and centre values
*/
void calibrateJoystick();

/**
*function which when called upon will update the joystick position
*/
void updateJoystick();

/**
*function which shows initial display and sleeps(low power sleep) untill interrupt made and game started
*/
void startUpDisplay();

/**
*function for gameplay
*/
void Game();

/**
*function which reads the position of the bullet on screen and reprints according to position also checking
*whether it has made contact with enemies or the end of the screen
*/
void ReadBullet();

/**
*function which reads the position and state of the asteroid and refeshes accordingly
*/
void ReadAsteroid();

/**
*function which shows end display with score, and sleeps(low power sleep) untill interrupt made and game re-started
*/
void EndDisplay();

/**
*function which shows Pause Display and will restart when shoot button pressed , sleep() mode untill then
*/
void PauseDisplay();

/**
*funtion to read from the local file system created
*used in highscore
*/
void Read();

/**
*funtion to write to the local file system created
*used in highscore
*/
void Write();

/**
*funtion to flag the end screen for timeout
*/
void endState();



#endif