/*
 * Project: Analog Clock - Display an analog clock with NXP LPC1768 or LPC1114
 * File: main.cpp
 * Author: Steve Baumann
 * Created: December, 2015
 * Revised: 
 * Description:
 * --------------
 * The time read from tinyRTC (DS1307) will be displayed on a 96x96 Sharp Memory LCD 
 * LS013B4DN04 (bought from Adafruit http://www.adafruit.com/products/1393).
 * The RTC-lib created by Henry Leinen is used and the algorithm for calculation the
 * clock-hands is based on input from http://arduino-project.net
 * The RTC itself was bought via Amazon http://www.amazon.de/gp/product/B00CWX6UXY/
 */
 
#include <algorithm>
#include "mbed.h"
#include "Adafruit_SharpMem.h"
#include "Rtc_Ds1307.h"

#ifdef TARGET_LPC176X
// Display LS013B4DN04
#define ENABLE p15  // DISP
#define CS p8       // CS
#define MOSI p5     // DI
#define MISO p6     // not used (no wiring necessary)
#define SCLK p7     // CLK
// RTC 1307
#define SDA p28     // SDA for RTC
#define SCL p27    // SCL for RTC
#endif

#ifdef TARGET_LPC11XX
// Display LS013B4DN04
#define ENABLE dp9  // DISP
#define CS dp10     // CS
#define MOSI dp2    // DI
#define MISO dp1    // not used (no wiring necessary)
#define SCLK dp6    // CLK
// RTC 1307
#define SDA dp5     // SDA for RTC
#define SCL dp27    // SCL for RTC
#endif

Rtc_Ds1307 rtc(SDA, SCL);
Adafruit_SharpMem display(ENABLE, CS, MOSI, MISO, SCLK);

void drawClockMark(int hour)
{
    float x1, y1, x2, y2;
    
    hour = hour*30;
    hour = hour+270;
    
    x1 = 48*cos(hour*0.0175);
    y1 = 48*sin(hour*0.0175);
    x2 = 43*cos(hour*0.0175);
    y2 = 43*sin(hour*0.0175);
    
    display.drawLine(x1+48, y1+48, x2+48, y2+48, BLACK);
}

void drawClock()
{
    // Values for radius and positioning of labels need to be adapt for other displays
    display.setTextSize(1);
    display.setTextColor(BLACK);
    
    display.drawCircle(display.width()/2, display.height()/2, 48, BLACK);
    display.setCursor(88,44);
    display.printf("3");
    display.setCursor(46,88);
    display.printf("6");
    display.setCursor(3,44);
    display.printf("9");
    display.setCursor(43,3);
    display.printf("12");
    
    for (int i=0; i<12; i++)
    {
        if((i%3) != 0) drawClockMark(i);
    }
}

void drawSecond(int s)
{
    float x1, y1, x2, y2;
    
    s = s*6;
    s = s+270;
    
    x1 = 45*cos(s*0.0175);
    y1 = 45*sin(s*0.0175);
    x2 = 4*cos(s*0.0175);
    y2 = 4*sin(s*0.0175);
    
    display.drawLine(x1+48, y1+48, x2+48, y2+48, BLACK);
}

void drawMinute(int m)
{
    float x1, y1, x2, y2;
    
    m = m*6;
    m = m+270;
    
    x1 = 45*cos(m*0.0175);
    y1 = 45*sin(m*0.0175);
    x2 = 4*cos(m*0.0175);
    y2 = 4*sin(m*0.0175);
    
    display.drawLine(x1+48, y1+48, x2+48, y2+48, BLACK);
}

void drawHour(int h, int m)
{
    float x1, y1, x2, y2;
    
    h = (h*30)+(m/2);
    h = h+270;
    
    x1 = 30*cos(h*0.0175);
    y1 = 30*sin(h*0.0175);
    x2 = 4*cos(h*0.0175);
    y2 = 4*sin(h*0.0175);
    
    display.drawLine(x1+48, y1+48, x2+48, y2+48, BLACK);
}

int main()
{
    Rtc_Ds1307::Time_rtc tm = {};
    
    // start & clear the display
    display.begin();
    display.setRotation(0);
    display.enableDisplay();
    display.clearDisplay();
    
    while(1) {
        
        drawClock();
        
        if (rtc.getTime(tm)) { 
            drawSecond(tm.sec);
            drawMinute(tm.min);
            drawHour(tm.hour, tm.min);
            display.refresh();
        } else {
            tm.date = 6; tm.mon = 12; tm.year = 2015;
            tm.hour = 13; tm.min = 50; tm.sec = 0;
            rtc.setTime(tm, false, false);
            rtc.startClock();
        }
        wait(0.5);
        display.clearDisplay();
    }
}