#include "mbed.h"
#include "Adafruit_Trellis.h"

#define HT16K33_BLINK_CMD       0x80
#define HT16K33_BLINK_DISPLAYON 0x01
#define HT16K33_CMD_BRIGHTNESS  0xE0

/*
 These are the lookup tables that convert the LED/button #
 to the memory address in the HT16K33 - don't mess with them :)
 */

static const uint8_t ledLUT[16] =
{ 0x3A, 0x37, 0x35, 0x34,
    0x28, 0x29, 0x23, 0x24,
    0x16, 0x1B, 0x11, 0x10,
    0x0E, 0x0D, 0x0C, 0x02 },
buttonLUT[16] =
{ 0x07, 0x04, 0x02, 0x22,
    0x05, 0x06, 0x00, 0x01,
    0x03, 0x10, 0x30, 0x21,
    0x13, 0x12, 0x11, 0x31 };

Adafruit_Trellis::Adafruit_Trellis(void) {
}

void Adafruit_Trellis::begin(I2C *_wire, uint8_t _addr = 0x70) {
    // The mbed API uses 8 bit addresses, so make sure to take
    // that 7 bit address and left shift it by 1 before passing it.
    i2c_addr = (_addr << 1);
    // store pointer to the I2C class object
    wire = _wire;
    
    // turn on oscillator
    cmd[0] = 0x21;
    wire->write(i2c_addr, cmd, 1);
    // set blink rate
    blinkRate(HT16K33_BLINK_OFF);
    // set brightness
    setBrightness(15); // max brightness
    // setup interrupt, active low
    cmd[0] = 0xA1;
    wire->write(i2c_addr, cmd, 1);
}

/*
 Helper button functions, the data is updated every readSwitches() call!
 */

bool Adafruit_Trellis::isKeyPressed(uint8_t k) {
    if (k > 15) return false;
    k = buttonLUT[k];
    return (keys[k>>4] & _BV(k & 0x0F));
}
bool Adafruit_Trellis::wasKeyPressed(uint8_t k) {
    if (k > 15) return false;
    k = buttonLUT[k];
    return (lastkeys[k>>4] & _BV(k & 0x0F));
}

bool Adafruit_Trellis::justPressed(uint8_t k) {
    return (isKeyPressed(k) & !wasKeyPressed(k));
}
bool Adafruit_Trellis::justReleased(uint8_t k) {
    return (!isKeyPressed(k) & wasKeyPressed(k));
}

/*
 Helper LED functions, the data is written on writeDisplay()
 */


bool Adafruit_Trellis::isLED(uint8_t x) {
    if (x > 15) return false;
    x = ledLUT[x];
    return ((displaybuffer[x >> 4] & _BV(x & 0x0F)) > 0);
}
void Adafruit_Trellis::setLED(uint8_t x) {
    if (x > 15) return;
    x = ledLUT[x];
    displaybuffer[x >> 4] |= _BV(x & 0x0F);
}
void Adafruit_Trellis::clrLED(uint8_t x) {
    if (x > 15) return;
    x = ledLUT[x];
    displaybuffer[x >> 4] &= ~_BV(x & 0x0F);
}


/*
 Gets the switch memory data and updates the last/current read
 */

bool Adafruit_Trellis::readSwitches(void) {
    memcpy(lastkeys, keys, sizeof(keys));
    
    cmd[0] = 0x40;
    wire->write(i2c_addr, cmd, 1);
    wire->read(i2c_addr, cmd, 6);
    
    for (uint8_t i=0; i<6; i++)
    keys[i] = cmd[i];
    
    for (uint8_t i=0; i<6; i++) {
        if (lastkeys[i] != keys[i]) {
            for (uint8_t j=0; j<6; j++) {
                //Serial.print(keys[j], HEX); Serial.print(" ");
            }
            //Serial.println();
            return true;
        }
    }
    return false;
}

void Adafruit_Trellis::setBrightness(uint8_t b) {
    if (b > 15) b = 15;
    cmd[0] = HT16K33_CMD_BRIGHTNESS | b;
    wire->write(i2c_addr, cmd, 1);
}

void Adafruit_Trellis::blinkRate(uint8_t b) {
    if (b > 3) b = 0; // turn off if not sure
    cmd[0] = HT16K33_BLINK_CMD | HT16K33_BLINK_DISPLAYON | (b << 1);
    wire->write(i2c_addr, cmd, 1);
}


void Adafruit_Trellis::writeDisplay(void) {
    cmd[0] = 0x00;
    for (uint8_t i=0; i<8; i++) {
        cmd[2*i+1] = displaybuffer[i] & 0xFF;
        cmd[2*i+2] = displaybuffer[i] >> 8;
    }
    wire->write(i2c_addr, cmd, 17);
    
}

void Adafruit_Trellis::clear(void) {
    memset(displaybuffer, 0, sizeof(displaybuffer));
}


