#include "mbed.h"
#include "rtos.h"
#include "ble/BLE.h"

// Define the interface to the leds, they are digital outputs when can be
// enabled or disabled
enum { total_leds = 4 };
DigitalOut led[total_leds] = { D6, D7, D8, D9 };

// Function to enable led with index 0..3
void EnableLed(uint8_t index)
{
    if(index < total_leds)
        led[index] = 0;
}

// Function to disable led with index 0..3
void DisableLed(uint8_t index)
{
    if(index < total_leds)
        led[index] = 1;
}

// Define the interface to the buttons, they are configured to 
// generate an interrupt and call the ButtonPress function
enum { total_buttons = 4 };
InterruptIn buttons[total_buttons] = { D2, D3, D4, D5 };

// The name of this device
// TODO .. Change the name of this device
const static char DEVICE_NAME[] = "Nioc_BLE";

// The UUID (Unique Universal Identifier) of our Alert service and Alert characteristic
enum { ALERT_SERVICE_UUID = 0x1811 };
enum { ALERT_VALUE_UUID = 0x2A06 };

// Definition our alert characteristic
ReadWriteGattCharacteristic<uint8_t> alert_characteristic(ALERT_VALUE_UUID, NULL, 
    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);

// This function is called when data is written to a characteristic
void onDataWrittenCallback(const GattWriteCallbackParams *params)
{
    // Check if it really was the alert characteristic that was written by checking the handle
    if(params->handle == alert_characteristic.getValueHandle() && params->len == sizeof(uint8_t))
    {
        // We got a new value for the alert characteristic, do something useful with this
        uint8_t value = *(uint8_t const *)(params->data);
        // TODO: .. do something useful with this value
    }
}

// This function is called when the gatt connection is disconnected
// we have to manually re-enable the advertisements again when this occurs
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *)
{
    BLE::Instance().gap().startAdvertising();
}

// This function is called when the BLE stack is initialized
// Setup the GATT Service and Advertisement here for the example
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    if(params->error != BLE_ERROR_NONE)
        return;
    
    BLE &ble = BLE::Instance();
    
    // Initialize the default of our alert characteristic
    uint8_t init_alert_value = 0;
    BLE::Instance().gattServer().write(alert_characteristic.getValueHandle(), &init_alert_value, sizeof(uint8_t)); 
    
    // First create our GATT Service (basicly a list of characteristics )
    enum { TOTAL_SERVICE_CHARACTERTERISTICS = 1 };
    GattCharacteristic *characteristics_table[TOTAL_SERVICE_CHARACTERTERISTICS] = { &alert_characteristic };
    GattService alert_service(ALERT_SERVICE_UUID, characteristics_table, TOTAL_SERVICE_CHARACTERTERISTICS);
    ble.gattServer().addService(alert_service);
    
    // Setup the BLE callbacks    
    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);
    
    // Configure the advertisement, we add the name of the device to the advertisement and some service data
    ble.gap().setDeviceName((uint8_t const *)DEVICE_NAME);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME, (uint8_t const *)DEVICE_NAME, sizeof(DEVICE_NAME) - 1);
    
    uint8_t service_data[] = { 0x11, 0x18, 0x00 };
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SERVICE_DATA, service_data, sizeof(service_data));

    // Start advertising our device information now
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(250);
    ble.gap().startAdvertising();
}

// This function is called when a button is pressed
void ButtonPressed(uint8_t index)
{
    // TODO: ... do something useful here when a button is pressed
}

// Eventqueue to handle incoming events
EventQueue eventQueue(16 * EVENTS_EVENT_SIZE);

// Function required to process events from the BLE Stack
void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext *context)
{
    eventQueue.call(Callback<void()>(&BLE::Instance(), &BLE::processEvents));
}

// Let's setup the BLE stack
int main(void)
{    
    // Configure the led to the default state
    EnableLed(0);
    DisableLed(1);
    DisableLed(2);
    DisableLed(3);
    
    // Configure the button callbacks
    buttons[0].mode(PullUp);
    buttons[0].fall(eventQueue.event(ButtonPressed, 0));
    buttons[1].mode(PullUp);
    buttons[1].fall(eventQueue.event(ButtonPressed, 1));
    buttons[2].mode(PullUp);
    buttons[2].fall(eventQueue.event(ButtonPressed, 2));
    buttons[3].mode(PullUp);
    buttons[3].fall(eventQueue.event(ButtonPressed, 3));
    
    // Start the BLE Stack 
    BLE::Instance().onEventsToProcess(scheduleBleEventsProcessing);
    BLE::Instance().init(bleInitComplete);
    eventQueue.dispatch_forever();
    return 0;
}