#include "mbed.h"
#include "NokiaLCD.h"
#include "PS2Keyboard.h"
#include "ball.h"
#include "paddle.h"

// State enumerator
typedef enum {
  RESET, RUN, PAUSE
} STATES;

NokiaLCD lcd(p5, p7, p8, p9, NokiaLCD::LCD6610); // mosi, sclk, cs, rst, type

PS2Keyboard ps2kb(p12, p11); // CLK, DAT

DigitalIn sw2(p24);
DigitalIn sw1(p25);

PwmOut g(p21);
PwmOut b(p22);
PwmOut r(p23);

// Button enumerator for PS/2 keyboard
enum BUTTONS{
  UP = 0xe75,
  DOWN = 0xe72,
};

/* 
 * Subroutine drawScreen:
 * Description: Draws both paddles
 *  and the ball.
 */
void drawScreen(Paddle paddle1, Paddle paddle2, Ball theBall, bool isBlack) {
  paddle1.draw(lcd, isBlack);
  paddle2.draw(lcd ,isBlack);
  theBall.draw(lcd ,isBlack);
}

/* 
 * Subroutine drawScores:
 * Description: Draws the scoreboard
 */
void drawScores(Paddle paddle1, Paddle paddle2) {
  lcd.locate(7,0);
  lcd.putc('0' + paddle1.getScore());
  lcd.locate(9,0);
  lcd.putc('0' + paddle2.getScore());
  lcd.fill(66,0,2,130,0xFFFFFF);
  lcd.locate(7,15);
  lcd.putc('0' + paddle1.getLives());
  lcd.locate(9,15);
  lcd.putc('0' + paddle2.getLives());
}

int main() {
  PS2Keyboard::keyboard_event_t evt_kb; // Setup keyboard interrupt
  lcd.background(0x000000);
  lcd.cls();
  Paddle paddle1, paddle2;
  Ball theBall;
  int temp, count=0;
  drawScreen(paddle1, paddle2, theBall, false);
  drawScores(paddle1, paddle2);
  STATES state = RESET; // Initial state is RESET
  while(1) {
    switch(state) {
      case RESET: // Reset objects, draw the screen, state = PAUSE
        lcd.cls();
        paddle1 = Paddle(1,10,5,25,0xFFFFFF,paddle1.getLives(),paddle1.getScore());
        paddle2 = Paddle(125,3,5,25,0xFFFFFF,paddle2.getLives(),paddle2.getScore());
        theBall = Ball(6,25,5,5,0xFFFF00,1,1);
        drawScreen(paddle1, paddle2, theBall, false);
        drawScores(paddle1, paddle2);
        state = PAUSE;
        break;
      case PAUSE: // Set RGB LED to Red, wait for switch input
        r = 0;
        b = g = 1;
        if(!sw2) {
          while(!sw2);
          paddle2.setLives(3);
          paddle1.setLives(3);
          state = RESET;
          break;
        }
        if(!sw1) {
          while(!sw1);
          state = RUN;
        }
        break;
      case RUN: // Set RGB LED to Blue and run program
        r = g = 1;
        b = 0;
        if(!sw2) { // Reset if SW2 is pressed
          while(!sw2);
          paddle2.setLives(3);
          paddle1.setLives(3);
          state = RESET;
          break;
        }
        if(!sw1) { // Pause if SW1 is pressed
          while(!sw1);
          state = PAUSE;
          break;
        }
        if (ps2kb.processing(&evt_kb)) { // Executes if a key is pressed
          temp = evt_kb.scancode[0];
          for (int i = 1; i < evt_kb.length; i++) { // Parse keyboard input into a key
            temp <<= 4;
            temp |= evt_kb.scancode[i];
          }
          switch(temp) { // Use key enumerator to move paddle1
            case UP:
              if(paddle1.getY()>2)
                paddle1.move(lcd, -2);
              break;
            case DOWN: 
              if(paddle1.getY()+paddle1.getHeight()<128)
                paddle1.move(lcd, 2);
              break;
          }
        }
        if(count%2) // Only let CPU move once every 2 times through the loop
          paddle2.moveCPU(lcd, theBall.getY());
        if(++count==5) { // Only move the ball once every 5 times through the loop
          count = 0;
          if(theBall.hitP1((paddle1.getX()+paddle1.getWidth()), paddle1.getY(), paddle1.getHeight()))
            theBall.reverseX();
          if(theBall.hitP2(paddle2.getX(), paddle2.getY(), paddle2.getHeight()))
            theBall.reverseX();
          if(theBall.hitX()) { // If the ball hits one of the sides of the screen
            if(theBall.getX()<7) { // If the ball hit paddle1's side
              if(!paddle1.loseLife()) { // If paddle1 has no more lives
                paddle1.setLives(3);
                paddle2.setLives(3);
                paddle2.addPoint();
              }
            }
            else if(theBall.getX()>120) { // If the ball hit paddle2's side
              if(!paddle2.loseLife()) { // If paddle2 has no more lives
                paddle2.setLives(3);
                paddle1.setLives(3);
                paddle1.addPoint();
              }
            }
            theBall.reverseX();
            state = RESET; // Reset the objects
          }
          if(theBall.hitY()) 
            theBall.reverseY();
          theBall.move(lcd);
        }
        break;
    }
    drawScreen(paddle1, paddle2, theBall, false);
    drawScores(paddle1, paddle2);
  }
}
