/*
 *     ____             _________                __                _
 *    / __ \___  ____ _/ /_  __(_)___ ___  ___  / /   ____  ____ _(_)____
 *   / /_/ / _ \/ __ `/ / / / / / __ `__ \/ _ \/ /   / __ \/ __ `/ / ___/
 *  / _, _/  __/ /_/ / / / / / / / / / / /  __/ /___/ /_/ / /_/ / / /__
 * /_/ |_|\___/\__,_/_/ /_/ /_/_/ /_/ /_/\___/_____/\____/\__, /_/\___/
 *                                                       /____/
 *
 *                 SharkSSL Embedded SSL/TLS Stack
 ****************************************************************************
 *   PROGRAM MODULE
 *
 *   $Id: SharkSslCrypto.h 3686 2015-04-27 06:35:45Z gianluca $
 *
 *   COPYRIGHT:  Real Time Logic LLC, 2010 - 2013
 *
 *   This software is copyrighted by and is the sole property of Real
 *   Time Logic LLC.  All rights, title, ownership, or other interests in
 *   the software remain the property of Real Time Logic LLC.  This
 *   software may only be used in accordance with the terms and
 *   conditions stipulated in the corresponding license agreement under
 *   which the software has been supplied.  Any unauthorized use,
 *   duplication, transmission, distribution, or disclosure of this
 *   software is expressly forbidden.
 *
 *   This Copyright notice may not be removed or modified without prior
 *   written consent of Real Time Logic LLC.
 *
 *   Real Time Logic LLC. reserves the right to modify this software
 *   without notice.
 *
 *               http://www.realtimelogic.com
 *               http://www.sharkssl.com
 ****************************************************************************
 *
 */
#ifndef _SharkSslCrypto_h
#define _SharkSslCrypto_h

#define SHARKSSL_LIB 1
#include "SharkSSL.h"


#define SHARKSSL_MD5_HASH_LEN       16
#define SHARKSSL_SHA1_HASH_LEN      20
#define SHARKSSL_SHA256_HASH_LEN    32
#define SHARKSSL_SHA384_HASH_LEN    48
#define SHARKSSL_SHA512_HASH_LEN    64
#define SHARKSSL_POLY1305_HASH_LEN  16


#if SHARKSSL_USE_MD5
/** MD5
    \ingroup RayCryptoMD5
*/
typedef struct SharkSslMd5Ctx
{
   U32 total[2];
   U32 state[4];
   U8  buffer[64];
} SharkSslMd5Ctx;
#endif


#if SHARKSSL_USE_SHA1
/** SHA1
    \ingroup RayCryptoSHA1
*/
typedef struct SharkSslSha1Ctx
{
   U32 total[2];
   U32 state[5];
   U8  buffer[64];
} SharkSslSha1Ctx;
#endif


#if SHARKSSL_USE_SHA_256
/** SHA256
    \ingroup RayCryptoSHA256
*/
typedef struct SharkSslSha256Ctx
{
   U32 total[2];
   U32 state[8];
   U8  buffer[64];
} SharkSslSha256Ctx;
#endif


#if (SHARKSSL_USE_SHA_384 || SHARKSSL_USE_SHA_512)
/** SHA384
    \ingroup RayCryptoSHA384
*/
typedef struct SharkSslSha384Ctx
{
   U32 total[4];
   U64 state[8];
   U8  buffer[128];
} SharkSslSha384Ctx;
#endif


#if SHARKSSL_USE_SHA_512
/** SHA512
    \ingroup RayCryptoSHA512
*/
typedef struct SharkSslSha384Ctx SharkSslSha512Ctx;
#endif


#if SHARKSSL_USE_POLY1305
/** POLY1305
    \ingroup RayCryptoPOLY1305
*/
typedef struct SharkSslPoly1305Ctx
{
   U32 r[5];
   U32 key[4];
   U32 nonce[4];
   U8  buffer[16];
   U8  flag, blen;
} SharkSslPoly1305Ctx;
#endif


#if SHARKSSL_USE_CHACHA20
/** CHACHA20
    \ingroup RayCryptoCHACHA20
*/
typedef struct
{
   U32 state[16];
} SharkSslChaChaCtx;
#endif


#if SHARKSSL_USE_ARC4
/** ARC4
    \ingroup RayCryptoARC4
*/
typedef struct SharkSslArc4Ctx
{
   U8 x;
   U8 y;
   U8 b[256];
} SharkSslArc4Ctx;
#endif


#if (SHARKSSL_USE_DES || SHARKSSL_USE_3DES)
/** DES
    \ingroup RayCryptoDES
*/
typedef struct SharkSslDesCtx
{
   #if SHARKSSL_NOPACK
   U32 key[96]; /* Max size == 3DES */
   U8  tdea;
   #else

   #if (SHARKSSL_USE_CAU || SHARKSSL_USE_MMCAU)
   #if SHARKSSL_USE_3DES
   U32 key[6];
   #else
   U32 key[2];
   #endif

   #else

   #if SHARKSSL_USE_3DES
   U32 key[96];
   #else
   U32 key[32];
   #endif
   #endif

   #if (SHARKSSL_USE_DES && SHARKSSL_USE_3DES)
   U8  tdea;
   #endif

   #endif
} SharkSslDesCtx;


/* Encrypt/decrypt type */
typedef enum
{
   SharkSslDesCtx_Decrypt, /*!< Decrypt */
   SharkSslDesCtx_Encrypt /*!< Encrypt */
} SharkSslDesCtx_Type;
#endif


#if (SHARKSSL_USE_AES_256 || SHARKSSL_USE_AES_192 || SHARKSSL_USE_AES_128)
/** AES
    \ingroup RayCryptoAES
*/
typedef struct SharkSslAesCtx
{
   #if (SHARKSSL_USE_AES_256 || SHARKSSL_NOPACK)
   U32 key[60];
   #elif SHARKSSL_USE_AES_192
   U32 key[52];
   #else
   U32 key[44];
   #endif
   U16 nr;
} SharkSslAesCtx;

typedef enum
{
   SharkSslAesCtx_Decrypt,
   SharkSslAesCtx_Encrypt
} SharkSslAesCtx_Type;

#if SHARKSSL_ENABLE_AES_GCM
/** AesGcmCtx
    \ingroup RayCryptoAesGcm
*/
typedef struct SharkSslAesGcmCtx
{
   SharkSslAesCtx super;
   U8 M0[16][16];
} SharkSslAesGcmCtx;
#endif

#if SHARKSSL_ENABLE_AES_CCM
/** AesCcmCtx
    \ingroup RayCryptoAesCcm
*/
typedef struct SharkSslAesCcmCtx
{
   SharkSslAesCtx super;
   U8 tagLen;
} SharkSslAesCcmCtx;
#endif
#endif  /* SHARKSSL_USE_AES_256 || SHARKSSL_USE_AES_192 || SHARKSSL_USE_AES_128 */


#ifdef __cplusplus
extern "C" {
#endif

/* SharkSslCrypto.c */
SHARKSSL_API int   sharkssl_entropy(U32);
SHARKSSL_API int   sharkssl_rng(U8*, U16);
SHARKSSL_API int   sharkssl_kmemcmp(const void *a, const void *b, U32 n);


#if SHARKSSL_USE_MD5
/** Initialize
    \ingroup RayCryptoMD5

    \param ctx Uninitialized data of size sizeof(SharkSslMd5Ctx).
*/
SHARKSSL_API void  SharkSslMd5Ctx_constructor(SharkSslMd5Ctx* ctx);

/** append
    \ingroup RayCryptoMD5
*/
SHARKSSL_API void  SharkSslMd5Ctx_append(SharkSslMd5Ctx* ctx, const U8* data, U32 len);

/** finish
    \ingroup RayCryptoMD5
*/
SHARKSSL_API void  SharkSslMd5Ctx_finish(SharkSslMd5Ctx* ctx, U8 digest[SHARKSSL_MD5_HASH_LEN]);

/** md5
    \ingroup RayCryptoMD5
*/
SHARKSSL_API int   sharkssl_md5(const U8*, U16, U8*);
#endif

#if SHARKSSL_USE_SHA1
/** Initialize
    \ingroup RayCryptoSHA1

    \param ctx Uninitialized data of size sizeof(SharkSslSha1Ctx).
*/
SHARKSSL_API void  SharkSslSha1Ctx_constructor(SharkSslSha1Ctx* ctx);

/** append
    \ingroup RayCryptoSHA1
*/
SHARKSSL_API void  SharkSslSha1Ctx_append(SharkSslSha1Ctx* ctx, const U8* data, U32 len);

/** finish
    \ingroup RayCryptoSHA1
*/
SHARKSSL_API void  SharkSslSha1Ctx_finish(SharkSslSha1Ctx*, U8 digest[SHARKSSL_SHA1_HASH_LEN]);

/** sha1
    \ingroup RayCryptoSHA1
*/
SHARKSSL_API int   sharkssl_sha1(const U8*, U16, U8*);
#endif

#if SHARKSSL_USE_SHA_256
/** Initialize
    \ingroup RayCryptoSHA256

    \param ctx Uninitialized data of size sizeof(SharkSslSha256Ctx).
*/
SHARKSSL_API void  SharkSslSha256Ctx_constructor(SharkSslSha256Ctx* ctx);

/** append
    \ingroup RayCryptoSHA256
*/
SHARKSSL_API void  SharkSslSha256Ctx_append(SharkSslSha256Ctx*, const U8* data, U32 len);

/** finish
    \ingroup RayCryptoSHA256
*/
SHARKSSL_API void  SharkSslSha256Ctx_finish(SharkSslSha256Ctx*, U8 digest[SHARKSSL_SHA256_HASH_LEN]);

/** sha256
    \ingroup RayCryptoSHA256
*/
SHARKSSL_API int   sharkssl_sha256(const U8*, U16, U8*);
#endif

#if SHARKSSL_USE_SHA_384
/** Initialize
    \ingroup RayCryptoSHA384

    \param ctx Uninitialized data of size sizeof(SharkSslSha384Ctx).
*/
SHARKSSL_API void  SharkSslSha384Ctx_constructor(SharkSslSha384Ctx* ctx);

/** append
    \ingroup RayCryptoSHA384
*/
SHARKSSL_API void  SharkSslSha384Ctx_append(SharkSslSha384Ctx*, const U8* data, U32 len);

/** finish
    \ingroup RayCryptoSHA384
*/
SHARKSSL_API void  SharkSslSha384Ctx_finish(SharkSslSha384Ctx*, U8 digest[SHARKSSL_SHA384_HASH_LEN]);

/** sha384
    \ingroup RayCryptoSHA384
*/
SHARKSSL_API int   sharkssl_sha384(const U8*, U16, U8*);
#endif

#if SHARKSSL_USE_SHA_512
/** Initialize
    \ingroup RayCryptoSHA512

    \param ctx Uninitialized data of size sizeof(SharkSslSha512Ctx).
*/
SHARKSSL_API void  SharkSslSha512Ctx_constructor(SharkSslSha512Ctx* ctx);

/** append
    \ingroup RayCryptoSHA512
*/
#define SharkSslSha512Ctx_append(ctx, d, l) \
   SharkSslSha384Ctx_append((SharkSslSha384Ctx*)ctx, d, l)

/** finish
    \ingroup RayCryptoSHA512
*/
SHARKSSL_API void  SharkSslSha512Ctx_finish(SharkSslSha512Ctx*, U8 digest[SHARKSSL_SHA512_HASH_LEN]);

/** sha512
    \ingroup RayCryptoSHA512
*/
SHARKSSL_API int   sharkssl_sha512(const U8*, U16, U8*);
#endif

#if SHARKSSL_USE_POLY1305
/** Initialize
    \ingroup RayCryptoPOLY1305

    \param ctx Uninitialized data of size sizeof(SharkSslPoly1305Ctx).
    \param key 16-byte key + 16-byte nonce
*/
SHARKSSL_API void  SharkSslPoly1305Ctx_constructor(SharkSslPoly1305Ctx *ctx, const U8 key[32]);

#define SharkSslPoly1305Ctx_destructor(o) memset(o, 0, sizeof(SharkSslPoly1305Ctx))

/** append
    \ingroup RayCryptoPOLY1305
*/
SHARKSSL_API void  SharkSslPoly1305Ctx_append(SharkSslPoly1305Ctx *ctx, const U8 *in, U32 len);

/** finish
    \ingroup RayCryptoPOLY1305
*/
SHARKSSL_API void  SharkSslPoly1305Ctx_finish(SharkSslPoly1305Ctx *ctx, U8 digest[SHARKSSL_POLY1305_HASH_LEN]);

/** poly1305
    \ingroup RayCryptoPOLY1305
*/
SHARKSSL_API int   sharkssl_poly1305(const U8 *data, U16 len, U8 *digest, const U8 key[32]);
#endif

#if SHARKSSL_USE_CHACHA20
/** Initialize
    \ingroup RayCryptoCHACHA20

    \param ctx Uninitialized data of size sizeof(SharkSslChaChaCtx)
    \param key 16- or 32-byte key
    \param keyLen the key length in bytes (either 16 or 32)
*/
SHARKSSL_API void SharkSslChaChaCtx_constructor(SharkSslChaChaCtx *ctx,
                                                const U8 *key, U8 keyLen);

#define SharkSslChaChaCtx_destructor(ctx) memset(ctx, 0, sizeof(SharkSslChaChaCtx))

/** Initialize
    \ingroup RayCryptoCHACHA20

    \param ctx Initialized data of size sizeof(SharkSslChaChaCtx)
    \param IV 8-byte initialization vector
*/
SHARKSSL_API void SharkSslChaChaCtx_setIV(SharkSslChaChaCtx *ctx, const U8 IV[8]);

/** Encrypt or decrypt
    \ingroup RayCryptoCHACHA20
*/
SHARKSSL_API void  SharkSslChaChaCtx_crypt(
   SharkSslChaChaCtx *ctx, const U8 *input, U8 *output, U32 len);
#endif

#if SHARKSSL_USE_ARC4
/** Initialize
    \ingroup RayCryptoARC4

*/
SHARKSSL_API void  SharkSslArc4Ctx_constructor(SharkSslArc4Ctx *ctx,
                                               const U8 *key, U8 keyLen);

#define SharkSslArc4Ctx_destructor(ctx) memset(ctx, 0, sizeof(SharkSslArc4Ctx))

/** Encrypt or decrypt
    \ingroup RayCryptoARC4
*/
SHARKSSL_API void  SharkSslArc4Ctx_crypt(
   SharkSslArc4Ctx *ctx, U8 *input, U8 *output, U16 len);
#endif

#if (SHARKSSL_USE_DES || SHARKSSL_USE_3DES)
/** Initialize
    \ingroup RayCryptoDES

    \param ctx Uninitialized data of size sizeof(SharkSslDesCtx)
    \param type encrypt/decrypt
    \param key the encryption/decryption key
    \param keyLen 'key' length
*/
SHARKSSL_API void  SharkSslDesCtx_constructor(SharkSslDesCtx *ctx,
                                              SharkSslDesCtx_Type type,
                                              const U8 *key, U8 keyLen);
#define SharkSslDesCtx_destructor(ctx)  memset(ctx, 0, sizeof(SharkSslDesCtx))

/** Encrypt
    \ingroup RayCryptoDES
*/
SHARKSSL_API void  SharkSslDesCtx_encrypt(
   SharkSslDesCtx *ctx, U8 input[8], U8 output[8]);
#if (SHARKSSL_USE_CAU || SHARKSSL_USE_MMCAU)

/** Decrypt
    \ingroup RayCryptoDES
*/
SHARKSSL_API void  SharkSslDesCtx_decrypt(
   SharkSslDesCtx *ctx, U8 input[8], U8 output[8]);
#else
#define SharkSslDesCtx_decrypt(ctx,i,o) SharkSslDesCtx_encrypt((ctx),(i),(o))
#endif

/** Encrypt
    \ingroup RayCryptoDES
*/
SHARKSSL_API void  SharkSslDesCtx_cbc_encrypt(SharkSslDesCtx *ctx, U8 vect[8],
                                              U8 *input, U8 *output, U16 len);

/** Decrypt
    \ingroup RayCryptoDES
*/
SHARKSSL_API void  SharkSslDesCtx_cbc_decrypt(SharkSslDesCtx *ctx, U8 vect[8],
                                              U8 *input, U8 *output, U16 len);
#endif

#if (SHARKSSL_USE_AES_256 || SHARKSSL_USE_AES_192 || SHARKSSL_USE_AES_128)
/** Initialize
    \ingroup RayCryptoAES

    \param ctx Uninitialized data of size sizeof(SharkSslAesCtx)
    \param type encrypt/decrypt
    \param key the encryption/decryption key
    \param keyLen 'key' length
*/
SHARKSSL_API void  SharkSslAesCtx_constructor(SharkSslAesCtx *ctx,
                                              SharkSslAesCtx_Type type,
                                              const U8 *key, U8 keyLen);
#define SharkSslAesCtx_destructor(ctx) memset(ctx, 0, sizeof(SharkSslAesCtx))

#if (!SHARKSSL_DISABLE_AES_ECB_DECRYPT)
/** Decrypt
    \ingroup RayCryptoAES
*/
SHARKSSL_API void  SharkSslAesCtx_decrypt(SharkSslAesCtx *ctx, U8 input[16], U8 output[16]);
#endif

/** Encrypt
    \ingroup RayCryptoAES
*/
SHARKSSL_API void  SharkSslAesCtx_encrypt(SharkSslAesCtx *ctx, U8 input[16], U8 output[16]);

#if (!SHARKSSL_DISABLE_AES_CBC_MODE)
/** Encrypt
    \ingroup RayCryptoAES
*/
SHARKSSL_API void  SharkSslAesCtx_cbc_encrypt(SharkSslAesCtx *ctx, U8 vect[16],
                                              U8 *input, U8 *output, U16 len);

/** Decrypt
    \ingroup RayCryptoAES
*/
SHARKSSL_API void  SharkSslAesCtx_cbc_decrypt(SharkSslAesCtx *ctx, U8 vect[16],
                                              U8 *input, U8 *output, U16 len);
#endif
#if SHARKSSL_ENABLE_AES_CTR_MODE

/**
    \ingroup RayCryptoAES
*/
SHARKSSL_API void  SharkSslAesCtx_ctr_mode(SharkSslAesCtx *ctx, U8 ctr[16],
                                           U8 *input, U8 *output, U16 len);
#endif
#if SHARKSSL_ENABLE_AES_GCM
/** Initialize
    \ingroup RayCryptoAesGcm

    \param ctx Uninitialized data of size sizeof(SharkSslAesGcmCtx).
    \param key the encryption/decryption key
    \param keyLen 'key' length
*/
SHARKSSL_API void  SharkSslAesGcmCtx_constructor(SharkSslAesGcmCtx *ctx,
                                                 const U8 *key, U8 keyLen);

#define SharkSslAesGcmCtx_destructor(ctx) \
   memset(ctx, 0, sizeof(SharkSslAesGcmCtx))

/** Encrypt
    \ingroup RayCryptoAesGcm
*/
SHARKSSL_API int   SharkSslAesGcmCtx_encrypt(SharkSslAesGcmCtx *ctx,
                                             const U8 vect[12], U8 tagout[16],
                                             const U8 *auth, U16 authlen,
                                             U8 *input, U8 *output, U16 len);


/** Decrypt
    \ingroup RayCryptoAesGcm
*/
SHARKSSL_API int   SharkSslAesGcmCtx_decrypt(SharkSslAesGcmCtx *ctx,
                                             const U8 vect[12], U8 tagin[16],
                                             const U8 *auth, U16 authlen,
                                             U8 *input, U8 *output, U16 len);
#endif
#if SHARKSSL_ENABLE_AES_CCM
/** Initialize
    \ingroup RayCryptoAesCcm

    \param ctx Uninitialized data of size sizeof(SharkSslAesCcmCtx).
*/
SHARKSSL_API void  SharkSslAesCcmCtx_constructor(SharkSslAesCcmCtx *ctx,
                                                 const U8 *key, U8 keyLen, U8 tagLen);

#define SharkSslAesCcmCtx_destructor(ctx) memset(ctx, 0, sizeof(SharkSslAesCcmCtx))

/** Encrypt
    \ingroup RayCryptoAesCcm
*/
SHARKSSL_API int   SharkSslAesCcmCtx_encrypt(SharkSslAesCcmCtx *ctx,
                                             const U8 vect[12], U8 *tagout,
                                             const U8 *auth, U16 authlen,
                                             U8 *input, U8 *output, U16 len);


/** Decrypt
    \ingroup RayCryptoAesCcm
*/
SHARKSSL_API int   SharkSslAesCcmCtx_decrypt(SharkSslAesCcmCtx *ctx,
                                             const U8 vect[12], U8 *tagin,
                                             const U8 *auth, U16 authlen,
                                             U8 *input, U8 *output, U16 len);
#endif
#endif

#ifdef __cplusplus
}
#endif


#endif /* _SharkSslCrypto_h */
