/* Starburst - display driver for 14 segment starburst display
 *
 * Copyright (c) 2011 Wim Huiskamp
 *
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * version 0.2 Initial Release
*/
#include "mbed.h"
#include "starburst.h"

/*****************************************************************************/
/*********************  DEFINITIONS FOR STARBURST  ***************************/
/*****************************************************************************/

// Definitions
const char digit_select[STAR_NR_DIGITS] = {0x01, 0x02, 0x04, 0x08};     //digit select
const char char_segm[STAR_NR_CHARS][2] = {
                                     {0x00, 0x00},  // space (32)
                                     {0x06, 0x00},  // !
                                     {0x22, 0x00},  // "
                                     {0xC7, 0x12},  // #

                                     {0xED, 0x12},  // $
                                     {0x24, 0x24},  // %
                                     {0x5D, 0x0B},  // &                                    
                                     {0x02, 0x00},  // '                                                                        

                                     {0x39, 0x00},  // (                                   
                                     {0x0F, 0x00},  // )                                     
                                     {0xC0, 0x3F},  // *                                                                         
                                     {0xC0, 0x12},  // +                                                                        

                                     {0x04, 0x00},  // ,
                                     {0xC0, 0x00},  // -
                                     {0x04, 0x00},  // .                                    
                                     {0x00, 0x24},  // /                                                                        

                                     {0x3F, 0x24},  // 0
                                     {0x06, 0x00},  // 1
                                     {0xDB, 0x00},  // 2                                    
                                     {0xCF, 0x00},  // 3                                                                      

                                     {0xE6, 0x00},  // 4
                                     {0xED, 0x00},  // 5
                                     {0xFD, 0x00},  // 6                                    
                                     {0x07, 0x00},  // 7                                                                       

                                     {0xFF, 0x00},  // 8
                                     {0xEF, 0x00},  // 9

                                     {0x00, 0x00},  // :
                                     {0x00, 0x00},  // ;                                     
                                     {0x00, 0x0C},  // <
                                     {0xC8, 0x00},  // =
                                     {0x00, 0x21},  // >
                                     {0x83, 0x10},  // ?                                     
                                     {0xDF, 0x00},  // @  (64)                                   
                                     
                                     {0xF7, 0x00},  // A  (65)
                                     {0xCF, 0x12},  // B
                                     {0x39, 0x00},  // C
                                     {0x0F, 0x12},  // D

                                     {0xF9, 0x00},  // E
                                     {0xF1, 0x00},  // F
                                     {0xBD, 0x00},  // G                                    
                                     {0xF6, 0x00},  // H                                                                        

                                     {0x09, 0x12},  // I
                                     {0x1E, 0x00},  // J
                                     {0x70, 0x0C},  // K                                    
                                     {0x38, 0x00},  // L                                                                        

                                     {0x36, 0x05},  // M
                                     {0x36, 0x09},  // N
                                     {0x3F, 0x00},  // O                                    
                                     {0xF3, 0x00},  // P                                                                        

                                     {0x3F, 0x08},  // Q
                                     {0xF3, 0x08},  // R
                                     {0xED, 0x00},  // S                                    
                                     {0x01, 0x12},  // T                                                                       

                                     {0x3E, 0x00},  // U
                                     {0x30, 0x24},  // V
                                     {0x36, 0x28},  // W                                    
                                     {0x00, 0x2D},  // X                                                                       

                                     {0x00, 0x15},  // Y
                                     {0x09, 0x24},  // Z  (90)

                                     {0x39, 0x00},  // [
                                     {0x00, 0x09},  // '\'                                     
                                     {0x0F, 0x00},  // ]
                                     {0x00, 0x28},  // ^
                                     {0x08, 0x00},  // _
                                     {0x00, 0x01},  // '                                     
                                                                          
                                     {0xF7, 0x00},  // A replacing a
                                     {0xCF, 0x12},  // B
                                     {0x39, 0x00},  // C
                                     {0x0F, 0x12},  // D

                                     {0xF9, 0x00},  // E
                                     {0xF1, 0x00},  // F
                                     {0xBD, 0x00},  // G                                    
                                     {0xF6, 0x00},  // H                                                                        

                                     {0x09, 0x12},  // I
                                     {0x1E, 0x00},  // J
                                     {0x70, 0x0C},  // K                                    
                                     {0x38, 0x00},  // L                                                                        

                                     {0x36, 0x05},  // M
                                     {0x36, 0x09},  // N
                                     {0x3F, 0x00},  // O                                    
                                     {0xF3, 0x00},  // P                                                                        

                                     {0x3F, 0x08},  // Q
                                     {0xF3, 0x08},  // R
                                     {0xED, 0x00},  // S                                    
                                     {0x01, 0x12},  // T                                                                       

                                     {0x3E, 0x00},  // U
                                     {0x30, 0x24},  // V
                                     {0x36, 0x28},  // W                                    
                                     {0x00, 0x2D},  // X                                                                       

                                     {0x00, 0x15},  // Y
                                     {0x09, 0x24},  // Z replacing z

                                     {0x39, 0x00},  // {
                                     {0x00, 0x12},  // |                                     
                                     {0x0F, 0x00},  // }
                                     {0xD2, 0x00},  // ~
                                     {0x00, 0x00}}; // DEL (127)                                  

/** Create a starburst object connected to the proper SPI pins
 *
 * @param  Pinname mosi, miso, sck, ce to connect to 
 * @brief 
*/
Starburst::Starburst(SPI &spi, PinName CE, PinName nOE) : _spi(spi), _ShiftLatch(CE), _nOE(nOE) {
  _init();
}

/** Display ASCII character at selected cursor location
 *
 * @param  int location cursor location, valid range is 0 - NR_DIGITS
 * @param  char byte to display, valid range is (char) 32 - (char) 127
 * @brief 
*/
void Starburst::putc(int location, char byte) {
  int charcode = (int) byte;
  
  // Check valid range
  if ((charcode < STAR_CHR_MIN) | (charcode > STAR_CHR_MAX)) {
    charcode = 0; 
  }
  else {
    charcode = charcode - STAR_CHR_MIN;
  };

  // Check valid range
  if ((location < 0) | (location > (STAR_NR_DIGITS-1))) {
    location = 0; 
  };

  // Lookup segmentcode for the character and store in digit's location while protecting the symbol bits       
  _digit_segments[location][1] = (_digit_segments[location][1] & SYM_MSK) | char_segm[charcode][1];     //segments H-N, LEDs
  _digit_segments[location][0] =                                            char_segm[charcode][0];     //segments A-G2
}      



/*---------------------------------------------------------------------------*\
 |
 |  Function:       putc
 |
 |  Description:    Displays specified character on the display at current
 |                  cursor position. Increments cursor.
 |                  Position on the display (0 to 3, Leftmost = 0)
 |
 |  Parameters:     disp_char - single character to display
 |                    - ASCII characters: 128 values between 0-127
 |                    - UDC character, 15 possible UDC values from 128-142 (not supported yet)
 |
 |  Returns:        Nothing
 |
\*---------------------------------------------------------------------------*/
void Starburst::putc(char disp_char) {

    // Write selected character to display at current position
    putc(_column, disp_char);  
    
    // Incr and wrap around cursorposition
    _column++; 
    _column = _column % STAR_NR_DIGITS;    
}

/** Display colon symbol
 *
 * @param  bool dp colon on or off
 * @brief 
*/
void Starburst::set_dp(bool dp) {
  
  if (dp) {
    _digit_segments[DP_IDX][1] = (_digit_segments[DP_IDX][1] & ~DP_MSK) | DP_ON;
  }
  else {
    _digit_segments[DP_IDX][1] = (_digit_segments[DP_IDX][1] & ~DP_MSK);  
  };
}      

/** Display LED symbol
 *
 * @param  bool led LED on or off
 * @brief 
*/
void Starburst::set_led(bool led) {
  
  if (led) {
    _digit_segments[D3_IDX][1] = (_digit_segments[D3_IDX][1] & ~D3_MSK) | D3_ON;
  }
  else {
    _digit_segments[D3_IDX][1] = (_digit_segments[D3_IDX][1] & ~D3_MSK);  
  };
}      


/** Display Bi-Color LED symbol
 *
 * @param  int Bicolor LED off, Red, Green or Yellow
 * @brief 
*/
void Starburst::set_bi_led(BiLED bi_led) {

  switch (bi_led) {
    case BI_OFF:   
            _digit_segments[D1_IDX][1] = (_digit_segments[D1_IDX][1] & ~D1_MSK);    
            break;  
    case BI_RED:
            _digit_segments[D1_IDX][1] = (_digit_segments[D1_IDX][1] & ~D1_MSK) | D1_RED;           
            break;      
    case BI_GRN:
            _digit_segments[D1_IDX][1] = (_digit_segments[D1_IDX][1] & ~D1_MSK) | D1_GRN;        
            break;         
    case BI_YEL:
            _digit_segments[D1_IDX][1] = (_digit_segments[D1_IDX][1] & ~D1_MSK) | D1_YEL;        
            break;
  };                               
}      

/*---------------------------------------------------------------------------*\
 |
 |  Function:       printf
 |
 |  Description:    Displays specified string on the display at current
 |                  cursor position. Increments cursor.
 |                  Max stringlen is 64 chars.
 |
 |  Parameters:     format - format string
 |                  args   - data
 |
 |  Returns:        Nothing
 |
\*---------------------------------------------------------------------------*/
void Starburst::printf (char * format, ...) {
   char display_string[64];
   int rv, string_len, i;   
   va_list args;
   va_start (args, format);

   rv=vsprintf (display_string, format, args);
   va_end (args);   
// printf("printing:'%s'\n", display_string);

   // loop round, writing characters
   string_len = strlen(display_string);   // obtain length of string
   for (i = 0; i < string_len; i++) {
     putc(display_string[i]);
   };

   // return rv;
}



/*---------------------------------------------------------------------------*\
 |
 |  Function:       _scroll_printf
 |
 |  Description:    Stores specified string in the scroll buffer.
 |                  The string will be automatically displayed in a scrolling fashion.
 |                  Max stringlen is 64 chars.
 |
 |  Parameters:     format - format string
 |                  args   - data
 |
 |  Returns:        Nothing
 |
\*---------------------------------------------------------------------------*/
void Starburst::scroll_printf (char * format, ...) {
   int rv;
   va_list args;
   va_start (args, format);

   rv=vsprintf (_scroll_display_string, format, args);
   va_end (args);   
// printf("printing:'%s'\n", _scroll_display_string);
   
//   _scroll_string_len = rv;   // obtain length of string
   _scroll_string_len = strlen(_scroll_display_string);   // obtain length of string

//   _scroll = (_scroll_string_len != 0);
   _scroll = false;   // Initially scroll is disabled
   _scroll_ptr = 0;  

   // return rv;
}


/*---------------------------------------------------------------------------*\
 |
 |  Function:       locate
 |
 |  Description:    Set the cursor address where the next character will be written. 
 |
 |  Parameters:     Cursor Column address
 |
 |  Returns:        Nothing
 |
\*---------------------------------------------------------------------------*/
void Starburst::locate(uint8_t column) {

//    _row = row % STAR_NR_ROWS;
    _column = column % STAR_NR_DIGITS;
}



/*---------------------------------------------------------------------------*\
 |
 |  Function:       cls
 |
 |  Description:    Clears the displayed data and flash RAM, but not the user
 |                  defined characters.
 |
 |  Parameters:     None
 |
 |  Returns:        Nothing
 |
\*---------------------------------------------------------------------------*/
void Starburst::cls(void) {
  int location;

  for (location=0; location<STAR_NR_DIGITS; location++)
    putc(location, ' ');

  // Reset cursor 
  locate(0);
}

/** Activate Blinking Display
 *
 * @param  bool enable Blink on or off
 * @brief 
*/
void Starburst::set_blink_mode(bool enable) {
  _blink = enable;
}  


/** Activate Scrolling Display
 *
 * @param  bool enable Scroll On or Off
 * @brief 
*/
void Starburst::set_scroll_mode(bool enable) {

  if (enable) {
    // Scroll only when a scroll string is available
    _scroll = (_scroll_string_len != 0);
  }
  else {
    _scroll = false;  
  }  

  // Init starting idx in scroll string
  _scroll_ptr = 0;    
}  



                                     
/** Multiplex Call to activate next digit, Called by timer interrupt at 75Hz per digit
 *
 * @param  
 * @brief 
*/
void Starburst::_mpx_digits() {
  static int digit_idx = 0;

  // Select the device by setting chip select low
  _ShiftLatch = 0;

  if (_blink) {
    // Blink activated 
    if (_blinkstate) {
      // Data visible, Send data
      _spi.write( digit_select[digit_idx]);        //digit select
      _spi.write(_digit_segments[digit_idx][1]);   //segments H-N, LEDs
      _spi.write(_digit_segments[digit_idx][0]);   //segments A-G2
    }
    else {
      // Data blanked, Send cleared digits
      _spi.write( digit_select[digit_idx]);        //digit select
      _spi.write(_digit_segments[digit_idx][1] & SYM_MSK);   //segments H-N, LEDs
      _spi.write(0x00);                                      //segments A-G2
    }
  }
  else {
    // Blink not activated   
    // Send data
    _spi.write( digit_select[digit_idx]);        //digit select
    _spi.write(_digit_segments[digit_idx][1]);   //segments H-N, LEDs
    _spi.write(_digit_segments[digit_idx][0]);   //segments A-G2        
  }    

  // latch the value in the device
  _ShiftLatch = 1;
  
  digit_idx++;
  if (digit_idx==STAR_NR_DIGITS) digit_idx=0;
}


/** Attach Multiplex Call to timer interrupt
 *
 * @param  
 * @brief 
*/
void Starburst::_mpx_start() {
  _mpx.attach(this, &Starburst::_mpx_digits, 0.013 / STAR_NR_DIGITS); // Update each digit at 75 Hz
}


/** Blinker Call to toggle _blinkstate, Called by timer interrupt at 2 Hz
 *
 * @param  
 * @brief 
*/
void Starburst::_blink_tick() {
  _blinkstate = !_blinkstate;
  
}


/** Attach Blink Call to timer interrupt
 *
 * @param  
 * @brief 
*/
void Starburst::_blink_start() {
  _blnk.attach(this, &Starburst::_blink_tick, 0.25); // Blink at 2 Hz
}


/** Scroll Call to scroll display text, Called by timer interrupt at 2 Hz
 *
 * @param  
 * @brief 
*/
void Starburst::_scroll_tick() {
  int tmp_ptr, idx; 

  // scroll text
  if (_scroll) {
     tmp_ptr = _scroll_ptr;
     
     // write next set of characters in display
     for (idx = 0; idx < STAR_NR_DIGITS; idx++) {     
       putc(idx, _scroll_display_string[tmp_ptr]);   // write next char in display
       
       tmp_ptr++;                                    // incr the ptr, wrap around if needed       
       tmp_ptr = tmp_ptr % _scroll_string_len;       
     };
       
//WRONG
//     _scroll_ptr = _scroll_ptr++ % _scroll_string_len; // incr start idx in scroll string for next time
     
//WRONG     
//     _scroll_ptr = (_scroll_ptr++) % _scroll_string_len; // incr start idx in scroll string for next time     

//OK
     _scroll_ptr++;
     _scroll_ptr = _scroll_ptr % _scroll_string_len; // incr start idx in scroll string for next time     


//OK     
//     _scroll_ptr = (_scroll_ptr + 1) % _scroll_string_len; // incr start idx in scroll string for next time          
  }
  
}


/** Attach Scroll Call to timer interrupt
 *
 * @param  
 * @brief 
*/
void Starburst::_scroll_start() {
//  _scrl.attach(this, &Starburst::_scroll_tick, 0.75); // Scroll at 1.5 Hz
  _scrl.attach(this, &Starburst::_scroll_tick, 0.5); // Scroll at 2 Hz  
}



void Starburst::_init() {

  // Disable Display
  _nOE = 1;
  _ShiftLatch = 1;

  // Setup the spi for 8 bit data, low steady state clock,
  // rising edge capture, with a 1MHz clock rate  
  _spi.format(8,0);
  _spi.frequency(500000);
//   _spi.frequency(1000000);

  // Select the device by setting chip select low
  _ShiftLatch = 0;

  // Send data to Clear all
  _spi.write(0x00);  //digit select
  _spi.write(0x00);  //segments H-N, LEDs
  _spi.write(0x00);  //segments A-G2
        
  // Latch the value in the device
  _ShiftLatch = 1;

  _blink=false; // Blink display
  _blinkstate=false;  
  
//  _flash=false; // Flash digit  
  
  //Enable display 
  _nOE = 0;


  // Clear display 
  cls();
  // Reset cursor 
  locate(0);

  //Enable blinking timer
  _blink_start();

  //Enable scroll timer
  _scroll_start();

  //Enable multiplexing timer
  _mpx_start();
}
   

/*****************************************************************************/
/******************************  END OF FILE  ********************************/
/*****************************************************************************/
