/* MBED_ControlBus - Use MBED Port Pins for controlling the external Bus
 * Copyright (c) 2011 Wim Huiskamp
 *
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * version 0.2 Initial Release
*/
#include "mbed.h"
#include "MBED_ControlBus.h"


/** Create an MBED_ControlBus object connected to the specified Pins
 *
 * @param PinName WR the Write pin 
 * @param PinName RD the Read pin 
 * @param PinName DTR the databuffer Transmit/Receive direction pin 
 * @param PinName CDBUF the databuffer enable pin
 * @param PinName CDINT the Keyboard interrupt pin  
*/
MBED_ControlBus::MBED_ControlBus(PinName WR, PinName RD, PinName DTR, PinName CDBUF, PinName CDINT) :
   _WR(WR),        // WR pin
   _RD(RD),        // RD pin
   _DTR(DTR),      // DTR pin
   _CDBUF(CDBUF),  // CDBUF pin   
   _CDINT(CDINT){  // CDINT pin      
   _init();
}


/** Set or Clear the WR pin on Control Bus
 *
 * @param Bit_Level wr_level
*/
void MBED_ControlBus::WR (Bit_Level wr_level) {

    if (wr_level == LOW) {
        // Make sure that databus buffer is enabled for Write
        //busctrl(ENABLE, WRITE);        
             
        _RD = 1;                    // RD Pin High, make sure there is no collision    
        _WR = 0;                    // WR Pin Low
      }               
    else {
        _WR = 1;                    // WR Pin High
    }      
}
 

/** Set or Clear the RD pin on Control Bus
 *
 * @param Bit_Level rd_level
*/
void MBED_ControlBus::RD (Bit_Level rd_level) {

    if (rd_level == LOW) {
        // Make sure that databus buffer is enabled for Read
        //busctrl(ENABLE, READ);

        _WR = 1;                    // WR Pin High, make sure there is no collision    
        _RD = 0;                    // RD Pin Low
      }               
    else {
        _RD = 1;                    // RD Pin High
    }      
}
               

/** Enable databus buffer for Write or Read
 *
 * @param Bus_Dir buf_dir
*/
void MBED_ControlBus::busdir (Bus_Dir buf_dir) {

    if (buf_dir == READ) {
        // Make sure that databus buffer is enabled for READ
        _DTR   = 0;                 // DTR Pin Low, Read Direction            
        _CDBUF = 0;                 // CDBUF Pin Low, Enable Buffer                
      }                
    else {
        // Make sure that databus buffer is enabled for Write   
        _DTR   = 1;                 // DTR Pin High, Write Direction            
        _CDBUF = 0;                 // CDBUF Pin Low, Enable Buffer
    }      
}


/** Enable/Disable databus buffer and control Write or Read Direction
 *
 * @param Bus_Ena buf_ena
 * @param Bus_Dir buf_dir
*/
void MBED_ControlBus::busctrl (Bus_Ena buf_ena, Bus_Dir buf_dir) {

    if (buf_ena == ENABLE) {
        _CDBUF = 0;                 // CDBUF Pin Low, Enable Buffer
      }                
    else {
        _CDBUF = 1;                 // CDBUF Pin High, Disable Buffer                
    }      

    if (buf_dir == READ) {
        _DTR   = 0;                 // DTR Pin Low, Read Direction            
      }                
    else {
        _DTR   = 1;                 // DTR Pin High, Write Direction            
    }      
}


/** Get the CDINT pin value from Control Bus
 *
 * @returns Bit_Level CDINT_level
*/
Bit_Level MBED_ControlBus::CDINT () {
    
    // CDINT Pin value, used as regular Pin rather than as interrupt
    if (_CDINT == 0)
      return LOW;
    else 
      return HIGH;      
}


/** Init MBED_ControlBus
 * @param
 * @returns 
 */
void MBED_ControlBus::_init() {
    // Make sure that all Control pins are disabled
    _RD    = 1;                        // RD Pin High    
    _WR    = 1;                        // WR Pin High        
    
    // Make sure that databus buffer is enabled for Write
    busctrl(ENABLE, WRITE);
    
}