/* mbed TM1651 Library, for TM1651 LED controller
 * Copyright (c) 2017, v01: WH, Initial version, Battery monitor
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, inclumosig without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUmosiG BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef TM1651_H
#define TM1651_H

// Select one of the testboards for TM1651 LED controller
#include "TM1651_Config.h"

/** An interface for driving TM1651 LED controller
 *
 * @code
 * #include "mbed.h"
 * #include "TM1651.h" 
 * 
 * Serial pc(USBTX, USBRX);
 *
 * //DisplayData_t size is 4 bytes (4 grids @ 7 segments)
 * TM1651::DisplayData_t all_str  = {0x7F, 0x7F, 0x7F, 0x7F};  
 *
 * // KeyData_t size is 1 bytes  
 * TM1651::KeyData_t keydata; 
 *
 * // TM1651 declaration, Select the desired type in TM1651_Config.h
 * TM1651_OPENSMART OPENSMART(p6, p7);       //LPC1768
 * //TM1651_OPENSMART OPENSMART(D9, D10);      //F401
 *
 * int main() {
 *   OPENSMART.cls(); 
 *   OPENSMART.writeData(all_str);
 *   wait(1);
 *   OPENSMART.setBrightness(TM1651_BRT0);
 *   wait(1);
 *   OPENSMART.setBrightness(TM1651_BRT3);
 *
 *   float delay=0.1;
 *   while (1) {
 *     OPENSMART.cls(); 
 *     wait(0.5); 
 *
 *     // Levels
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_0); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_1); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_2); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_3); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_4); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_5); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_6); wait(delay);
 *     // Levels off
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_5); wait(delay);
 *     OPENSMART.setLevel(TM1651_OPENSMART::LVL_4); wait(delay);
 *
 *     // Check and read keydata
 *     if (OPENSMART.getKeys(&keydata)) {
 *       pc.printf("Keydata = 0x%02x\r\n", keydata);
 *
 *       if (keydata == TM1651_SW9_BIT) { //sw9  
 *         OPENSMART.cls(); 
 *         OPENSMART.printf("--09"); 
 *       }  
 *     } // Check keydata
 *   } // while 
 * }
 * @endcode
 */


//TM1651 Display data
#define TM1651_MAX_NR_GRIDS    4
#define TM1651_BYTES_PER_GRID  1

//Significant bits Keymatrix data
//#define TM1638_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define TM1651_DISPLAY_MEM  (TM1651_MAX_NR_GRIDS * TM1651_BYTES_PER_GRID)
#define TM1651_KEY_MEM         1

//Reserved bits for commands
#define TM1651_CMD_MSK      0xC0

//Data setting commands
#define TM1651_DATA_SET_CMD 0x40
#define TM1651_DATA_WR      0x00
#define TM1651_KEY_RD       0x02
#define TM1651_ADDR_INC     0x00
#define TM1651_ADDR_FIXED   0x04
#define TM1651_MODE_NORM    0x00
#define TM1651_MODE_TEST    0x08

//Address setting commands
#define TM1651_ADDR_SET_CMD 0xC0
#define TM1651_ADDR_MSK     0x03 //0..3

//Display control commands
#define TM1651_DSP_CTRL_CMD 0x80
#define TM1651_BRT_MSK      0x07
#define TM1651_BRT0         0x00 //Pulsewidth 1/16
#define TM1651_BRT1         0x01
#define TM1651_BRT2         0x02
#define TM1651_BRT3         0x03
#define TM1651_BRT4         0x04
#define TM1651_BRT5         0x05
#define TM1651_BRT6         0x06
#define TM1651_BRT7         0x07 //Pulsewidth 14/16

#define TM1651_BRT_DEF      TM1651_BRT3

#define TM1651_DSP_OFF      0x00
#define TM1651_DSP_ON       0x08


//Access to 8 Switches
//S0 S1 S2 K1 K2 1 1 1
//K1 = 0 1
//K2 =   1
#define TM1651_SW1_BIT      0xEF
#define TM1651_SW2_BIT      0x6F
#define TM1651_SW3_BIT      0xAF
#define TM1651_SW4_BIT      0x2F
#define TM1651_SW5_BIT      0xCF
#define TM1651_SW6_BIT      0x4F
#define TM1651_SW7_BIT      0x8F

#define TM1651_SW_NONE      0xFF


//Segments
#define S7_S1            (1 << 0)
#define S7_S2            (1 << 1)
#define S7_S3            (1 << 2)
#define S7_S4            (1 << 3)
#define S7_S5            (1 << 4)
#define S7_S6            (1 << 5)
#define S7_S7            (1 << 6)


/** A class for driving TM1651 LED controller
 *
 * @brief Supports 4 Grids @ 7 Segments and 7 Keys. 
 *        Serial bus interface device. 
 */
class TM1651 {
 public:

  /** Datatype for displaydata */
  typedef char DisplayData_t[TM1651_DISPLAY_MEM];

  /** Datatypes for keymatrix data */
  typedef char KeyData_t;


 /** Constructor for class for driving TM1651 LED controller
  *
  * @brief Supports 4 Grids @ 7 segments and 7 Keys. 
  *        Serial bus interface device. 
  *
  *  @param  PinName dio Serial bus DIO pin
  *  @param  PinName sck Serial bus CLK pin 
  */
  TM1651(PinName dio, PinName clk);


  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to TM1651
   *  @param  char data byte written at given address
   *  @param  int address display memory location to write byte
   *  @return none
   */ 
   void writeData(char data, int address); 

   /** Write Display datablock to TM1651
    *  @param  DisplayData_t data Array of TM1651_DISPLAY_MEM (=4) bytes for displaydata
    *  @param  length number bytes to write (valid range 0..(TM1651_MAX_NR_GRIDS * TM1651_BYTES_PER_GRID) (=4), when starting at address 0)  
    *  @param  int address display memory location to write bytes (default = 0) 
    *  @return none
    */ 
    void writeData(DisplayData_t data, int length = (TM1651_MAX_NR_GRIDS * TM1651_BYTES_PER_GRID), int address = 0);

  /** Read keydata block from TM1651
   *  @param  *keydata Ptr to bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   */   
  bool getKeys(KeyData_t *keydata);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = TM1651_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
    DigitalInOut _dio;
    DigitalOut   _clk;  

    char _display;
    char _bright; 
  
  /** Init the Serial interface and the controller
    * @param  none
    * @return none
    */ 
    void _init();


  /** Generate Start condition for TM1651
    *  @param  none
    *  @return none
    */ 
    void _start();
  
  /** Generate Stop condition for TM1651
    *  @param  none
    *  @return none
    */ 
    void _stop();

  /** Send byte to TM1651
    *  @param  int data
    *  @return none
    */ 
    void _write(int data);

  /** Read byte from TM1651
    *  @return read byte 
    */ 
    char _read();

  /** Write command and parameter to TM1651
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
    void _writeCmd(int cmd, int data);  
};

#if (OPENSMART_TEST == 1) 
// Derived class for TM1651 used in OPEN-SMART battery display unit with 10 Segments
//

#define OPENSMART_NR_GRIDS  4
#define OPENSMART_NR_DIGITS 4
#define OPENSMART_NR_UDC    0

//Battery level Icons mapping onto Segments
#define R12                S7_S1
#define Y3                 S7_S2
#define Y4                 S7_S3
#define Y5                 S7_S4
#define G67                S7_S5
#define G89                S7_S6
#define B10                S7_S7


/** Constructor for class for driving TM1651 controller as used in OPENSMART
  *
  *  @brief Supports battery display unit with 10 Segments.
  *  
  *  @param  PinName dio Serial bus DIO pin  
  *  @param  PinName clk Serial bus CLK pin 
  */
class TM1651_OPENSMART : public TM1651 {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Pattern encoded in 16 LSBs
  enum Icon {
    LD12  = ( 1<<24) | R12,   /**<  LED1 (Red) & LED2 (Red)*/
    LD3   = ( 1<<24) | Y3,    /**<  LED3 (Yellow)*/
    LD4   = ( 1<<24) | Y4,    /**<  LED4 (Yellow)*/
    LD5   = ( 1<<24) | Y5,    /**<  LED5 (Yellow)*/
    LD67  = ( 1<<24) | G67,   /**<  LED6 (Green) & LED7 (Green)*/
    LD89  = ( 1<<24) | G89,   /**<  LED8 (Green) & LED9 (Green)*/
    LD10  = ( 1<<24) | B10,   /**<  LED10 (Blue) */
  };

  /** Enums for Batterylevels */
  enum Level {
    LVL_0,   /**<  Level 0 = R12       */
    LVL_1,   /**<  Level 1 = R12 - Y3  */
    LVL_2,   /**<  Level 2 = R12 - Y4  */
    LVL_3,   /**<  Level 3 = R12 - Y5  */
    LVL_4,   /**<  Level 4 = R12 - G67 */
    LVL_5,   /**<  Level 5 = R12 - G89 */    
    LVL_6    /**<  Level 6 = R12 - B10 */    

#if(0)    
    LVL_0   = ( 1<<24) | R12,                                   /**<  Level 0 */
    LVL_1   = ( 1<<24) | R12 | Y3,                              /**<  Level 1 */
    LVL_2   = ( 1<<24) | R12 | Y3 | Y4,                         /**<  Level 2 */
    LVL_3   = ( 1<<24) | R12 | Y3 | Y4 | Y5,                    /**<  Level 3 */
    LVL_4   = ( 1<<24) | R12 | Y3 | Y4 | Y5 | G67,              /**<  Level 4 */
    LVL_5   = ( 1<<24) | R12 | Y3 | Y4 | Y5 | G67 | G89,        /**<  Level 5 */
    LVL_6   = ( 1<<24) | R12 | Y3 | Y4 | Y5 | G67 | G89 | B10,  /**<  Level 6 */   
#endif   
  };
  
  
 /** Constructor for class for driving TM1651 LED controller
  *
  * @brief Supports battery display unit with 10 Segments.
  *        Serial bus interface device. 
  *
  *  @param  PinName dio Serial bus DIO pin
  *  @param  PinName sck Serial bus CLK pin 
  */
  TM1651_OPENSMART(PinName dio, PinName clk);
   
    /** Clear the screen and locate to 0
     * @param none
     * @return none         
     */
    void cls();

  /** Set Icon
    *
    * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Pattern encoded in 16 LSBs
    * @return none
    */
    void setIcon(Icon icon);

   /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set Level
     *
     * @param Level level Enums Level indicates the Battery level to be displayed
     * @return none
     */
    void setLevel(Level level);


   /** Write databyte to TM1651
     *  @param  char data byte written at given address
     *  @param  int address display memory location to write byte
     *  @return none
     */ 
    void writeData(char data, int address){
      TM1651::writeData(data, address);
    }        

   /** Write Display datablock to TM1651
     *  @param  DisplayData_t data Array of TM1651_DISPLAY_MEM (=4) bytes for displaydata
     *  @param  length number bytes to write (valid range 0..(OPENSMART_NR_GRIDS * TM1651_BYTES_PER_GRID) (=4), when starting at address 0)  
     *  @param  int address display memory location to write bytes (default = 0)
     *  @return none
     */   
    void writeData(DisplayData_t data, int length = (OPENSMART_NR_GRIDS * TM1651_BYTES_PER_GRID), int address = 0) {
      TM1651::writeData(data, length, address);
    }  

protected:  

private:
    DisplayData_t _displaybuffer;

};
#endif

#endif