/* mbed TM1650 Library, for TM1650 LED controller
 * Copyright (c) 2017, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, inclumosig without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUmosiG BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "TM1650.h"

/** Constructor for class for driving TM1650 LED controller with Serial bus interface device. 
 *  @brief Supports 4 digits @ 8 segments. Also supports upto 28 Keys.
 *   
 *  @param  PinName dio Serial bus DIO pin 
 *  @param  PinName clk Serial bus CLK pin
*/
TM1650::TM1650(PinName dio, PinName clk) : _dio(dio), _clk(clk) {

  _init();
}


/** Init the Serial interface and the controller
  * @param  none
  * @return none
  */ 
void TM1650::_init(){
  
//TM1650 uses a Serial bus that looks like I2C, but really is not.
//It has Start and Stop conditions like I2C and an Ack pulse, but instead of slaveaddresses and a RW bit it just transmits commands and data.

//init Serial bus
  _dio.output();
//  _dio.mode(PullUp);
  wait_us(1);

  _dio=1;  
  _clk=1;  

//init controller  
  _bright  = TM1650_BRT_DEF; 
  _segment = TM1650_DSP_8S; 
  _display = TM1650_DSP_ON;  
  _writeCmd(TM1650_DSP_CTRL_CMD, _bright | _segment | _display );  // Display control cmd, brightness, 7/8 segments, display on/off
}   


/** Clear the screen and locate to 0
 */  
void TM1650::cls() {

  for (int cnt=0; cnt<TM1650_DISPLAY_MEM; cnt++) {
    _start();     
    _write(TM1650_DATA_WR_CMD | (cnt << 1)); // Set Address cmd (no auto incr supported)
    _write(0x00);                            // Write data 
    _stop();      
  }

}  

/** Set Brightness
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void TM1650::setBrightness(char brightness){

  _bright = brightness & TM1650_BRT_MSK; // mask invalid bits
  
  _writeCmd(TM1650_DSP_CTRL_CMD, _bright | _segment | _display );  // Display control cmd, brightness, 7/8 segments, display on/off
}

/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void TM1650::setDisplay(bool on) {
  
  if (on) {
    _display = TM1650_DSP_ON;
  }
  else {
    _display = TM1650_DSP_OFF;
  }
  
  _writeCmd(TM1650_DSP_CTRL_CMD, _bright | _segment | _display );  // Display control cmd, brightness, 7/8 segments, display on/off
}

/** Write databyte to TM1650
  *  @param  int address display memory location to write byte
  *  @param  char data byte written at given address
  *  @return none
  */ 
void TM1650::writeData(char data, int address) {
  
  _start();

  _write(TM1650_DATA_WR_CMD | ((address & TM1650_ADDR_MSK) << 1)); // Set Address cmd     
  _write(data);                                                    // Write data 

  _stop();  
}

/** Write Display datablock to TM1650
  *  @param  DisplayData_t data Array of TM1650_DISPLAY_MEM (=4) bytes for displaydata
  *  @param  length number bytes to write (valid range 0..(TM1650_MAX_NR_GRIDS * TM1650_BYTES_PER_GRID) (=4), when starting at address 0)  
  *  @param  int address display memory location to write bytes (default = 0) 
  *  @return none
  */  
void TM1650::writeData(DisplayData_t data, int length, int address) {

// sanity check
  address &= TM1650_ADDR_MSK;
  if (length < 0) {length = 0;}
  if ((length + address) > TM1650_DISPLAY_MEM) {length = (TM1650_DISPLAY_MEM - address);}
    
  for (int idx=0; idx<length; idx++) {    
    _start();
    _write(TM1650_DATA_WR_CMD | ((address + idx) << 1)); // Set Address cmd (no auto incr supported)
    _write(data[address + idx]);                         // Write data 
    _stop();      
  }
}

/** Read keydata block from TM1650
  *  @param  *keydata Ptr to bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  */   
bool TM1650::getKeys(KeyData_t *keydata) {

  _start();

  // Enable Key Read mode
  _write(TM1650_KEY_RD_CMD); // Read Key cmd

  // Read key
  *keydata = _read();
//  printf("Key = 0x%02x\r\n", *keydata);  // debug

  _stop();  
      
  // Check if valid key bits are set
  return ( (*keydata & TM1650_SW_MSK) == TM1650_SW_MSK );
}
  

/** Generate Start condition for TM1650
  *  @param  none
  *  @return none
  */ 
void TM1650::_start() {

  _dio=0;
  wait_us(1);
  _clk=0;
  wait_us(1);
}
  
/** Generate Stop condition for TM1650
  *  @param  none
  *  @return none
  */ 
void TM1650::_stop() {

  _dio=0;
  wait_us(1);  
  _clk=1;
  wait_us(1);
  _dio=1;
  wait_us(1);
}

/** Send byte to TM1650
  *  @param  int data
  *  @return none
  */ 
void TM1650::_write(int data) {
 
  for (int bit=7; bit >= 0; bit--) {    
    //The TM1650 expects MSB first
    if (((data >> bit) & 0x01) == 0x01) {
      _dio=1;      
    }
    else {    
      _dio=0;      
    }  
    wait_us(1);
    _clk=1;
    wait_us(1);
    _clk=0;  
    wait_us(1);
  }  

  _dio=1;
  
  // Prepare DIO to read data
  _dio.input();
  wait_us(3);
      
  // dummy Ack
  _clk=1;
  wait_us(1);
//  _ack = _dio;  
  _clk=0;  
  wait_us(1); 
  
  // Return DIO to output mode
  _dio.output();  
  wait_us(3);  

  _dio=1; //idle  
}

/** Read byte from TM1650
  *  @param  none
  *  @return read byte 
  */ 
char TM1650::_read() {
  char keycode = 0;

  // Prepare DIO to read data
  _dio.input();
  wait_us(3);
    
  for (int bit=0; bit<8; bit++) {    
   
    //The TM1650 sends bitpattern: D7..D0
    //Data is shifted out by the TM1650 on the falling edge of CLK
    //Observe sufficient delay to allow the Open Drain DIO to rise to H levels
    // Prepare to read next bit, MSB (ie D7) first. 
    keycode = keycode << 1;  

    _clk=1;
    wait_us(1);
    
    // Read next bit
    if (_dio) { keycode |= 0x01; }        

    _clk=0;        
    wait_us(5); // Delay to allow for slow risetime
  }  
  
  // Return DIO to output mode
  _dio.output();
  wait_us(3);  

  // dummy Ack
  _dio=0; //Ack   
  wait_us(1);
  
  _clk=1;
  wait_us(1);
  _clk=0;  
  wait_us(1); 

  _dio=1; //idle

  return keycode;
}

/** Write command (and parameters) to TM1650
  *  @param  int cmd Command byte
  *  @Param  int data Parameters for command
  *  @return none
  */  
void TM1650::_writeCmd(int cmd, int data){
    
  _start();

  _write(cmd);   
  _write(data);  
   
  _stop();          
}  


#if (MEIBAI_TEST == 1) 
// Derived class for TM1650 used in MEIBAI display unit
//

/** Constructor for class for driving TM1650 LED controller
 *
 * @brief Supports 4 Digits of 7 Segments + DP.
 *        Also supports 3 keys.
 *        Serial bus interface device. 
 *
 *  @param  PinName dio Serial bus DIO pin
 *  @param  PinName sck Serial bus CLK pin 
 */
TM1650_MEIBAI::TM1650_MEIBAI(PinName dio, PinName clk) : TM1650(dio, clk) {
  _column  = 0;
  _columns = MEIBAI_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     * @return c
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  * @return none  
  */
void TM1650_MEIBAI::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int TM1650_MEIBAI::columns() {
    return _columns;
}

    
/** Clear the screen and locate to 0
  * 
  * @param bool clrAll Clear Icons also (default = false)
  * @return none  
  */ 
void TM1650_MEIBAI::cls(bool clrAll) {  

  if (clrAll) {
    //clear local buffer (inclumosig Icons)
    for (int idx=0; idx < MEIBAI_NR_GRIDS; idx++) {
      _displaybuffer[idx] = 0x00;  
    }
  }  
  else {
    //clear local buffer (preserving Icons)
    for (int idx=0; idx < MEIBAI_NR_GRIDS; idx++) {
      _displaybuffer[idx] = _displaybuffer[idx] & MASK_ICON_GRID[idx];  
    }  
  }

  writeData(_displaybuffer, (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID), 0);

  _column = 0;   
}     

/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void TM1650_MEIBAI::setIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1);
    
  //Save char...and set bits for icon to write
  _displaybuffer[addr] = _displaybuffer[addr] | LO(icn);      
//  writeData(_displaybuffer, (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID), 0);
  writeData(_displaybuffer, TM1650_BYTES_PER_GRID, addr);  
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void TM1650_MEIBAI::clrIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1);
    
  //Save char...and clr bits for icon to write
  _displaybuffer[addr] = _displaybuffer[addr] & ~LO(icn);      
//  writeData(_displaybuffer, (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID), 0);
  writeData(_displaybuffer, TM1650_BYTES_PER_GRID, addr);    
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (8 bits)
  * @return none  
  */
void TM1650_MEIBAI::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (MEIBAI_NR_UDC-1)) {
    return;
  }
  // Mask out Icon bits?

  _UDC_7S[udc_idx] = LO(udc_data);
}


/** Write a single character (Stream implementation)
  *
  * @param c The character to write to the display
  * @return c
  */
int TM1650_MEIBAI::_putc(int value) {
//The MEIBAI mapping between Digit positions (Left to Right) and Grids is:
//  GR1 GR2 GR3 GR4
//The memory addresses or column numbers are:
//   0   1   2   3
    
    int addr;
    bool validChar = false;
    char pattern   = 0x00;
    
    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    }
    else if ((value == '.') || (value == ',')) {
      //No character to write
      validChar = false;
      pattern = S7_DP; // placeholder for all DPs
      
      // Check to see that DP can be shown for current column
      if (_column > 0) {
        //Translate between _column and displaybuffer entries
        //Add DP to bitpattern of digit left of current column.
        addr = (_column - 1);
      
        //Save icons...and set bits for decimal point to write
        _displaybuffer[addr] = _displaybuffer[addr] | pattern;
//        writeData(_displaybuffer, (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID));
        writeData(_displaybuffer, TM1650_BYTES_PER_GRID, addr); 
        
        //No Cursor Update
      }
    }
    else if ((value >= 0) && (value < MEIBAI_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_7S[value];
    }  
    
#if (SHOW_ASCII == 1)
    //display all ASCII characters
    else if ((value >= FONT_7S_START) && (value <= FONT_7S_END)) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - FONT_7S_START];
    } // else
#else    
    //display only digits and hex characters      
    else if (value == '-') {
      //Character to write
      validChar = true;
      pattern = C7_MIN;         
    }
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'a'];
    } //else
#endif

    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      addr = _column;

      //Save icons...and set bits for character to write
      _displaybuffer[addr] = (_displaybuffer[addr] & MASK_ICON_GRID[_column]) | pattern;

//      writeData(_displaybuffer, (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID));
      writeData(_displaybuffer, TM1650_BYTES_PER_GRID, addr);        
                                
      //Update Cursor
      _column++;
      if (_column > (MEIBAI_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}


/** Get a single character (Stream implementation)
  *
  * @param none
  * @return -1
  */
int TM1650_MEIBAI::_getc() {
    return -1;
}

#endif
