/* mbed TM1650 Library, for TM1650 LED controller
 * Copyright (c) 2017, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, inclumosig without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUmosiG BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef TM1650_H
#define TM1650_H

// Select one of the testboards for TM1650 LED controller
#include "TM1650_Config.h"

/** An interface for driving TM1650 LED controller
 *
 * @code
 * #include "mbed.h"
 * #include "TM1650.h" 
 * 
 * Serial pc(USBTX, USBRX);
 *
 * //DisplayData_t size is 4 bytes (4 grids @ 8 segments)
 * TM1650::DisplayData_t all_str  = {0xFF, 0xFF, 0xFF, 0xFF};  
 *
 * // KeyData_t size is 1 bytes  
 * TM1650::KeyData_t keydata; 
 *
 * // TM1650 declaration, Select the desired type in TM1650_Config.h
 * TM1650_MEIBAI MEIBAI(p6, p7);       //LPC1768
 * //TM1650_MEIBAI MEIBAI(D9, D10);      //F401
 *
 * int main() {
 *   MEIBAI.cls(); 
 *   MEIBAI.writeData(all_str);
 *   wait(1);
 *   MEIBAI.setBrightness(TM1650_BRT0);
 *   wait(1);
 *   MEIBAI.setBrightness(TM1650_BRT3);
 *
 *   while (1) {
 *     MEIBAI.cls(); 
 *     wait(0.5); 
 *     MEIBAI.writeData(all_str);
 *     wait(1.0);
 *     MEIBAI.cls(); 
 *     MEIBAI.printf(" HI ");  
 *     wait(1.0); 
 *
 *     // Check and read keydata
 *     if (MEIBAI.getKeys(&keydata)) {
 *       pc.printf("Keydata = 0x%02x\r\n", keydata);
 *
 *       if (keydata == TM1650_SW1_BIT) { //sw1  
 *         MEIBAI.cls(); 
 *         MEIBAI.printf("--01"); 
 *       }  
 *     } // Check keydata
 *   } // while 
 * }
 * @endcode
 */


//TM1650 Display data
#define TM1650_MAX_NR_GRIDS    4
#define TM1650_BYTES_PER_GRID  1

//Significant bits Keymatrix data
//#define TM1638_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define TM1650_DISPLAY_MEM  (TM1650_MAX_NR_GRIDS * TM1650_BYTES_PER_GRID)
#define TM1650_KEY_MEM         1

//Data write command
//Valid data addresses are 0x68, 0x6A, 0x6C, 0x6E
#define TM1650_DATA_WR_CMD  0x68
#define TM1650_ADDR_MSK     0x03 //0..3

//Display control command
#define TM1650_DSP_CTRL_CMD 0x48

#define TM1650_BRT_MSK      0x70
//#define TM1650_BRT_SHFT        4
#define TM1650_BRT0         0x10 //Pulsewidth 1/16
#define TM1650_BRT1         0x20
#define TM1650_BRT2         0x30
#define TM1650_BRT3         0x40
#define TM1650_BRT4         0x50
#define TM1650_BRT5         0x60
#define TM1650_BRT6         0x70
#define TM1650_BRT7         0x00 //Pulsewidth 15/16

#define TM1650_BRT_DEF      TM1650_BRT3

#define TM1650_DSP_8S       0x00
#define TM1650_DSP_7S       0x08

#define TM1650_DSP_OFF      0x00
#define TM1650_DSP_ON       0x01


//Keydata read command
#define TM1650_KEY_RD_CMD   0x49

//Access to 28 Switches
//d7 d6 d5 d4 d3 d2 d1 d0
// .  1  .  .  .  1  .  .   = Valid Key
#define TM1650_SW_MSK       0x44

//A -> Dig1 .. Dig4
#define TM1650_SW1_BIT      0x44
#define TM1650_SW2_BIT      0x45
#define TM1650_SW3_BIT      0x46
#define TM1650_SW4_BIT      0x47
//B -> Dig1 .. Dig4
#define TM1650_SW5_BIT      0x4C
#define TM1650_SW6_BIT      0x4D
#define TM1650_SW7_BIT      0x4E
#define TM1650_SW8_BIT      0x4F
//C -> Dig1 .. Dig4
#define TM1650_SW9_BIT      0x54
#define TM1650_SW10_BIT     0x55
#define TM1650_SW11_BIT     0x56
#define TM1650_SW12_BIT     0x57
//D -> Dig1 .. Dig4
#define TM1650_SW13_BIT     0x5C
#define TM1650_SW14_BIT     0x5D
#define TM1650_SW15_BIT     0x5E
#define TM1650_SW16_BIT     0x5F
//E -> Dig1 .. Dig4
#define TM1650_SW17_BIT     0x64
#define TM1650_SW18_BIT     0x65
#define TM1650_SW19_BIT     0x66
#define TM1650_SW20_BIT     0x67
//F -> Dig1 .. Dig4
#define TM1650_SW21_BIT     0x6C
#define TM1650_SW22_BIT     0x6D
#define TM1650_SW23_BIT     0x6E
#define TM1650_SW24_BIT     0x6F
//G -> Dig1 .. Dig4
#define TM1650_SW25_BIT     0x74
#define TM1650_SW26_BIT     0x75
#define TM1650_SW27_BIT     0x76
#define TM1650_SW28_BIT     0x77



/** A class for driving TM1650 LED controller
 *
 * @brief Supports 4 Grids @ 8 Segments and upto 28 Keys. 
 *        Serial bus interface device. 
 */
class TM1650 {
 public:

  /** Datatype for displaydata */
  typedef char DisplayData_t[TM1650_DISPLAY_MEM];

  /** Datatypes for keymatrix data */
  typedef char KeyData_t;

 /** Constructor for class for driving TM1650 LED controller
  *
  * @brief Supports 4 Grids @ 8 segments and 28 Keys. 
  *        Serial bus interface device. 
  *
  *  @param  PinName dio Serial bus DIO pin
  *  @param  PinName sck Serial bus CLK pin 
  */
  TM1650(PinName dio, PinName clk);


  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to TM1650
   *  @param  char data byte written at given address
   *  @param  int address display memory location to write byte
   *  @return none
   */ 
   void writeData(char data, int address); 

   /** Write Display datablock to TM1650
    *  @param  DisplayData_t data Array of TM1650_DISPLAY_MEM (=4) bytes for displaydata
    *  @param  length number bytes to write (valid range 0..(TM1650_MAX_NR_GRIDS * TM1650_BYTES_PER_GRID) (=4), when starting at address 0)  
    *  @param  int address display memory location to write bytes (default = 0) 
    *  @return none
    */ 
    void writeData(DisplayData_t data, int length = (TM1650_MAX_NR_GRIDS * TM1650_BYTES_PER_GRID), int address = 0);

  /** Read keydata block from TM1650
   *
   *  @param  *keydata Ptr to bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   */   
  bool getKeys(KeyData_t *keydata);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = TM1650_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    * @return none
    */
  void setDisplay(bool on);
  
 private:  
    DigitalInOut _dio;
    DigitalOut   _clk;  

    char _bright;
    char _segment;
    char _display;    
  
  /** Init the Serial interface and the controller
    * @param  none
    * @return none
    */ 
    void _init();

  /** Generate Start condition for TM1650
    *  @param  none
    *  @return none
    */ 
    void _start();
  
  /** Generate Stop condition for TM1650
    *  @param  none
    *  @return none
    */ 
    void _stop();

  /** Send byte to TM1650
    *  @param  int data
    *  @return none
    */ 
    void _write(int data);

  /** Read byte from TM1650
    *  @param  none  
    *  @return read byte 
    */ 
    char _read();

  /** Write command and parameter to TM1650
    *  @param  int cmd Command byte
    *  @Param  int data Parameters for command
    *  @return none
    */ 
    void _writeCmd(int cmd, int data);  
};

#if (MEIBAI_TEST == 1) 
// Derived class for TM1650 used in MEIBAI display unit with 4 Digits and 3 keys
//

#include "Font_7Seg.h"

#define MEIBAI_NR_GRIDS  4
#define MEIBAI_NR_DIGITS 4
#define MEIBAI_NR_UDC    8


/** Constructor for class for driving TM1650 controller as used in MEIBAI
  *
  *  @brief Supports 4 Digits of 7 Segments + DP. 
  *         Also Supports 3 Keys. Serial bus interface device. 
  *  
  *  @param  PinName dio Serial bus DIO pin
  *  @param  PinName sck Serial bus CLK pin  
  */
class TM1650_MEIBAI : public TM1650, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  enum Icon {
    DP1   = ( 1<<24) | S7_DP1,  /**<  Digit 1 */
    DP2   = ( 2<<24) | S7_DP2,  /**<  Digit 2 */
    DP3   = ( 3<<24) | S7_DP3,  /**<  Digit 3 */
    DP4   = ( 4<<24) | S7_DP4,  /**<  Digit 4 */
  };
  
  typedef char UDCData_t[MEIBAI_NR_UDC];


 /** Constructor for class for driving TM1650 LED controller
  *
  * @brief Supports 4 Digits of 7 Segments + DP.
  *        Also Supports 3 Keys. Serial bus interface device. 
  *
  *  @param  PinName dio Serial bus DIO pin
  *  @param  PinName sck Serial bus CLK pin
  */
  TM1650_MEIBAI(PinName dio, PinName clk);

  
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     * @return c
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     * @return chars written
     */
    int printf(const char* format, ...);   
#endif

    /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @return none     
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     * @return none     
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)
     * @return none            
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to TM1650
     *  @param  char data byte written at given address
     *  @param  int address display memory location to write byte
     *  @return none
     */ 
    void writeData(char data, int address){
      TM1650::writeData(data, address);
    }        

   /** Write Display datablock to TM1650
    *  @param  DisplayData_t data Array of TM1650_DISPLAY_MEM (=4) bytes for displaydata
    *  @param  length number bytes to write (valid range 0..(MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID) (=4), when starting at address 0)  
    *  @param  int address display memory location to write bytes (default = 0)
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (MEIBAI_NR_GRIDS * TM1650_BYTES_PER_GRID), int address = 0) {
      TM1650::writeData(data, length, address);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;
    int _columns;   
    
    DisplayData_t _displaybuffer;
    UDCData_t _UDC_7S; 
};
#endif

#endif