/* mbed TM1638 Library, for TM1638 LED controller
 * Copyright (c) 2015, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef TM1638_H
#define TM1638_H

// Select one of the testboards for TM1638 LED controller
#include "TM1638_Config.h"

/** An interface for driving TM1638 LED controller
 *
 * @code
 * #include "mbed.h"
 * #include "TM1638.h" 
 * 
 * DisplayData_t size is 16 bytes (8 grids @ 10 segments)
 * TM1638::DisplayData_t all_str  = {0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F, 0xFF,0x3F};  
 *
 * // KeyData_t size is 4 bytes  
 * TM1638::KeyData_t keydata; 
 *
 * // TM1638 declaration
 * TM1638 TM1638(p5,p6,p7, p8);
 *
 * int main() {
 *   TM1638.cls(); 
 *   TM1638.writeData(all_str);
 *   wait(1);
 *   TM1638.setBrightness(TM1638_BRT0);
 *   wait(1);
 *   TM1638.setBrightness(TM1638_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (TM1638.getKeys(&keydata)) {
 *      pc.printf("Keydata 0..3 = 0x%02x 0x%02x 0x%02x 0x%02x\r\n", keydata[0], keydata[1], keydata[2], keydata[3]);
 *
 *      if (keydata[0] == 0x01) { //sw1   
 *        TM1638.cls(); 
 *        TM1638.writeData(all_str);
 *      }  
 *    } 
 *   }   
 * }
 * @endcode
 */

//Memory size in bytes for Display and Keymatrix
#define TM1638_DISPLAY_MEM    16
#define TM1638_KEY_MEM         4
//Significant bits Keymatrix data
#define TM1638_KEY_BITS        6 

//Reserved bits for commands
#define TM1638_CMD_MSK      0xC0

//Data setting commands
#define TM1638_DATA_SET_CMD 0x40
#define TM1638_DATA_WR      0x00
#define TM1638_KEY_RD       0x02
#define TM1638_ADDR_INC     0x00
#define TM1638_ADDR_FIXED   0x04
#define TM1638_MODE_NORM    0x00
#define TM1638_MODE_TEST    0x08

//Address setting commands
#define TM1638_ADDR_SET_CMD 0xC0
#define TM1638_ADDR_MSK     0x0F

//Display control commands
#define TM1638_DSP_CTRL_CMD 0x80
#define TM1638_BRT_MSK      0x07
#define TM1638_BRT0         0x00 //Pulsewidth 1/16
#define TM1638_BRT1         0x01
#define TM1638_BRT2         0x02
#define TM1638_BRT3         0x03
#define TM1638_BRT4         0x04
#define TM1638_BRT5         0x05
#define TM1638_BRT6         0x06
#define TM1638_BRT7         0x07 //Pulsewidth 14/16

#define TM1638_BRT_DEF      TM1638_BRT3

#define TM1638_DSP_OFF      0x00
#define TM1638_DSP_ON       0x08


/** A class for driving TM1638 LED controller
 *
 * @brief Supports 8 Grids @ 10 Segments. 
 *        Also supports a scanned keyboard of upto 24 keys.
 *        SPI bus interface device. 
 */
class TM1638 {
 public:

  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[TM1638_DISPLAY_MEM];
  typedef char KeyData_t[TM1638_KEY_MEM];
    
 /** Constructor for class for driving TM1638 LED controller
  *
  * @brief Supports 8 Grids @ 10 segments. 
  *        Also supports a scanned keyboard of upto 24 keys.
  *        SPI bus interface device. 
  *
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
  TM1638(PinName mosi, PinName miso, PinName sclk, PinName cs);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to TM1638
   *  @param  int address display memory location to write byte
   *  @param  char data byte written at given address
   *  @return none
   */ 
  void writeData(int address, char data); 
 
 /** Write Display datablock to TM1638
   *  @param  DisplayData_t data Array of TM1638_DISPLAY_MEM (=16) bytes for displaydata (starting at address 0)
   *  @param  length number bytes to write (valid range 0..TM1638_DISPLAY_MEM (=16), starting at address 0)   
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = TM1638_DISPLAY_MEM);

  /** Read keydata block from TM1638
   *  @param  *keydata Ptr to Array of TM1638_KEY_MEM (=4) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the TM1638 key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = TM1638_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  char _display;
  char _bright; 
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The TM1638 expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write command and parameter to TM1638
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
  void _writeCmd(int cmd, int data);  
};


#if (LEDKEY8_TEST == 1) 
// Derived class for TM1638 used in LED&KEY display unit
//

#include "Font_7Seg.h"

#define LEDKEY8_NR_GRIDS  8
#define LEDKEY8_NR_DIGITS 8
#define LEDKEY8_NR_UDC    8

//Access to 8 Switches
#define LEDKEY8_SW1_IDX   0
#define LEDKEY8_SW1_BIT   0x01
#define LEDKEY8_SW2_IDX   1
#define LEDKEY8_SW2_BIT   0x01
#define LEDKEY8_SW3_IDX   2
#define LEDKEY8_SW3_BIT   0x01
#define LEDKEY8_SW4_IDX   3
#define LEDKEY8_SW4_BIT   0x01

#define LEDKEY8_SW5_IDX   0
#define LEDKEY8_SW5_BIT   0x10
#define LEDKEY8_SW6_IDX   1
#define LEDKEY8_SW6_BIT   0x10
#define LEDKEY8_SW7_IDX   2
#define LEDKEY8_SW7_BIT   0x10
#define LEDKEY8_SW8_IDX   3
#define LEDKEY8_SW8_BIT   0x10

/** Constructor for class for driving TM1638 controller as used in LEDKEY8
  *
  *  @brief Supports 8 Digits of 7 Segments + DP + LED Icons, Also supports a scanned keyboard of 8 keys.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class TM1638_LEDKEY8 : public TM1638, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  enum Icon {
    LD1  = (1<<24) | S7_LD1,
    LD2  = (2<<24) | S7_LD2,
    LD3  = (3<<24) | S7_LD3,
    LD4  = (4<<24) | S7_LD4,
    LD5  = (5<<24) | S7_LD5,
    LD6  = (6<<24) | S7_LD6,
    LD7  = (7<<24) | S7_LD7,
    LD8  = (8<<24) | S7_LD8,
                                
    DP1  = (1<<24) | S7_DP1,
    DP2  = (2<<24) | S7_DP2,
    DP3  = (3<<24) | S7_DP3,
    DP4  = (4<<24) | S7_DP4,
    DP5  = (5<<24) | S7_DP5,
    DP6  = (6<<24) | S7_DP6,
    DP7  = (7<<24) | S7_DP7,
    DP8  = (8<<24) | S7_DP8                            
  };
  
  typedef char UDCData_t[LEDKEY8_NR_UDC];
  
 /** Constructor for class for driving TM1638 LED controller as used in LEDKEY8
   *
   * @brief Supports 8 Digits of 7 Segments + DP + LED Icons. Also supports a scanned keyboard of 8 keys.
   *  
   * @param  PinName mosi, miso, sclk, cs SPI bus pins
   */
  TM1638_LEDKEY8(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to TM1638
     *  @param  int address display memory location to write byte
     *  @param  char data byte written at given address
     *  @return none
     */ 
    void writeData(int address, char data){
      TM1638::writeData(address, data);
    }        

   /** Write Display datablock to TM1638
    *  @param  DisplayData_t data Array of TM1638_DISPLAY_MEM (=16) bytes for displaydata (starting at address 0)
    *  @param  length number bytes to write (valid range 0..(LEDKEY8_NR_GRIDS*2) (=16), starting at address 0)   
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (LEDKEY8_NR_GRIDS*2)) {
      TM1638::writeData(data, length);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;
    int _columns;   
    
    DisplayData_t _displaybuffer;
    UDCData_t _UDC_7S; 
};
#endif

#endif