/* mbed SWO Library
 *  Copyright (c) 2014, v01: WH. Ported from Segger example
 *
 * Simple implementation for tracing via Single Wire Output(SWO) for Cortex-M processors.
 * It can be used with Host PC software such as ST-LINK Utility or Segger J-Link SWO viewer.
 * This sample implementation ensures that output via SWO is enabled in order to guarantee
 * that the application does not hang.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "SWO.h"

/**
 *
 * Defines for Cortex-M debug unit
 */
#define ITM_STIM_U32(n) (*(volatile unsigned int*)(0xE0000000+4*n))  // Stimulus Port n Register word access
#define ITM_STIM_U8(n)  (*(volatile         char*)(0xE0000000+n))    // Stimulus Port n Register byte access
//#define ITM_STIM_U32_0  (*(volatile unsigned int*)0xE0000000)        // Stimulus Port 0 Register word access
//#define ITM_STIM_U8_0   (*(volatile         char*)0xE0000000)        // Stimulus Port 0 Register byte access
#define ITM_ENA         (*(volatile unsigned int*)0xE0000E00)        // Trace Enable Ports Register
#define ITM_TCR         (*(volatile unsigned int*)0xE0000E80)        // Trace control register

/**
 *
 * SWO_PrintChar()
 *
 * @brief 
 *   Checks if SWO is set up. If it is not, return,
 *    to avoid program hangs if no debugger is connected.
 *   If it is set up, print a character to the ITM_STIM register
 *    in order to provide data for SWO.
 * @param c The Character to be printed.
 * @notes   Additional checks for device specific registers can be added.
 */
void SWO_PrintChar(char c) {

  // Check if ITM_TCR.ITMENA is set
  if ((ITM_TCR & 1) == 0) {
    return;
  }

  // Check if stimulus port is enabled
  if ((ITM_ENA & 1) == 0) {
    return;
  }

  // Wait until STIMx FIFO is ready, then send data
  while ((ITM_STIM_U8(0) & 1) == 0);
  ITM_STIM_U8(0) = c;

//  while ((ITM_STIM_U32(0) & 1) == 0);
//  ITM_STIM_U32(0) = c;
}

/**
 *
 * SWO_PrintString()
 *
 * @brief Print a string via SWO.
 * @param *s The string to be printed.
 *
 */
void SWO_PrintString(const char *s) {

  // Print out characters until \0
  while (*s) {
    SWO_PrintChar(*s++);
  }
}