/* mbed STLED316S Library, for STLED316S LED controller
 * Copyright (c) 2016, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef STLED316S_H
#define STLED316S_H

// Select one of the testboards for the STM STLED316S LED controller
#include "STLED316S_Config.h"

#include "Font_7Seg.h"

/** An interface for driving STLED316S LED controller
 *
 * @code
 * #include "mbed.h"
 * #include "STLED316S.h" 
 *
 * DisplayData_t size is  1 bytes (1 grid @ 8 segments) thru 6 bytes (6 grids @ 8 segments)
 * STLED316S::DisplayData_t all_str  = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};  
 * STLED316S::DisplayData_t bye_str  = {LO(C7_B), LO(C7_Y), LO(C7_E), 0x00, 0x00, 0x00};
 * STLED316S::LedData_t all_led  = {0xFF};  
 *
 * // KeyData_t size is 2 bytes  
 * STLED316S::KeyData_t keydata; 
 *
 * // STLED316S declaration, Default setting 6 Grids @ 8 Segments
 * STLED316S STLED316S(p5,p6,p7, p8);
 *
 * int main() {
 *     
 *   STLED316S.cls(); 
 *   STLED316S.writeData(all_str);
 *   STLED316S.writeLedData(all_led);    
 *   wait(2);
 *
 *   STLED316S.setBrightMode(STLED316S::GlobalBright);  
 *   STLED316S.setBrightness(STLED316S_BRT3);    
 *   wait(1);
 *   STLED316S.setBrightness(STLED316S_BRT0);        
 *   wait(1);
 *   STLED316S.setBrightness(STLED316S_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (STLED316S.getKeys(&keydata)) {
 *      //pc.printf("Keydata 0..1 = 0x%02x 0x%02x\r\n", keydata[0], keydata[1]);
 *
 *      if (keydata[0] == 0x01) { // Key1&KS1
 *        wait(1);
 *        STLED316S.setLed(STLED316S_LED_L1);
 *        wait(1);
 *        STLED316S.clrLed(STLED316S_LED_L1);
 *      }  
 *    } 
 *   }   
 * }
 * @endcode
 */

//STLED316S Display and Keymatrix data
#define STLED316S_MAX_NR_GRIDS    6
#define STLED316S_BYTES_PER_GRID  1
//Significant bits Keymatrix data
#define STLED316S_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define STLED316S_DISPLAY_MEM  (STLED316S_MAX_NR_GRIDS * STLED316S_BYTES_PER_GRID)
#define STLED316S_KEY_MEM         2

//Address set commands
#define STLED316S_CMD_MSK      0x7F
#define STLED316S_DATA_WR      0x00
#define STLED316S_DATA_RD      0x40
#define STLED316S_ADDR_INC     0x00
#define STLED316S_ADDR_FIXED   0x20
#define STLED316S_PAGE_MSK     0x18
#define STLED316S_PAGE_SHFT       3
#define STLED316S_ADDR_MSK     0x07
#define STLED316S_ADDR_SHFT       0
#define STLED316S_IDX_MSK      (STLED316S_PAGE_MSK | STLED316S_ADDR_MSK)

//Read and Write commands
#define STLED316S_ADDR_WR_CMD  (STLED316S_DATA_WR | STLED316S_ADDR_INC) 
#define STLED316S_ADDR_RD_CMD  (STLED316S_DATA_RD | STLED316S_ADDR_INC) 

//Combined Page and Address parameters
#define STLED316S_IDX(page, addr) ( ((page << STLED316S_PAGE_SHFT) & STLED316S_PAGE_MSK) | ((addr << STLED316S_ADDR_SHFT) & STLED316S_ADDR_MSK) )   


//Digit Data address (Digit2..Digit7)
#define STLED316S_DIG_PAGE        0
#define STLED316S_DIG2_ADDR    0x00
#define STLED316S_DIG3_ADDR    0x01
#define STLED316S_DIG4_ADDR    0x02
#define STLED316S_DIG5_ADDR    0x03
#define STLED316S_DIG6_ADDR    0x04
#define STLED316S_DIG7_ADDR    0x05

//LED Data address (also Digit1 Data address)
#define STLED316S_DIG1_LED_PAGE         1
#define STLED316S_DIG1_LED_ADDR      0x00

//LED Data used to select individual LEDs
#define STLED316S_LED_L1       0x01
#define STLED316S_LED_L2       0x02
#define STLED316S_LED_L3       0x04
#define STLED316S_LED_L4       0x08
#define STLED316S_LED_L5       0x10
#define STLED316S_LED_L6       0x20
#define STLED316S_LED_L7       0x40
#define STLED316S_LED_L8       0x80
#define STLED316S_LED_ALL      0xFF

//Digit Data used to select individual digits
//#define STLED316S_DIG_D1       0x01
#define STLED316S_DIG_D2       0x02
#define STLED316S_DIG_D3       0x04
#define STLED316S_DIG_D4       0x08
#define STLED316S_DIG_D5       0x10
#define STLED316S_DIG_D6       0x20
#define STLED316S_DIG_D7       0x40
#define STLED316S_DIG_ALL      0x7E


//Key Data address
#define STLED316S_KEY_PAGE        1
#define STLED316S_KEY1_ADDR    0x01
#define STLED316S_KEY2_ADDR    0x02

//Display On_Off address
#define STLED316S_DSP_PAGE        1
#define STLED316S_DSP_ON_ADDR  0x05
#define STLED316S_DSP_OFF_ADDR 0x06

//Config set address
#define STLED316S_CONF_PAGE             2
#define STLED316S_CONF_ADDR          0x00

//Config parameters
#define STLED316S_CONF_GRID_MSK      0x07
#define STLED316S_CONF_GRID_SHFT        0
#define STLED316S_CONF_BRT_MODE_MSK  0x18
#define STLED316S_CONF_BRT_MODE_SHFT    3
#define STLED316S_CONF_BRT_GLOB_MSK  0xE0
#define STLED316S_CONF_BRT_GLOB_SHFT    5

//Grid Config parameters
#define STLED316S_GR1_SEG8     0x00
#define STLED316S_GR2_SEG8     0x01
#define STLED316S_GR3_SEG8     0x02
#define STLED316S_GR4_SEG8     0x03 //default
#define STLED316S_GR5_SEG8     0x04
#define STLED316S_GR6_SEG8     0x05

//Brightness Mode Config parameters
#define STLED316S_BRT_INDIV    0x00
#define STLED316S_BRT_GLOB     0x03 //default

//Digit Brightness address (used when Brightness mode is set as 'individual')
#define STLED316S_DIG_BRT_PAGE          2
#define STLED316S_DIG2_3_BRT_ADDR    0x01
#define STLED316S_DIG4_5_BRT_ADDR    0x02
#define STLED316S_DIG6_7_BRT_ADDR    0x03

//LED Brightness address (used when Brightness mode is set as 'individual')
#define STLED316S_LED_BRT_PAGE          3
#define STLED316S_LED1_2_BRT_ADDR    0x00
#define STLED316S_LED3_4_BRT_ADDR    0x01
#define STLED316S_LED5_6_BRT_ADDR    0x02
#define STLED316S_LED7_8_BRT_ADDR    0x03

//Brightness control parameters (used for both Brightness modes 'global' and 'individual')
#define STLED316S_BRT_MSK      0x07

#define STLED316S_BRT0         0x00 //Pulsewidth 1/16
#define STLED316S_BRT1         0x01
#define STLED316S_BRT2         0x02
#define STLED316S_BRT3         0x03
#define STLED316S_BRT4         0x04
#define STLED316S_BRT5         0x05
#define STLED316S_BRT6         0x06
#define STLED316S_BRT7         0x07 //Pulsewidth 14/16 (default)

#define STLED316S_BRT_DEF      STLED316S_BRT3



/** A class for driving STM STLED316S LED controller
 *
 * @brief Supports 1..6 Grids @ 8 Segments and 8 LEDs. 
 *        Also supports a scanned keyboard of upto 16 keys.
 *        SPI bus interface device. 
 */
class STLED316S {
 public:

  /** Enums for Display mode */
  enum Mode {
    Grid1_Seg8 = STLED316S_GR1_SEG8,
    Grid2_Seg8 = STLED316S_GR2_SEG8,   
    Grid3_Seg8 = STLED316S_GR3_SEG8,
    Grid4_Seg8 = STLED316S_GR4_SEG8,
    Grid5_Seg8 = STLED316S_GR5_SEG8,
    Grid6_Seg8 = STLED316S_GR6_SEG8   
  };
  
  /** Enums for Brightness mode */
  enum BrightMode {
    IndivBright  = STLED316S_BRT_INDIV,
    GlobalBright = STLED316S_BRT_GLOB      
  };

  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[STLED316S_DISPLAY_MEM];
  typedef char LedData_t;  
  typedef char KeyData_t[STLED316S_KEY_MEM];
    
 /** Constructor for class for driving STLED316S LED controller
  *
  * @brief Supports 1..6 Grids @ 8 Segments and 8 LEDs. 
  *        Also supports a scanned keyboard of upto 16 keys.
  *        SPI bus interface device. 
  *
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  *  @param  Mode selects either Grids/Segments (default 6 Grids @ 8 Segments)
  */
  STLED316S(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode=Grid6_Seg8);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to STLED316S
   *  @param  int address display memory location to write byte
   *  @param  char data byte written at given address
   *  @return none
   */ 
  void writeData(int address, char data); 
 
 /** Write Display datablock to STLED316S
   *  @param  DisplayData_t data Array of STLED316S_DISPLAY_MEM (=6) bytes for displaydata (starting at address 0)
   *  @param  length number bytes to write (valid range 0..STLED316S_DISPLAY_MEM (=6), starting at address 0)   
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = STLED316S_DISPLAY_MEM);

 /** Write LED data to STLED316S
   *  @param  LedData_t leds LED data   
   *  @return none
   */   
  void writeLedData(LedData_t leds);

  /** Set LED
    *
    * @param LedData_t leds pattern of LED data      
    * @return none
    */
  void setLed(LedData_t leds);

  /** Clr LED
    *
    * @param LedData_t leds pattern of LED data      
    * @return none
    */
  void clrLed(LedData_t leds);


  /** Read keydata block from STLED316S
   *  @param  *keydata Ptr to Array of STLED316S_KEY_MEM (=2) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the STLED316S key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);

  /** Set Brightness for all Digits and LEDs (value is used in GlobalBrightness mode)
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = STLED316S_BRT_DEF);

  /** Set Individual LED Brightness (value is used in IndivBright mode)
    *
    * @param  LedData_t leds pattern of LED data          
    * @param  char led_brt (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
    * @return none
    */
  void setLedBrightness(LedData_t leds = STLED316S_LED_ALL, char led_brt = STLED316S_BRT_DEF);
  
  /** Set Individual Digit Brightness (value is used in IndivBright mode)
    *
    * @param  LedData_t digits pattern of Digit data
    * @param  char dig_brt (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
    * @return none
    */    
  void setDigitBrightness(LedData_t digits = STLED316S_DIG_ALL, char dig_brt = STLED316S_BRT_DEF);

  /** Set Brightness mode 
    *
    * @param  BrightMode brt_mode (value is IndivBright or GlobalBright)  
    * @return none
    */
  void setBrightMode(BrightMode brt_mode = GlobalBright);


  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  BrightMode _brt_mode;
  char _bright;
  LedData_t _leds;
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The STLED316S expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write parameter to STLED316S Register
    *  @param  int idx Register address
    *  @param  int data Parameter for Register
    *  @return none
    */ 
  void _writeReg(int idx, int data);  
  
/** Write merged command and parameter to STLED316S
  *  @param  int cmd Command & Parameter byte
  *  @return none
  */  
  void _writeReg(int cmd);

  /** Read parameter from STLED316S Register
    *  @param  int idx Register address
    *  @return char data from Register
    */ 
  char _readReg(int idx);    
};



#if(ST316BOARD_TEST == 1)
// Derived class for STLED316S used in test display module
//
#include "Font_7Seg.h"

#define ST316BOARD_NR_GRIDS    6
#define ST316BOARD_NR_DIGITS   6
//#define ST316BOARD_DIG1_IDX    0
#define ST316BOARD_NR_UDC      8

/** Constructor for class for driving STM STLED316S controller as used in ST316S test display
  *
  *  @brief Supports 6 Digits of 8 Segments, 1 Grid of 3 LEDs. Also supports a scanned keyboard of 3 keys.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class STLED316S_BOARD : public STLED316S, public Stream {
 public:

  /** Enums for LEDs */
  //  Grid encoded in 8 MSBs, LED pattern encoded in 16 LSBs
  enum Icon {
    LD1  = (0<<24) | STLED316S_LED_L1,
    LD2  = (0<<24) | STLED316S_LED_L2,
    LD3  = (0<<24) | STLED316S_LED_L3,
  };
  
  typedef short UDCData_t[ST316BOARD_NR_UDC];
  
 /** Constructor for class for driving STLED316S controller as used in ST316S test display
   *
   * @brief Supports 6 Digits of 8 Segments, 1 Grid of 3 LEDs. Also supports a scanned keyboard of 3 keys.
   *  
   * @param  PinName mosi, miso, sclk, cs SPI bus pins
   */
  STLED316S_BOARD(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);   

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to STLED316S
     *  @param  int address display memory location to write byte
     *  @param  char data byte written at given address
     *  @return none
     */ 
    void writeData(int address, char data){
      STLED316S::writeData(address, data);
    }        

   /** Write Display datablock to STLED316S
    *  @param  DisplayData_t data Array of STLED316S_DISPLAY_MEM (=6) bytes for displaydata (starting at address 0)
    *  @param  length number bytes to write (valid range 0..(ST316BOARD_NR_GRIDS) (=6), starting at address 0)   
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (ST316BOARD_NR_GRIDS)) {
      STLED316S::writeData(data, length);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;
    int _columns;   
    
    DisplayData_t _displaybuffer;
    UDCData_t _UDC_7S; 
};
#endif


#endif