/* mbed STLED316S Library, for STLED316S LED controller
 * Copyright (c) 2016, v01: WH, Initial version
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "STLED316S.h"
#include "Font_7Seg.h"

/** Constructor for class for driving STLED316S LED controller with SPI bus interface device. 
 *  @brief Supports 1..6 digits @ 8 segments and 8 LEDs. 
 *         Also supports a scanned keyboard of upto 16 keys.
 *   
 *  @param  PinName mosi, miso, sclk, cs SPI bus pins
 *  @param  Mode selects 1..6 Digits of 8 Segments (default 6 Digits of 8 Segments) 
*/
STLED316S::STLED316S(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode) : _spi(mosi,miso,sclk), _cs(cs), _mode(mode) {

  _init();
}

/** Init the SPI interface and the controller
  * @param  none
  * @return none
  */ 
void STLED316S::_init(){
  char config;  

//init SPI
  _cs=1;
  _spi.format(8,3); //STLED316S uses mode 3 (Clock High on Idle, Data latched on second (=rising) edge)
  _spi.frequency(500000);   

//init controller  
  _brt_mode = GlobalBright; //STLED316S_BRT_GLOB;
//  _brt_mode = IndivBright;  //STLED316S_BRT_INDIV;  
  _bright   = STLED316S_BRT_DEF; 

  config = ((_mode << STLED316S_CONF_GRID_SHFT) & STLED316S_CONF_GRID_MSK) | 
           ((_brt_mode << STLED316S_CONF_BRT_MODE_SHFT) & STLED316S_CONF_BRT_MODE_MSK) |
           ((_bright << STLED316S_CONF_BRT_GLOB_SHFT) & STLED316S_CONF_BRT_GLOB_MSK);
            
  _writeReg(STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config);    // Config set command 
//  printf("Cmd= 0x%02X, Conf= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config); // Debug   
   
  _writeReg(STLED316S_IDX(STLED316S_DSP_PAGE,STLED316S_DSP_ON_ADDR));           // Display On command      
}   

                          
/** Clear the screen and locate to 0
 */  
void STLED316S::cls() {
  
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG_PAGE,STLED316S_DIG2_ADDR))); // Address set cmd, 0
      
  for (int cnt=0; cnt<STLED316S_DISPLAY_MEM; cnt++) {
    _spi.write(0x00); // data 
  }
  
  wait_us(1);
  _cs=1;      
}  

/** Set Brightness for all Digits and LEDs (value is used in GlobalBright mode)
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */ 
void STLED316S::setBrightness(char brightness){
  char config;
  
  _bright = brightness & STLED316S_BRT_MSK; // mask invalid bits
  
  config = ((_mode << STLED316S_CONF_GRID_SHFT) & STLED316S_CONF_GRID_MSK) | 
           ((_brt_mode << STLED316S_CONF_BRT_MODE_SHFT) & STLED316S_CONF_BRT_MODE_MSK) |
           ((_bright << STLED316S_CONF_BRT_GLOB_SHFT) & STLED316S_CONF_BRT_GLOB_MSK);
            
  _writeReg(STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config);    // Config set command 
//  printf("Cmd= 0x%02X, Conf= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config); // Debug   
}


/** Set Individual LED Brightness (value is used in IndivBright mode)
  *
  * @param  LedData_t leds pattern of LED data
  * @param  char led_brt (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void STLED316S::setLedBrightness(LedData_t leds, char led_brt) {
  char brt_old;
  
  //Sanity check
  led_brt &= STLED316S_BRT_MSK;  
  
  if (leds & STLED316S_LED_L1) { // LED_1
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED1_2_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED1_2_BRT_ADDR), (brt_old & 0xF0) | led_brt);         // LED Brt set command 
  }

  if (leds & STLED316S_LED_L2) { // LED_2
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED1_2_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED1_2_BRT_ADDR), (brt_old & 0x0F) | (led_brt << 4));  // LED Brt set command 
  }

  if (leds & STLED316S_LED_L3) { // LED_3
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED3_4_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED3_4_BRT_ADDR), (brt_old & 0xF0) | led_brt);         // LED Brt set command 
  }

  if (leds & STLED316S_LED_L4) { // LED_4
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED3_4_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED3_4_BRT_ADDR), (brt_old & 0x0F) | (led_brt << 4));  // LED Brt set command 
  }

  if (leds & STLED316S_LED_L5) { // LED_5
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED5_6_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED5_6_BRT_ADDR), (brt_old & 0xF0) | led_brt);         // LED Brt set command 
  }

  if (leds & STLED316S_LED_L6) { // LED_6
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED5_6_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED5_6_BRT_ADDR), (brt_old & 0x0F) | (led_brt << 4));  // LED Brt set command 
  }

  if (leds & STLED316S_LED_L7) { // LED_7
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED7_8_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED7_8_BRT_ADDR), (brt_old & 0xF0) | led_brt);         // LED Brt set command 
  }

  if (leds & STLED316S_LED_L8) { // LED_8
    brt_old = _readReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED7_8_BRT_ADDR));                            // LED Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_LED_BRT_PAGE,STLED316S_LED7_8_BRT_ADDR), (brt_old & 0x0F) | (led_brt << 4));  // LED Brt set command 
  }
}


/** Set Individual Digit Brightness (value is used in IndivBright mode)
  *
  * @param  LedData_t digits pattern of Digit data
  * @param  char dig_brt (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void STLED316S::setDigitBrightness(LedData_t digits, char dig_brt) {
  char brt_old;
  
  //Sanity check
  dig_brt &= STLED316S_BRT_MSK;  

  if (digits & STLED316S_DIG_D2) { // DIGIT_2
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG2_3_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG2_3_BRT_ADDR), (brt_old & 0xF0) | dig_brt);         // Digit Brt set command 
  }

  if (digits & STLED316S_DIG_D3) { // DIGIT_3
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG2_3_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG2_3_BRT_ADDR), (brt_old & 0x0F) | (dig_brt << 4));  // Digit Brt set command 
  }

  if (digits & STLED316S_DIG_D4) { // DIGIT_4
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG4_5_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG4_5_BRT_ADDR), (brt_old & 0xF0) | dig_brt);         // Digit Brt set command 
  }

  if (digits & STLED316S_DIG_D5) { // DIGIT_5
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG4_5_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG4_5_BRT_ADDR), (brt_old & 0x0F) | (dig_brt << 4));  // Digit Brt set command 
  }

  if (digits & STLED316S_DIG_D6) { // DIGIT_6
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG6_7_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG6_7_BRT_ADDR), (brt_old & 0xF0) | dig_brt);         // Digit Brt set command 
  }

  if (digits & STLED316S_DIG_D7) { // DIGIT_7
    brt_old = _readReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG6_7_BRT_ADDR));                            // Digit Brt read command 
    _writeReg(STLED316S_IDX(STLED316S_DIG_BRT_PAGE,STLED316S_DIG6_7_BRT_ADDR), (brt_old & 0x0F) | (dig_brt << 4));  // Digit Brt set command 
  }
}


/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void STLED316S::setDisplay(bool on) {
  
  if (on) {
    _writeReg(STLED316S_IDX(STLED316S_DSP_PAGE,STLED316S_DSP_ON_ADDR));  // Display On command       
  }
  else {
   _writeReg(STLED316S_IDX(STLED316S_DSP_PAGE,STLED316S_DSP_OFF_ADDR));  // Display Off command          
  }
}

/** Set Brightness mode 
  *
  * @param  BrightMode brt_mode (value is IndivBright or GlobalBright)  
  * @return none
  */
void STLED316S::setBrightMode(BrightMode brt_mode) {
  char config;
  
  _brt_mode = brt_mode; // mask invalid bits
  
  config = ((_mode << STLED316S_CONF_GRID_SHFT) & STLED316S_CONF_GRID_MSK) | 
           ((_brt_mode << STLED316S_CONF_BRT_MODE_SHFT) & STLED316S_CONF_BRT_MODE_MSK) |
           ((_bright << STLED316S_CONF_BRT_GLOB_SHFT) & STLED316S_CONF_BRT_GLOB_MSK);
            
  _writeReg(STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config);    // Config set command 
//  printf("Cmd= 0x%02X, Conf= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_CONF_PAGE,STLED316S_CONF_ADDR), config); // Debug   
}


/** Write databyte to STLED316S
  *  @param  int address display memory location to write byte
  *  @param  char data byte written at given address
  *  @return none
  */ 
void STLED316S::writeData(int address, char data) {
  _cs=0;
  wait_us(1);    

  _spi.write(_flip(STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG_PAGE, address))); // Address set cmd
  _spi.write(_flip(data)); // data 
      
  wait_us(1);
  _cs=1;             
}

/** Write Display datablock to STLED316S
  *  @param  DisplayData_t data Array of STLED316S_DISPLAY_MEM (=6) bytes for displaydata (starting at address 0)
  *  @param  length number bytes to write (valide range 0..STLED316S_DISPLAY_MEM (=6), starting at address 0)     
  *  @return none
  */ 
void STLED316S::writeData(DisplayData_t data, int length) {
  _cs=0;
  wait_us(1);    

  _spi.write(_flip(STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG_PAGE, 0x00))); // Set Address at 0

//  printf("Cmd= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG_PAGE, 0x00)); // Debug   
      
// sanity check
  if (length < 0) {length = 0;}
  if (length > STLED316S_DISPLAY_MEM) {length = STLED316S_DISPLAY_MEM;}

//  for (int idx=0; idx<STLED316S_DISPLAY_MEM; idx++) {  
  for (int idx=0; idx<length; idx++) {    
    _spi.write(_flip(data[idx])); // data 

//    printf("Data= 0x%02X\r\n", data[idx]); // Debug       
  }
  
  wait_us(1);
  _cs=1;             
}


/** Write LED data to STLED316S
  *  @param  LedData_t leds LED data
  *  @return none
  */   
void STLED316S::writeLedData(LedData_t leds) {
  _cs=0;
  wait_us(1);    

  _leds = leds;
  
  _spi.write(_flip(STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG1_LED_PAGE, STLED316S_DIG1_LED_ADDR))); // Set Address
//  printf("Cmd= 0x%02Xr\n", STLED316S_ADDR_WR_CMD | STLED316S_IDX(STLED316S_DIG1_LED_PAGE, STLED316S_DIG1_LED_ADDR)); // Debug          

  _spi.write(_flip(leds)); // data 
//  printf("Data 0x%02X\r\n", leds); // Debug       
  
  wait_us(1);
  _cs=1;             
}


/** Set LED
  *
  * @param LedData_t leds pattern of LED data
  * @return none
  */
void STLED316S::setLed(LedData_t leds){

  _leds |= leds;
  writeLedData(_leds);  
}

/** Clr LED
  *
  * @param LedData_t leds pattern of LED data  
  * @return none
  */
void STLED316S::clrLed(LedData_t leds){
  _leds &= ~leds;
  writeLedData(_leds);  
}

/** Read keydata block from STLED316S
  *  @param  *keydata Ptr to Array of STLED316S_KEY_MEM (=5) bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  *
  * Note: Due to the hardware configuration the STLED316S key matrix scanner will detect multiple keys pressed at same time,
  *       but this may also result in some spurious keys being set in keypress data array.
  *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.  
  */ 
bool STLED316S::getKeys(KeyData_t *keydata) {
  int keypress = 0;
  char data;

  // Read keys
  _cs=0;
  wait_us(1);    
  
  // Enable Key Read mode
  _spi.write(_flip(STLED316S_ADDR_RD_CMD | STLED316S_IDX(STLED316S_KEY_PAGE, STLED316S_KEY1_ADDR))); // Set Address
//  printf("Cmd= 0x%02Xr\n", STLED316S_ADDR_RD_CMD | STLED316S_IDX(STLED316S_KEY_PAGE, STLED316S_KEY1_ADDR)); // Debug          

  for (int idx=0; idx < STLED316S_KEY_MEM; idx++) {
    data = _flip(_spi.write(0xFF));    // read keys and correct bitorder
//    printf("KeyData 0x%02X\r\n", data); // Debug       

    data = data & STLED316S_KEY_MSK; // Mask valid bits
    if (data != 0) {  // Check for any pressed key
      for (int bit=0; bit < 8; bit++) {
        if (data & (1 << bit)) {keypress++;} // Test all significant bits
      }
    }  

    (*keydata)[idx] = data;            // Store keydata after correcting bitorder
  }

  wait_us(1);
  _cs=1;    
     
#if(1)
// Dismiss multiple keypresses at same time
  return (keypress == 1);    
#else
// Allow multiple keypress and accept possible spurious keys
  return (keypress > 0);
#endif  
}
    

/** Helper to reverse all command or databits. The STLED316S expects LSB first, whereas SPI is MSB first
  *  @param  char data
  *  @return bitreversed data
  */ 
char STLED316S::_flip(char data) {
 char value=0;
  
 if (data & 0x01) {value |= 0x80;} ;  
 if (data & 0x02) {value |= 0x40;} ;
 if (data & 0x04) {value |= 0x20;} ;
 if (data & 0x08) {value |= 0x10;} ;
 if (data & 0x10) {value |= 0x08;} ;
 if (data & 0x20) {value |= 0x04;} ;
 if (data & 0x40) {value |= 0x02;} ;
 if (data & 0x80) {value |= 0x01;} ;
 return value;       
}


/** Write parameter to STLED316S Register
  *  @param  int idx Register address
  *  @Param  int data Parameter for Register
  *  @return none
  */  
void STLED316S::_writeReg(int idx, int data){
    
  _cs=0;
  wait_us(1);    
 
  _spi.write(_flip(STLED316S_ADDR_WR_CMD | (idx & STLED316S_IDX_MSK)));
  _spi.write(_flip(data) );

//  printf("Cmd= 0x%02X, Write= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | (idx & STLED316S_IDX_MSK), data); // Debug        
       
  wait_us(1);
  _cs=1;          
}  


/** Write merged command and parameter to STLED316S
  *  @param  int cmd Command & Parameter byte
  *  @return none
  */  
void STLED316S::_writeReg(int cmd){
    
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(STLED316S_ADDR_WR_CMD | (cmd & STLED316S_IDX_MSK)));

//  printf("Cmd= 0x%02X\r\n", STLED316S_ADDR_WR_CMD | (cmd & STLED316S_IDX_MSK)); // Debug        

  wait_us(1);
  _cs=1;          
}  


/** Read parameter from STLED316S Register
  *  @param  int idx Register address
  *  @return char data from Register
  */ 
char STLED316S::_readReg(int idx){
  char data;

  _cs=0;
  wait_us(1);    
  _spi.write(_flip(STLED316S_ADDR_RD_CMD | (idx & STLED316S_IDX_MSK)));

  data = _flip(_spi.write(0xFF));    // read data and correct bitorder

//  printf("Cmd= 0x%02X, Read= 0x%02X\r\n", STLED316S_ADDR_RD_CMD | (idx & STLED316S_IDX_MSK), data); // Debug        

  wait_us(1);
  _cs=1;          
 
  return data;
}        



#if(ST316BOARD_TEST == 1)
// Derived class for STLED316S used in test display module
//
//#include "Font_7Seg.h"

/** Constructor for class for driving STM STLED316S controller as used in ST316S test display
  *
  *  @brief Supports 6 Digits of 7 Segments and 3 LEDs. Also supports a scanned keyboard of 3.
  *   
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
STLED316S_BOARD::STLED316S_BOARD(PinName mosi, PinName miso, PinName sclk, PinName cs) : STLED316S(mosi, miso, sclk, cs, Grid6_Seg8) {
  _column  = 0;
  _columns = ST316BOARD_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  */
void STLED316S_BOARD::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int STLED316S_BOARD::columns() {
    return _columns;
}


/** Clear the screen and locate to 0
  * @param bool clrAll Clear Icons also (default = false)
  */ 
void STLED316S_BOARD::cls(bool clrAll) {  

  if (clrAll) {
    //clear local buffer for LEDs/Icons 
    setLed(0x00);    
  }  

  //clear local buffer for digits
  for (int idx=0; idx < (ST316BOARD_NR_GRIDS * STLED316S_BYTES_PER_GRID); idx++) {
    _displaybuffer[idx] = 0x00;  
  }
  writeData(_displaybuffer, (ST316BOARD_NR_GRIDS * STLED316S_BYTES_PER_GRID));

  _column = 0;   
}     


/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void STLED316S_BOARD::setIcon(Icon icon) {
//  int addr;
  int icn;

   icn =        icon  & 0xFF;
//  addr = (icon >> 24) & 0xFF; 

  setLed(icn);
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void STLED316S_BOARD::clrIcon(Icon icon) {
//  int addr;
  int icn;

   icn =        icon  & 0xFF;
//  addr = (icon >> 24) & 0xFF; 

  clrLed(icn);
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (16 bits)       
  */
void STLED316S_BOARD::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (ST316BOARD_NR_UDC-1)) {
    return;
  }
  // Mask out Icon bits?

  _UDC_7S[udc_idx] = udc_data;
}


/** Write a single character (Stream implementation)
  */
int STLED316S_BOARD::_putc(int value) {

    int addr;
    bool validChar = false;
    char pattern  = 0x00;

    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    }
    else if ((value == '.') || (value == ',')) {
      //No character to write
      validChar = false;
      pattern = S7_DP; // placeholder for all DPs
      
      // Check to see that DP can be shown for current column
      if (_column > 0) {
        //Translate between _column and displaybuffer entries
        //Add DP to bitpattern of digit left of current column.
        addr = (_column - 1);
      
        //Set bits for decimal point to write
        _displaybuffer[addr] = _displaybuffer[addr] | pattern;
        writeData(_displaybuffer, (ST316BOARD_NR_GRIDS * STLED316S_BYTES_PER_GRID));        
        
        //No Cursor Update
      }
    }
    else if ((value >= 0) && (value < ST316BOARD_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_7S[value];
    }  
    
#if (SHOW_ASCII == 1)
    //display all ASCII characters
    else if ((value >= FONT_7S_START) && (value <= FONT_7S_END)) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - FONT_7S_START];
    } // else
#else    
    //display only digits and hex characters      
    else if (value == '-') {
      //Character to write
      validChar = true;
      pattern = C7_MIN;         
    }
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'a'];
    } //else
#endif

    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      //_column == 0 => Grid0 => addr = 0
      //
      //_column == 5 => Grid5 => addr = 5            
      addr = _column;
      _displaybuffer[addr]   = pattern;

      writeData(_displaybuffer, (ST316BOARD_NR_GRIDS * STLED316S_BYTES_PER_GRID));
                                
      //Update Cursor
      _column++;
      if (_column > (ST316BOARD_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}

// get a single character (Stream implementation)
int STLED316S_BOARD::_getc() {
    return -1;
}
#endif