/** \file ssd1289.h
 *  \brief mbed TFT LCD library for displays with the SSD1289 LCD controller.
 *
 * A known display with this type of controller chip is the ITDB02-3.2S from
 * http://imall.iteadstudio.com
 *
 * This library is based on the Arduino/chipKIT UTFT library by Henning
 * Karlsen, http://henningkarlsen.com/electronics/library.php?id=52
 *
 * Copyright (C)2010-2012 Henning Karlsen. All right reserved.
 * Copyright (C)2012 Todor Todorov.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to:
 *
 * Free Software Foundation, Inc.
 * 51 Franklin St, 5th Floor, Boston, MA 02110-1301, USA
 *
 *********************************************************************/

#ifndef SSD1289_H
#define SSD1289_H

#include "mbed.h"
#include "fonts.h"

/** \def RGB(r,g,b)
 *  \brief Creates a RGB-565 color value.
 *
 * Displays which use 16 bits to assign colors to a specific pixel, use
 * 5 bits for the red component, 6 bits for the green component and 5
 * bits for the blue component.  This macro converts the 3 full-size
 * RGB bytes into one 16-bit RGB-565 value.
 */
#define RGB( r, g, b )          ( ( ( ( r ) & 248 ) | ( ( g ) >> 5 ) ) << 8 ) | ( ( ( ( g ) & 28 ) << 3 ) | ( ( b ) >> 3 ) )
/** \def COLOR_BLACK
 *  \brief Shorthand for RGB( 0, 0, 0 ).
 */
#define COLOR_BLACK             RGB( 0x00, 0x00, 0x00 )
/** \def COLOR_WHITE
 *  \brief Shorthand for RGB( 255, 255, 255 ).
 */
#define COLOR_WHITE             RGB( 0xFF, 0xFF, 0xFF )
/** \def COLOR_RED
 *  \brief Shorthand for RGB( 255, 0, 0 ).
 */
#define COLOR_RED               RGB( 0xFF, 0x00, 0x00 )
/** \def COLOR_GREEN
 *  \brief Shorthand for RGB( 255, 255, 255 ).
 */
#define COLOR_GREEN             RGB( 0x00, 0xFF, 0x00 )
/** \def COLOR_BLIUE
 *  \brief Shorthand for RGB( 0, 255, 0 ).
 */
#define COLOR_BLIUE             RGB( 0x00, 0x00, 0xFF )

/** \typedef orientation_t
 *  \brief Display orientation.
 */
typedef enum Orientation_enum
{
    PORTRAIT = 0, /**< Display height is bigger than its width. */ 
    LANDSCAPE = 1, /**< Display width is bigger than its height. */
} orientation_t;

/** \typedef align_t
 *  \brief Horizontal text alignment on the line.
 */
typedef enum Alignment_enum
{
    LEFT = 0, /**< Left-oriented, naturally gravitate closer to the left edge of the screen. */
    CENTER = 9998, /**< Center-oriented, try to fit in the middle of the available space with equal free space to the left and right of the text. */
    RIGHT = 9999, /**< Right-oriented, naturally gravitate closer to the right edge of the screen, leaving any remaining free space to the left of the text. */
} align_t;

/** \typedef font_metrics_t
 *  \brief Describes fonts and their properties.
 *  \sa Comments in fonts.h
 */
typedef struct Font_struct
{
    const    char* font; /**< A pointer to the first byte in the font. */
    unsigned char  width; /**< The width of each character, in pixels. */
    unsigned char  height; /**< Height of each character, in pixels. */
    unsigned char  offset; /**< Offset of the first character in the font. */
    unsigned char  numchars; /**< Count of the available characters in the font. */
} font_metrics_t;

/** \typedef bitmap_t
 *  \brief Pointer to the start of a block of pixel data, describing a picture.
 */
typedef unsigned short* bitmap_t;

/** Represents a LCD instance.
 *
 * This is the utility class, through which the display can be manipulated
 * and graphics objects can be shown to the user.  A known display, which
 * works with this library is the ITDB02-3.2S from iTeadStudio - a RGB TFT
 * with 240x320 pixels resolution and 65K colors, using 16-bit interface.
 *
 * The display needs 20 or 21 pins to work with mbed, so it is possibly not
 * the best of choices out there, but other than that it uses +3.3V for
 * power and logic, as well as the backlight, thus can be interfaced directly
 * to the mbed without the need of shields or level shifters as with Arduino.
 *
 * How to use:
 * \code
 * // include the library, this will also pull in the header for the provided fonts
 * #include "ssd1289.h"
 * 
 * // prepare the data bus for writing commands and pixel data
 * BusOut dataBus( p30, p29, p28, p27, p26, p25, p24, p23, p22, p21, p20, p19, p18, p17, p16, p15 ); // 16 pins
 * // create the lcd instance
 * SSD1289 lcd( p14, p13, p12, p11, &dataBus ); // control pins and data bus
 *
 * int main()
 * {
 *     // initialize display - place it in standard portrait mode and set background to black and
 *     //                      foreground to white color.
 *     lcd.Initialize();
 *     // set current font to the smallest 8x12 pixels font.
 *     lcd.SetFont( Font8x12 );
 *     // print something on the screen
 *     lcd.Print( "Hello, World!", CENTER, 25 ); // align text to center horizontally and use starndard colors
 *
 *     while ( 1 ) { }
 * }
 *
 * \endcode
 */
class SSD1289
{
public:
    /** Creates a new instance of the class.
     *
     * \param CS_PIN Pin for the ChipSelect signal.
     * \param RESET_PIN Pin for the RESET line.
     * \param RS_PIN Pin for the RS signal.
     * \param WR_PIN Pin for the WR signal.
     * \param DATA_PORT Address of the data bus for transfer of commands and pixel data.
     * \param RD_PIN Pin for the RD signal. This line is not needed by the driver, so if you would like to
     *       use the pin on the mbed for something else, just pull-up the respective pin on the LCD high,
     *       and do not assign a value to this parameter when creating the controller instance.
     */
    SSD1289( PinName CS_PIN, PinName RESET_PIN, PinName RS_PIN, PinName WR_PIN, BusOut* DATA_PORT, PinName RD_PIN = NC );
    
    /** Initialize display.
     *
     * Wakes up the display from sleep, initializes power parameters.
     * This function must be called first, befor any painting on the
     * display is done, otherwise the positioning of graphical elements
     * will not work properly and any paynt operation will not be visible
     * or produce garbage.
     *
     * \param oritentation The display orientation, landscape is default.
     */
    void Initialize( orientation_t orientation = LANDSCAPE );
    
    /** Set the foreground color for painting.
     *
     * This is the default foreground color to be used in painting operations.
     * If a specific output function allows for a different color to be specified
     * in place, the new setting will be used for that single operation only and
     * will not change this value.
     *
     * \param color The color to be used. The value must be in RGB-565 format.
     * \sa #RGB(r,g,b)
     */
    void SetForeground( unsigned short color = COLOR_WHITE );
    
    /** Set the background color for painting.
     *
     * This is the default color to be used for "empty" pixels while painting.
     * If a particular function allows for a different value to be specified
     * when the function is called, the new value will be used only for this
     * single call and will not change this setting.
     *
     * \param color The background color as RGB-565 value.
     * \sa #RGB(r,g,b)
     */
    void SetBackground( unsigned short color = COLOR_BLACK );
    
    /** Sets the font to be used for painting of text on the screen.
     * \param font A pointer to the font data.
     * \sa Comments in file fonts.h
     */
    void SetFont( const char* font );
    
    /** Fills the whole screen with a single color.
     * \param color The color to be used. The value must be in RGB-565 format.
     * \remarks The special values -1 and -2 signify the preset background and foreground colors, respectively.
     *          The backround color is the default.
     */
    void FillScreen( int color = -1 );
    
    /** Clears the screen.
     *
     * This is the same as calling #FillScreen() or #FillScreen( -1 ) to use the background color.
     */
    void ClearScreen( void );
    
    /** Draws a pixel at the specified location.
     *
     * By default the function will use the preset foreground color, but the background
     * or a custom color could be used as well.
     *
     * \param x The horizontal offset of the pixel from the upper left corner of the screen.
     * \param y The vertical offset of the pixel from the upper left corner of the screen.
     * \param color The color to be used. Must be in RGB-565 format, or -1 for background and -2 for foreground color.
     */
    void DrawPixel( unsigned short x, unsigned short y, int color = -2 );
    
    /** Draws a line.
     *
     * \param x1 Horizontal offset of the beginning point of the line.
     * \param y1 Vertical offset of the beginning point of the line.
     * \param x2 Horizontal offset of the end point of the line.
     * \param y2 Verical offset of the end point of the line.
     * \param color The color to use for painting, in RGB-565 format, or -1 for background, or -2 for foreground.
     */
    void DrawLine( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2, int color = -2 );
    
    /** Paints a rectangle.
     *
     * \param x1 The horizontal offset of the beginning point of one of the rectangle's diagonals.
     * \param y1 The vertical offset of the beginning point of one of the rectangle's diagonals.
     * \param x2 The horizontal offset of the end point of the same of the rectangle's diagonals.
     * \param y2 The vertical offset of the end point of the same of the rectangle's diagonals.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void DrawRect( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2, int color = -2 );
    
    /** Paints a rectangle and fills it with the paint color.
     *
     * \param x1 The horizontal offset of the beginning point of one of the rectangle's diagonals.
     * \param y1 The vertical offset of the beginning point of one of the rectangle's diagonals.
     * \param x2 The horizontal offset of the end point of the same of the rectangle's diagonals.
     * \param y2 The vertical offset of the end point of the same of the rectangle's diagonals.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void DrawRoundRect( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2, int color = -2 );
    
    /** Paints a rectangle with rounded corners.
     *
     * \param x1 The horizontal offset of the beginning point of one of the rectangle's diagonals.
     * \param y1 The vertical offset of the beginning point of one of the rectangle's diagonals.
     * \param x2 The horizontal offset of the end point of the same of the rectangle's diagonals.
     * \param y2 The vertical offset of the end point of the same of the rectangle's diagonals.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void FillRect( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2, int color = -2 );
    
    /** Paints a rectangle with rounded corners and fills it with the paint color.
     *
     * \param x1 The horizontal offset of the beginning point of one of the rectangle's diagonals.
     * \param y1 The vertical offset of the beginning point of one of the rectangle's diagonals.
     * \param x2 The horizontal offset of the end point of the same of the rectangle's diagonals.
     * \param y2 The vertical offset of the end point of the same of the rectangle's diagonals.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void FillRoundRect( unsigned short x1, unsigned short y1, unsigned short x2, unsigned short y2, int color = -2 );
    
    /** Paints a circle.
     *
     * \param x The offset of the circle's center from the left edge of the screen.
     * \param y The offset of the circle's center from the top edge of the screen.
     * \param radius The circle's radius.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void DrawCircle( unsigned short x, unsigned short y, unsigned short radius, int color = -2 );
    
    /** Paints a circle and fills it with the paint color.
     *
     * \param x The offset of the circle's center from the left edge of the screen.
     * \param y The offset of the circle's center from the top edge of the screen.
     * \param radius The circle's radius.
     * \param color The color to use for painting. -1 indicated background, -2 foreground, or custom color in RGB-565 format.
     */
    void FillCircle( unsigned short x, unsigned short y, unsigned short radius, int color = -2 );
    
    /** Print a text on the screen.
     *
     * \param str The text.
     * \param x The horizontal offset form the left edge of the screen. The special values LEFT, CENTER,
     *          or RIGHT can be used instead of pixel offset to indicate the text's horizontal alignment.
     * \param y The vertical offset of the text from the top of the screen.
     * \param fgColor The foreground to use for painting the text; -1 indicates background color, -2 the foreground setting, or custom color.
     * \param bgColor The color to use for painting the empty pixels; -1 indicates the background color, -2 the foreground setting, or custom color.
     * \param deg If different than 0, the text will be rotated at an angle this many degrees around its starting point. Default is not to ratate.
     */
    void Print( const char *str, unsigned short x, unsigned short y, int fgColor = -2, int bgColor = -1, unsigned short deg = 0 );
    
    /** Draw an image on the screen.
     *
     * The pixels of the picture must be in the RGB-565 format.  The data can be provided
     * as an array in a source or a header file.  To convert an image file to the appropriate
     * format, a special utility must be utilized.  One such tool is provided by Henning Karlsen,
     * the author of the UTFT display liberary and can be downloaded for free from his web site:
     * http://henningkarlsen.com/electronics/library.php?id=52
     *
     * \param x Horizontal offset of the first pixel of the image.
     * \param y Vertical offset of the first pixel of the image
     * \param sx Width of the image.
     * \param sy Height of the image.
     * \param data Image pixel array.
     * \param scale A value of 1 will produce an image with its original size, while a different value will scale the image.
     */
    void DrawBitmap( unsigned short x, unsigned short y, unsigned short sx, unsigned short sy, bitmap_t data, unsigned char scale = 1 );
    
    /** Draw an image on the screen.
     *
     * The pixels of the picture must be in the RGB-565 format.  The data can be provided
     * as an array in a source or a header file.  To convert an image file to the appropriate
     * format, a special utility must be utilized.  One such tool is provided by Henning Karlsen,
     * the author of the UTFT display liberary and can be downloaded for free from his web site:
     * http://henningkarlsen.com/electronics/library.php?id=52
     *
     * \param x Horizontal offset of the first pixel of the image.
     * \param y Vertical offset of the first pixel of the image
     * \param sx Width of the image.
     * \param sy Height of the image.
     * \param data Image pixel array.
     * \param deg Angle to rotate the image before painting on screen, in degrees.
     * \param rox
     * \param roy
     */
    void DrawBitmap( unsigned short x, unsigned short y, unsigned short sx, unsigned short sy, bitmap_t data, unsigned short deg, unsigned short rox, unsigned short roy );
    
private:
    DigitalOut _lcd_pin_cs, _lcd_pin_reset, _lcd_pin_rs, _lcd_pin_wr;
    BusOut* _lcd_port;
    DigitalOut* _lcd_pin_rd;
    orientation_t _orientation;
    static const long _disp_width = 239;
    static const long _disp_height = 319;
    unsigned short _foreground, _background;
    font_metrics_t _font;

private:
    virtual void writeCmd( unsigned short cmd );
    virtual void writeData( unsigned short data );
    virtual void writeCmdData( unsigned short cmd, unsigned short data );
    
    void setXY( uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2 );
    void clearXY( void );
    void drawHLine( unsigned short x, unsigned short y, unsigned short len, int color = -2 );
    void drawVLine( unsigned short x, unsigned short y, unsigned short len, int color = -2 );

    void printChar( char c, unsigned short x, unsigned short y, int fgColor = -2, int bgColor = -1 );
    void rotateChar( char c, unsigned short x, unsigned short y, int pos, int fgColor = -2, int bgColor = -1, unsigned short deg = 0 );
};

#endif /* SSD1289_H */
