/* mbed library for 240*320 pixel display TFT based on ILI9341 LCD Controller
 * Copyright (c) 2013 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
 //           PD Change the char position handling, use pixel (x,y) instead of (colum, row)
 //  30.03.14 WH Added some methods & defines, Fixed typos & warnings, General define for SPI_16 selection
 //

#ifndef MBED_SPI_TFT_ILI9341_H
#define MBED_SPI_TFT_ILI9341_H

#include "mbed.h"
#include "GraphicsDisplay.h"

/* Enable or disable 16 bit SPI communication */
#if defined TARGET_KL25Z 
//Always disable for KL25Z since it does not support 16 bit SPI.
#define SPI_16               0
#else
//Disable anyhow
//#define SPI_16               0

//Enable 16 bit SPI. Does improve performance for fill operations.
#define SPI_16               1
#endif

/*Enable characters with transparant background color */
#define TRANSPARANCY         1

/* Default Display Dimensions */
#define TFT_WIDTH          240
#define TFT_HEIGHT         320
/* Default Bits per pixel */
#define TFT_BPP             16

/** @def Compute RGB color in 565 format                                            */
#define RGB(r,g,b)  (((r&0xF8)<<8)|((g&0xFC)<<3)|((b&0xF8)>>3)) //5 red | 6 green | 5 blue

/** @def swap(type, a, b)
 *  @brief Convenience macro to swap two values.
 */
#define swap(type, a, b)      { type tmp = ( a ); ( a ) = ( b ); ( b ) = tmp; }
 
//#define POLY_Y(Z)          ((int32_t)((Points + Z)->X))
//#define POLY_X(Z)          ((int32_t)((Points + Z)->Y))
//
//#define max(a,b) (((a)>(b))?(a):(b))
//#define min(a,b) (((a)<(b))?(a):(b))
//#define ABS(X)  ((X) > 0 ? (X) : -(X))  

/* Some RGB color definitions in 565 format and equivalent 888 format               */
#define Black           0x0000      /*   0,   0,   0 */
#define Navy            0x000F      /*   0,   0, 128 */
#define DarkGreen       0x03E0      /*   0, 128,   0 */
#define DarkCyan        0x03EF      /*   0, 128, 128 */
#define Maroon          0x7800      /* 128,   0,   0 */
#define Purple          0x780F      /* 128,   0, 128 */
#define Olive           0x7BE0      /* 128, 128,   0 */
#define LightGrey       0xC618      /* 192, 192, 192 */
#define Grey            0xF7DE
#define DarkGrey        0x7BEF      /* 128, 128, 128 */
#define Blue            0x001F      /*   0,   0, 255 */
#define Blue2           0x051F
#define Green           0x07E0      /*   0, 255,   0 */
#define Cyan            0x07FF      /*   0, 255, 255 */
#define Cyan2           0x7FFF
#define Red             0xF800      /* 255,   0,   0 */
#define Magenta         0xF81F      /* 255,   0, 255 */
#define Yellow          0xFFE0      /* 255, 255,   0 */
#define White           0xFFFF      /* 255, 255, 255 */
#define Orange          0xFD20      /* 255, 165,   0 */
#define Orange2         0x051F
#define GreenYellow     0xAFE5      /* 173, 255,  47 */


/** 
  * @brief ILI9341 Registers  
  */ 
#define ILI9341_DISPLAY_RST          0x01   /* SW reset */
#define READ_DISPLAY_PIXEL_FORMAT    0x0C

#define ILI9341_SLEEP_OUT            0x11   /* Sleep out register */
#define ILI9341_PARTIAL_MODE         0x12   /* Partial Mode register */
#define ILI9341_NORMAL_MODE          0x13   /* Normal Mode register */
#define ILI9341_DISPLAY_INVERT_OFF   0x21   /* Display inversion off register */
#define ILI9341_DISPLAY_INVERT_ON    0x22   /* Display inversion on register */
#define ILI9341_GAMMA                0x26   /* Gamma register */
#define ILI9341_DISPLAY_OFF          0x28   /* Display off register */
#define ILI9341_DISPLAY_ON           0x29   /* Display on register */
#define ILI9341_COLUMN_ADDR          0x2A   /* Colomn address register */ 
#define ILI9341_PAGE_ADDR            0x2B   /* Page address register */ 
#define ILI9341_GRAM                 0x2C   /* GRAM register */   
#define READ_MEMORY                  0x2E
//
//
#define ILI9341_TEAR_OFF             0x34   /* tearing effect off */
#define ILI9341_TEAR_ON              0x35   /* tearing effect on */
#define ILI9341_MAC                  0x36   /* Memory Access Control register*/
#define ILI9341_PIXEL_FORMAT         0x3A   /* Pixel Format register */
#define READ_MEMORY_CONTINUE         0x3E

#define ILI9341_WDB                  0x51   /* Write Brightness Display register */
#define ILI9341_WCD                  0x53   /* Write Control Display register*/
#define ILI9341_RGB_INTERFACE        0xB0   /* RGB Interface Signal Control */
#define ILI9341_FRC                  0xB1   /* Frame Rate Control register */
#define ILI9341_BPC                  0xB5   /* Blanking Porch Control register*/
#define ILI9341_DFC                  0xB6   /* Display Function Control register*/
#define ILI9341_ENTRY_MODE           0xB7   /* Display Entry mode register*/
//
#define ILI9341_POWER1               0xC0   /* Power Control 1 register */
#define ILI9341_POWER2               0xC1   /* Power Control 2 register */
#define ILI9341_VCOM1                0xC5   /* VCOM Control 1 register */
#define ILI9341_VCOM2                0xC7   /* VCOM Control 2 register */
#define ILI9341_POWERA               0xCB   /* Power control A register */
#define ILI9341_POWERB               0xCF   /* Power control B register */
#define ILI9341_PGAMMA               0xE0   /* Positive Gamma Correction register*/
#define ILI9341_NGAMMA               0xE1   /* Negative Gamma Correction register*/
#define ILI9341_DTCA                 0xE8   /* Driver timing control A */
#define ILI9341_DTCB                 0xEA   /* Driver timing control B */
#define ILI9341_POWER_SEQ            0xED   /* Power on sequence register */
#define UNDOCUMENTED_0xEF            0xEF   // !@#$
#define ILI9341_3GAMMA_EN            0xF2   /* 3 Gamma enable register */
#define ILI9341_INTERFACE            0xF6   /* Interface control register */
#define ILI9341_PRC                  0xF7   /* Pump ratio control register */


/** Display control class, based on GraphicsDisplay and TextDisplay
 *
 * Example:
 * @code
 * #include "stdio.h"
 * #include "mbed.h"
 * #include "SPI_TFT_ILI9341.h"
 * #include "string"
 * #include "Arial12x12.h"
 * #include "Arial24x23.h"
 * 
 *
 *
 * // the TFT is connected to SPI pin 5-7 and IO's 8-10
 * SPI_TFT_ILI9341 TFT(p5, p6, p7, p8, p9, p10,"TFT"); // mosi, miso, sclk, cs, reset, dc
 * If your display need a signal for switch the backlight use a aditional IO pin in your program 
 *
 * int main() {
 *     TFT.claim(stdout);      // send stdout to the TFT display 
 *     //TFT.claim(stderr);      // send stderr to the TFT display
 *
 *     TFT.background(Black);    // set background to black
 *     TFT.foreground(White);    // set chars to white
 *     TFT.cls();                // clear the screen
 *     TFT.set_font((unsigned char*) Arial12x12);  // select the font
 *     
 *     TFT.set_orientation(0);
 *     printf("  Hello Mbed 0");
 *     TFT.set_font((unsigned char*) Arial24x23);  // select font 2
 *     TFT.locate(48,115);
 *     TFT.printf("Bigger Font");
 *  }
 * @endcode
 */
 class SPI_TFT_ILI9341 : public GraphicsDisplay {
 public:

   /** Display origin */
   enum Origin {
     Origin_LeftTop=0,   /**<  Left Top of panel is origin */    
     Origin_RightTop,    /**<  Right Top of panel is origin */
     Origin_RightBot,    /**<  Right Bottom of panel is origin */
     Origin_LeftBot      /**<  Left Bottom panel is origin */
   };

  /** Create a SPI_TFT object connected to SPI and three pins
   *
   * @param mosi pin connected to SDO of display
   * @param miso pin connected to SDI of display
   * @param sclk pin connected to RS of display 
   * @param cs pin connected to CS of display
   * @param reset pin connected to RESET of display (may be NC)
   * @param dc pin connected to WR of display
   * The IM pins have to be set to 1110 (3-0). Note: the M24SR board uses 0110 which also works.  
   */ 
  SPI_TFT_ILI9341(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName reset, PinName dc, const char* name ="TFT");


  /** Destructor for SPI_TFT object
   * @param  none
   * @return none
   */ 
  virtual ~SPI_TFT_ILI9341();
     

  /** Get the width of the screen in pixels
   *
   * @returns width of screen in pixels
   *
   */    
  virtual int width();

  /** Get the height of the screen in pixels
   *
   * @returns height of screen in pixels 
   *
   */     
  virtual int height();
    
  /** Draw a pixel at x,y with color 
   *  
   * @param x horizontal position
   * @param y vertical position
   * @param color 16 bit pixel color
   */    
  virtual void pixel(int x, int y, int colour);
    
  /** draw a circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param color 16 bit color                                                                 *
   *
   */    
  void circle(int x, int y, int r, int colour); 
  
  /** draw a filled circle
   *
   * @param x0,y0 center
   * @param r radius
   * @param color 16 bit color                                                                 *
   */    
  void fillcircle(int x, int y, int r, int colour); 
 
 
  /** draw an oval
   *
   * @param x,y center
   * @param b radius
   * @param aspect hor/ver ratio ( hor. oval < 1.0; circle = 1.0; vert. oval > 1.0 )  
   * @param color 16 bit color                                                                 *
   *
   */    
  void oval(int x, int y, int b, int color, float aspect=1.0);


  /** draw a filled oval
   *
   * @param x,y center
   * @param b radius
   * @param aspect hor/ver ratio ( hor. oval < 1.0; circle = 1.0; vert. oval > 1.0 )  
   * @param color 16 bit color                                                                 *
   *
   */    
  void filloval(int x, int y, int b, int color, float aspect=1.0);
    
  /** draw a 1 pixel line
   *
   * @param x0,y0 start point
   * @param x1,y1 stop point
   * @param color 16 bit color
   *
   */    
  void line(int x0, int y0, int x1, int y1, int colour);
    
  /** draw a rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *                                                   *
   */    
  void rect(int x0, int y0, int x1, int y1, int colour);
    
  /** draw a filled rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *
   */    
  void fillrect(int x0, int y0, int x1, int y1, int colour);



  /** draw a rounded rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *                                                   *
   */    
  void roundrect( int x0, int y0, int x1, int y1, int color );


  /** draw a filled rounded rect
   *
   * @param x0,y0 top left corner
   * @param x1,y1 down right corner
   * @param color 16 bit color
   *
   */    
  void fillroundrect( int x0, int y0, int x1, int y1, int color );
    
  /** setup cursor position
   *
   * @param x x-position (top left)
   * @param y y-position 
   */   
  virtual void locate(int x, int y);
    
 /** Fill the screen with _background color
   *  @param none
   *  @return none
   */    
  virtual void cls();   
      

 /** Fill the screen with _background color
   *  @param none
   *  @return none
   */    
  virtual void newcls();   



  /** calculate the max number of char in a line
   *
   * @returns max columns
   * depends on actual font size
   *
   */    
  virtual int columns(void);
    
  /** calculate the max number of rows
   *
   * @returns max row
   * depends on actual font size
   *
   */   
  virtual int rows(void);
    
  /** put a char on the screen
   *
   * @param value char to print
   * @returns printed char
   *
   */
  virtual int _putc(int value);
    
  /** draw a character on given position out of the active font to the TFT
   *
   * @param x x-position of char (top left) 
   * @param y y-position
   * @param c char to print
   *
   */    
  virtual void character(int x, int y, int c);
    
  /** paint a bitmap on the TFT 
   *
   * @param x,y : upper left corner 
   * @param w width of bitmap
   * @param h height of bitmap
   * @param *bitmap pointer to the bitmap data
   *
   *   bitmap format: 16 bit R5 G6 B5
   * 
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5            
   *   use winhex to load this file and mark data stating at offset 0x46 to end
   *   use edit -> copy block -> C Source to export C array
   *   paste this array into your program
   * 
   *   define the array as static const unsigned char to put it into flash memory
   *   cast the pointer to (unsigned char *) :
   *   tft.Bitmap(10,40,309,50,(unsigned char *)scala);
   */    
  void Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h, unsigned char *bitmap);
    
    
   /** paint a 16 bit BMP from filesytem on the TFT (slow) 
   *
   * @param x,y : position of upper left corner 
   * @param *Name_BMP name of the BMP file with drive: "/local/test.bmp"
   *
   * @returns 1 if bmp file was found and painted
   * @returns  0 if bmp file was found not found
   * @returns -1 if file is no bmp
   * @returns -2 if bmp file is no 16 bit bmp
   * @returns -3 if bmp file is to big for screen 
   * @returns -4 if buffer malloc go wrong
   *
   *   bitmap format: 16 bit R5 G6 B5
   * 
   *   use Gimp to create / load , save as BMP, option 16 bit R5 G6 B5
   *   copy to internal file system or SD card           
   */      
    
  int BMP_16(unsigned int x, unsigned int y, const char *Name_BMP);  
    


/*******************************************************************************
* Function Name  : WriteBMP_FAT
* @brief Displays a bitmap picture loaded in Flash.
* @param Xpos: specifies the X position.
* @param Ypos: specifies the Y position.
* @param BmpAddress: Bmp picture address in Flash.
* @return None
*******************************************************************************/
  void WriteBMP_FAT(uint16_t Xpos, uint16_t Ypos, const char* BmpName);    
 
    
  /** select the font to use
   *
   * @param f pointer to font array 
   *                                                                              
   *   font array can created with GLCD Font Creator from http://www.mikroe.com
   *   you have to add 4 parameter at the beginning of the font array to use: 
   *   - the number of byte / char
   *   - the vertial size in pixel
   *   - the horizontal size in pixel
   *   - the number of byte per vertical line
   *   you also have to change the array to char[]
   *
   */  
  void set_font(unsigned char* f);
   
  /** Set the orientation of the screen
   *  x,y: 0,0 is always top left 
   *
   * @param o direction to use the screen (0-3)  
   *
   */  
//WH  void set_orientation(unsigned int o);


  /** Set the origin of the screen
   *  x,y: 0,0 is always top left 
   *
   * @param origin top left corner of the screen 
   * @return None   
   */  
  void set_origin(Origin origin);

  
  /** Set background transparancy for characters, meaning that background pixels remain unchanged
   *
   * @param state transparancy on/off  
   * @return None      
   */  
   void set_transparancy(bool state);
  
  /** Enable the ILI9341 display 
   *
   * @param on: display On/Off 
   * @return None      
   */    
  void tft_on(bool on);
  
  /** read out the manufacturer ID of the LCD
   *  can used for checking the connection to the display
   *  @returns ID
   */ 
  int Read_ID(void);
  
   
protected:

  /** draw a horizontal line
   *
   * @param x0 horizontal start
   * @param x1 horizontal stop
   * @param y vertical position
   * @param color 16 bit color 
   * @return None                                                       
   */
  void hline(int x0, int x1, int y, int colour);
    
  /** draw a vertical line
   *
   * @param x horizontal position
   * @param y0 vertical start 
   * @param y1 vertical stop
   * @param color 16 bit color
   * @return None         
   */
  void vline(int x, int y0, int y1, int colour);
    
  /** Set draw window region
   *
   * @param x horizontal position
   * @param y vertical position
   * @param w window width in pixel
   * @param h window height in pixels
   * @return None         
   */    
  virtual void window (unsigned int x,unsigned int y, unsigned int w, unsigned int h);
    
  /** Set draw window region to whole screen
   *
   * @return None         
   */  
  void window_max (void);
 
    
  /** Init the ILI9341 controller 
   *
   * @return None         
   */    
  void tft_reset();
  
  
  /** HW Reset to ILI9341 controller
   * @return None        
   */
  void _hwreset();
  
    
   /** Write data to the LCD controller
   *
   * @param dat data written to LCD controller
   * @return None         
   */   
  //void wr_dat(unsigned int value);
  void wr_dat(unsigned char value);
    
  /** Write a command the LCD controller 
   *
   * @param cmd: command to be written   
   * @return None         
   */   
  void wr_cmd(unsigned char value);
    
  /** Read byte from the LCD controller
   *
   * @param cmd command to controller
   * @returns data from LCD controller
   */    
   char rd_byte(unsigned char cmd);
    
  
  /** Read int from the LCD controller
   *
   * @param cmd command to controller
   * @returns data from LCD controller
   */    
   int rd_32(unsigned char cmd);  
    
    
  /** Write a value to the to a LCD register
   *
   * @param reg register to be written
   * @param val data to be written
   * @return None         
   */   
  //void wr_reg (unsigned char reg, unsigned char val);
    
  /** Read a LCD register
   *
   * @param reg register to be read
   * @returns value of the register 
   */    
  //unsigned short rd_reg (unsigned char reg);
    

  SPI _spi;
  DigitalOut _cs; 
  DigitalOut* _reset;
  DigitalOut _dc;

  unsigned char* _font;  
  unsigned int   _char_x;
  unsigned int   _char_y;
  bool           _transparancy;

  //WH unsigned int orientation;
  Origin _origin;  
      
};

#endif
