/** 
 * @file SDA5708.h 
 * @brief  mbed SDA5708 LED matrix display Library. 
 * @author WH
 * @date   Copyright (c) 2014
 *         v01: WH, Initial release
 *         v02: WH, Added setMode() and UDC support 
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#ifndef _SDA5708_H
#define _SDA5708_H

#include "mbed.h"

//Enable Stream printf or minimalistic printf
#define SDA5708_PRINTF 1

// display brightness definitions, indicating percentage brightness
#define SDA5708_BRIGHT_100         0x00
#define SDA5708_BRIGHT_53          0x01
#define SDA5708_BRIGHT_40          0x02
#define SDA5708_BRIGHT_27          0x03
#define SDA5708_BRIGHT_20          0x04
#define SDA5708_BRIGHT_13          0x05
#define SDA5708_BRIGHT_6_6         0x06
#define SDA5708_BRIGHT_0           0x07

#define SDA5708_MAX_PEAK           0x00
#define SDA5708_12_5_PEAK          0x08

// default display brightness
#define SDA5708_DEF_BRIGHT         SDA5708_BRIGHT_27

//Commands
//Clear display. D2..D0 select brightness. D3 selects max peak current
#define SDA5708_CMD_CLR            0xC0
//Normal display. D2..D0 select brightness. D3 selects max peak current
#define SDA5708_CMD_NORMAL         0xE0
//Select Digit Address. D2..D0 select digit. Digit 0 is leftmost, Digit 7 is rightmost.
#define SDA5708_CMD_DIG_ADDR       0xA0 
//Write Digit row data as 8 sequential bytes. 
// First byte defines top row. D7..D5 are always 0, D4..D0 define bitpattern. D4 is leftmost, D0 is rightmost pixel.
//

extern const char udc_0[];       /* |> */
extern const char udc_1[];       /* <| */
extern const char udc_2[];       /* | */
extern const char udc_3[];       /* || */
extern const char udc_4[];       /* ||| */
extern const char udc_5[];       /* = */
extern const char udc_6[];       /* checkerboard */
extern const char udc_7[];       /* \ */

extern const char udc_Bat_Hi[];  /* Battery Full */
extern const char udc_Bat_Ha[];  /* Battery Half */
extern const char udc_Bat_Lo[];  /* Battery Low */
extern const char udc_AC[];      /* AC Power */
extern const char udc_smiley[];  /* Smiley */

/** A library for driving SDA5708 LED matrix displays
 *
 * @code
 * #include "mbed.h"
 * #include "SDA5708.h"
 *
 * // SPI Communication
 * SPI spi_led(p5, NC, p7); // MOSI, MISO, SCLK
 *
 * //Display
 * SDA5708 led(&spi_led, p8, p9); // SPI bus, CS pin, RST pin
 *
 * int main() {
 *   int cnt;
 *
 *   led.locate(0, 0);                               
 *
 *   //          12345678 
 *   led.printf("Hi mbed ");  
 *   wait(2);
 * 
 *   cnt=0x20;  
 *   while(1) {
 *     wait(0.5);
 *                       
 *     led.putc(cnt);
 *     cnt++;
 *     if (cnt == 0x80) cnt=0x20;
 *   } 
 * }
 * @endcode
 */


/** Create an SDA5708 Display object connected to the proper pins
 *
 * @param  *spi SPI port
 * @param  cs   PinName for Chip Select (active low)
 * @param  rst  PinName for Reset (active low)
*/
class SDA5708 : public Stream {  
//class SDA5708 {
    
//Unfortunately this #define selection breaks Doxygen !!!
//#if (SDA5708_PRINTF == 1)
//class SDA5708 : public Stream {  
//#else    
//class SDA5708 {
//#endif

public:

    /** Display control */
    enum DisplayMode {
        DispOff,  /**<  Display Off */    
        DispOn    /**<  Display On */            
    };

   /** Create an SDA5708 Display object connected to the proper pins
     *
     * @param  *spi SPI port
     * @param  cs   PinName for Chip Select (active low)
     * @param  rst  PinName for Reset (active low)
     */
   SDA5708(SPI *spi, PinName cs, PinName rst);
  
#if (SDA5708_PRINTF != 1)
   /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
   int putc(int c);

    /** Write a raw string to the Display
     *
     * @param string text, may be followed by variables to emulate formatting the string.
     *                     However, when (SDA5708_PRINTF != 1) then printf formatting is NOT supported and variables will be ignored!           
     */
    int printf(const char* text, ...);
#else    
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
    
#endif    

    /** Clear the screen and locate to 0,0
     */
    void cls();                              

    /** Locate cursor to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column=0, int row=0);

    /** Set the Displaymode
     *
     * @param displayMode The Display mode (DispOff, DispOn)
     */
    void setMode(DisplayMode displayMode);     

    /** Set Brightness
     *
     * @param brightness The brightness level (valid range 0..7)
     */
    void setBrightness(uint8_t brightness);  
    
    /** Set User Defined Characters (UDC)
     *
     * @param unsigned char c   The Index of the UDC (0..7)
     * @param char *udc_data    The bitpatterns for the UDC (7 bytes of 5 significant bits for bitpattern)       
     */
    void setUDC(unsigned char c, char *udc_data);   
    
protected:
    /** Low level Reset method for controller
      */  
    void _reset();

   /** Low level Init method for LCD controller
     */  
    void _init(); 

   /** Low level command byte write operation.
     */
    void _write(uint8_t data);

    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

//    int _row;
    int _column;
    int _peak, _brightness;

    // Local UDC memory
    char _udc[8][7]; 
    
    //SPI bus
    SPI *_spi;
    DigitalOut _cs;         

    DigitalOut _rst;             
};


#endif
