/* SAA1064 - I2C LED Driver used in multiplex mode (4x 7 Segments and Decimal Point)
 * Copyright (c) 2013 Wim Huiskamp
 *
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * version 0.2 Initial Release
*/
#ifndef _SAA1064_H
#define _SAA1064_H

/** Driver for SAA1064 I2C 4-Digit 7-Segment LED Driver
 *
 * @code
 * #include "mbed.h"
 * #include "SAA1064.h"
 * 
 * // I2C Communication
 * I2C i2c_lcd(p28,p27); // SDA, SCL for LPC1768
 * //I2C i2c_lcd(P0_10,P0_11); // SDA, SCL for LPC812
 *
 * SAA1064 LED(&i2c_lcd); // I2C bus, Default SAA1064 Slaveaddress
 * 
 * int main() {
 *   uint8_t count = 0; 
 *
 *   // Display 0, 1, 2, 3
 *   LED.write(SAA1064_SEGM[0], SAA1064_SEGM[1], SAA1064_SEGM[2], SAA1064_SEGM[3]);
 *   wait(1);    
 *   
 *   while(1) {
 *     wait(0.3);    
 *     count++;       
 *   
 *     LED.writeInt(-150 + count, 3, false);  // Display value, dont suppress leading zero's
 *   }
 *  
 * }
 * @endcode
 */


//Address Defines for SAA1064
#define SAA1064_SA0 0x70
#define SAA1064_SA1 0x72
#define SAA1064_SA2 0x74
#define SAA1064_SA3 0x76

//Register Defines for SAA1064
#define SAA1064_CTRL 0x00
#define SAA1064_DIG1 0x01
#define SAA1064_DIG2 0x02
#define SAA1064_DIG3 0x03
#define SAA1064_DIG4 0x04

//Control Register Defines for SAA1064
//Static display (2 digits) or Multiplexed (4 digits)
#define SAA1064_MPX  0x01
//Digits 1 and 2 On
#define SAA1064_B0   0x02
//Digits 3 and 4 On
#define SAA1064_B1   0x04
//Intensity of display
#define SAA1064_INT0 0x00
#define SAA1064_INT1 0x10
#define SAA1064_INT2 0x20
#define SAA1064_INT3 0x30
#define SAA1064_INT4 0x40
#define SAA1064_INT5 0x50
#define SAA1064_INT6 0x60
#define SAA1064_INT7 0x70

//Default Mode: Multiplex On, All Digits On
#define SAA1064_CTRL_DEF (SAA1064_MPX | SAA1064_B0 | SAA1064_B1)


//Pin Defines for SAA1064
#define D_L0                 0x01
#define D_L1                 0x02
#define D_L2                 0x04
#define D_L3                 0x08
#define D_L4                 0x10
#define D_L5                 0x20
#define D_L6                 0x40
#define D_L7                 0x80

//Defines for Segments
const uint8_t SAA1064_SEGM[] = {0x3F,  //0
                                0x06,  //1
                                0x5B,  //2
                                0x4F,  //3
                                0x66,  //4
                                0x6D,  //5
                                0x7D,  //6
                                0x07,  //7
                                0x7F,  //8
                                0x6F,  //9
                                0x77,  //A
                                0x7C,  //B
                                0x39,  //C
                                0x5E,  //D
                                0x79,  //E
                                0x71}; //F

#define SAA1064_DP              0x80   //Decimal Point
#define SAA1064_MINUS           0x40   //Minus Sign
#define SAA1064_BLNK            0x00   //Blank Digit
#define SAA1064_ALL             0xFF   //All Segments On
               



/** Create an SAA1064 object connected to the specified I2C bus and deviceAddress
 *
*/
class SAA1064 {
public:
  /** Create a SAA1064 LED displaydriver object using a specified I2C bus and slaveaddress
   *
   * @param I2C &i2c the I2C port to connect to 
   * @param char deviceAddress the address of the SAA1064
  */  
  SAA1064(I2C *i2c, uint8_t deviceAddress = SAA1064_SA0);
  
  /** Set segment brightness
   *
   * @param intensity       intensity value, valid Range between 0-7, 0 = 0 mA/segment, 1 = 3 mA/segment etc 
  */  
  void setIntensity(uint8_t intensity);


/** Write digits 
  *
  * @param digit1  LED segment pattern for digit1 (MSB) 
  * @param digit2  LED segment pattern for digit2 
  * @param digit3  LED segment pattern for digit3
  * @param digit4  LED segment pattern for digit4 (LSB)       
  */  
  void write(uint8_t digit1, uint8_t digit2, uint8_t digit3, uint8_t digit4);        

/** Write Integer 
  *
  * @param value     integer value to display, valid range -999...9999 
  * @param dp_digit  digit where decimal point is set, valid range 1..4 (no DP shown for dp_digit = 0)
  * @param leading   suppress leading zero (false=show leading zero, true=suppress leading zero)
  */  
  void writeInt(int value, uint8_t dp_digit=0, bool leading=true);


/** snake: show a short animation
  *
  * @param count     number of times animation is repeated, valid range 0..15 
  *
  */  
  void snake(uint8_t count);        

/** splash: show a short animation
  *
  * @param count     number of times animation is repeated, valid range 0..15 
  *
  */  
  void splash (uint8_t count);  
  
protected:
  I2C *_i2c;                    //I2C bus reference
  uint8_t _slaveAddress;        //I2C Slave address of device

/** Initialise LED driver 
  *
  */  
  void _init(); 
};

#endif