/* mbed PT6964 Library, for PT6964 LED controller
 * Copyright (c) 2015, v01: WH, Initial version
 *               2015, v02: WH, rename Digit/Grid 
 *               2016, v03: WH, updated Icon handling, UDCs and _putc()  
 *               2016, v04: WH, Refactored display and keyboard defines   
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6964_H
#define PT6964_H

#include "Font_7Seg.h"

/** An interface for driving PT6964 LED controller
 *
 * @code
 * #include "mbed.h"
 * #include "PT6964.h" 
 * 
 * DisplayData_t size is  8 bytes (4 grids @ 13 segments) OR 10 bytes (5 grids @ 12 segments) OR
 *                       12 bytes (6 grids @ 11 segments) OR 14 bytes (7 grids @ 10 segments) 
 * PT6964::DisplayData_t mbed_str = {0xDA,0x00, 0x7C,0x00, 0x3C,0x01, 0xF6,0x00, 0x00,0x00, 0x00,0x00, 0x00,0x00};  
 * PT6964::DisplayData_t all_str  = {0xFF,0x0F, 0xFF,0x0F, 0xFF,0x0F, 0xFF,0x0F, 0xFF,0x0F, 0xFF,0x0F, 0xFF,0x0F};  
 *
 * // KeyData_t size is 5 bytes  
 * PT6964::KeyData_t keydata; 
 *
 * // PT6964 declaration, Default setting 7 Grids @ 10 Segments
 * PT6964 PT6964(p5,p6,p7, p8);
 *
 * int main() {
 *   PT6964.cls(); 
 *   PT6964.writeData(all_str);
 *   wait(4);
 *   PT6964.writeData(mbed_str);    
 *   wait(1);
 *   PT6964.setBrightness(PT6964_BRT0);
 *   wait(1);
 *   PT6964.setBrightness(PT6964_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (PT6964.getKeys(&keydata)) {
 *      pc.printf("Keydata 0..4 = 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n", keydata[0], keydata[1], keydata[2], keydata[3], keydata[4]);
 *
 *      if (keydata[0] == 0x10) { //sw2   
 *        PT6964.cls(); 
 *        PT6964.writeData(all_str);
 *      }  
 *    } 
 *   }   
 * }
 * @endcode
 */

//PT6964 Display and Keymatrix data
#define PT6964_MAX_NR_GRIDS    7
#define PT6964_BYTES_PER_GRID  2
//Significant bits Keymatrix data
#define PT6964_KEY_MSK      0x1B 

//Memory size in bytes for Display and Keymatrix
#define PT6964_DISPLAY_MEM  (PT6964_MAX_NR_GRIDS * PT6964_BYTES_PER_GRID)
#define PT6964_KEY_MEM         5

//Reserved bits for commands
#define PT6964_CMD_MSK      0xC0

//Mode setting command
#define PT6964_MODE_SET_CMD 0x00
#define PT6964_GR4_SEG13    0x00
#define PT6964_GR5_SEG12    0x01
#define PT6964_GR6_SEG11    0x02
#define PT6964_GR7_SEG10    0x03 //default

//Data setting commands
#define PT6964_DATA_SET_CMD 0x40
#define PT6964_DATA_WR      0x00
#define PT6964_KEY_RD       0x02
#define PT6964_ADDR_INC     0x00
#define PT6964_ADDR_FIXED   0x04
#define PT6964_MODE_NORM    0x00
#define PT6964_MODE_TEST    0x08

//Address setting commands
#define PT6964_ADDR_SET_CMD 0xC0
#define PT6964_ADDR_MSK     0x0F

//Display control commands
#define PT6964_DSP_CTRL_CMD 0x80
#define PT6964_BRT_MSK      0x07
#define PT6964_BRT0         0x00 //Pulsewidth 1/16
#define PT6964_BRT1         0x01
#define PT6964_BRT2         0x02
#define PT6964_BRT3         0x03
#define PT6964_BRT4         0x04
#define PT6964_BRT5         0x05
#define PT6964_BRT6         0x06
#define PT6964_BRT7         0x07 //Pulsewidth 14/16

#define PT6964_BRT_DEF      PT6964_BRT3

#define PT6964_DSP_OFF      0x00
#define PT6964_DSP_ON       0x08


/** A class for driving Princeton PT6964 LED controller
 *  Note: the PT6964 is also available from other chipvendors eg AIP1628, HT1628, CM1628, SM1628
 *
 * @brief Supports 4 Grids @ 13 Segments or 5 Grids @ 12 Segments or 6 Grids @ 11 Segments or 7 Grids @ 10 Segments. 
 *        Also supports a scanned keyboard of upto 20 keys.
 *        SPI bus interface device. 
 */
class PT6964 {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid4_Seg13 = PT6964_GR4_SEG13,
    Grid5_Seg12 = PT6964_GR5_SEG12,   
    Grid6_Seg11 = PT6964_GR6_SEG11,
    Grid7_Seg10 = PT6964_GR7_SEG10
  };
  
  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[PT6964_DISPLAY_MEM];
  typedef char KeyData_t[PT6964_KEY_MEM];
    
 /** Constructor for class for driving Princeton PT6964 LED controller
  *
  * @brief Supports 4 Grids @ 13 segments or 5 Grids @ 12 segments or 6 Grids @ 11 Segments or 7 Grids @ 10 Segments. 
  *        Also supports a scanned keyboard of upto 20 keys.
  *        SPI bus interface device. 
  *
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  *  @param  Mode selects either Grids/Segments (default 7 Grids @ 10 Segments)
  */
  PT6964(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode=Grid7_Seg10);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to PT6964
   *  @param  int address display memory location to write byte
   *  @param  char data byte written at given address
   *  @return none
   */ 
  void writeData(int address, char data); 
 
 /** Write Display datablock to PT6964
   *  @param  DisplayData_t data Array of PT6964_DISPLAY_MEM (=14) bytes for displaydata (starting at address 0)
   *  @param  length number bytes to write (valid range 0..PT6964_DISPLAY_MEM (=14), starting at address 0)   
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = PT6964_DISPLAY_MEM);

  /** Read keydata block from PT6964
   *  @param  *keydata Ptr to Array of PT6964_KEY_MEM (=5) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the PT6964 key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = PT6964_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  char _display;
  char _bright; 
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The PT6964 expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write command and parameter to PT6964
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
  void _writeCmd(int cmd, int data);  
};



// Derived class for PT6964 used in DVD-538A front display unit
//

#define DVD538A_NR_GRIDS  5
#define DVD538A_NR_DIGITS 4
#define DVD538A_NR_UDC    8

/** Constructor for class for driving Princeton PT6964 controller as used in DVD538A
  *
  *  @brief Supports 4 Digits of 7 Segments, 1 Grid of 9 Icons. Also supports a scanned keyboard of 4 keys.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class PT6964_DVD538A : public PT6964, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  enum Icon {
    LD1  = (1<<24) | S7_LD1,
    LD2  = (1<<24) | S7_LD2,
    CD   = (1<<24) | S7_CD,
    DVD  = (1<<24) | S7_DVD,
    PSE  = (1<<24) | S7_PSE,
    PLY  = (1<<24) | S7_PLY,    
    COL2 = (1<<24) | S7_COL2,
    MP4  = (1<<24) | S7_MP4,
    MP3  = (1<<24) | S7_MP3    
  };
  
  typedef short UDCData_t[DVD538A_NR_UDC];
  
 /** Constructor for class for driving Princeton PT6964 VFD controller as used in DVD538A
   *
   * @brief Supports 4 Digits of 7 Segments, 1 Grid of 9 Icons. Also supports a scanned keyboard of 4 keys.
   *  
   * @param  PinName mosi, miso, sclk, cs SPI bus pins
   */
  PT6964_DVD538A(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);   

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write Display datablock to PT6964
    *  @param  DisplayData_t data Array of PT6964_DISPLAY_MEM (=14) bytes for displaydata (starting at address 0)
    *  @param  length number bytes to write (valid range 0..(DVD538A_NR_GRIDS*2) (=14), starting at address 0)   
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (DVD538A_NR_GRIDS*2)) {
      PT6964::writeData(data, length);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;
    int _columns;   
    
    DisplayData_t _displaybuffer;
    UDCData_t _UDC_7S; 
};


#endif