/* mbed PT6964 Library, for PT6964 LED controller
 * Copyright (c) 2015, v01: WH, Initial version
 *               2015, v02: WH, rename Digit/Grid 
 *               2016, v03: WH, updated Icon handling, UDCs and _putc() 
 *               2016, v04: WH, Refactored display and keyboard defines    
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "PT6964.h"
#include "Font_7Seg.h"

/** Constructor for class for driving PT6964 LED controller with SPI bus interface device. 
 *  Note: the PT6964 is also available from other chipvendors eg AIP1628, HT1628, CM1628, SM1628
 *  @brief Supports 4 digits @ 13 segments or 5 digits @ 12 segments or 6 Digits @ 11 Segments or 7 Digits @ 10 Segments. 
 *         Also supports a scanned keyboard of upto 20 keys.
 *   
 *  @param  PinName mosi, miso, sclk, cs SPI bus pins
 *  @param  Mode selects either 6 Digits of 12 Segments or 7 Digits of 10 Segments (default) 
*/
PT6964::PT6964(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode) : _spi(mosi,miso,sclk), _cs(cs), _mode(mode) {

  _init();
}

/** Init the SPI interface and the controller
  * @param  none
  * @return none
  */ 
void PT6964::_init(){
  
//init SPI
  _cs=1;
  _spi.format(8,3); //PT6964 uses mode 3 (Clock High on Idle, Data latched on second (=rising) edge)
  _spi.frequency(500000);   

//init controller  
  _writeCmd(PT6964_MODE_SET_CMD, _mode);                                               // Mode set command 

  _display = PT6964_DSP_ON;
  _bright  = PT6964_BRT_DEF; 
  _writeCmd(PT6964_DSP_CTRL_CMD, _display | _bright );                                 // Display control cmd, display on/off, brightness   
  
  _writeCmd(PT6964_DATA_SET_CMD, PT6964_DATA_WR | PT6964_ADDR_INC | PT6964_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}   


/** Clear the screen and locate to 0
 */  
void PT6964::cls() {
  
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6964_ADDR_SET_CMD | 0x00)); // Address set cmd, 0
      
  for (int cnt=0; cnt<PT6964_DISPLAY_MEM; cnt++) {
    _spi.write(0x00); // data 
  }
  
  wait_us(1);
  _cs=1;      
}  

/** Set Brightness
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void PT6964::setBrightness(char brightness){

  _bright = brightness & PT6964_BRT_MSK; // mask invalid bits
  
  _writeCmd(PT6964_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness  
}

/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void PT6964::setDisplay(bool on) {
  
  if (on) {
    _display = PT6964_DSP_ON;
  }
  else {
   _display = PT6964_DSP_OFF;
  }
  
  _writeCmd(PT6964_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness   
}

/** Write databyte to PT6964
  *  @param  int address display memory location to write byte
  *  @param  char data byte written at given address
  *  @return none
  */ 
void PT6964::writeData(int address, char data) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6964_ADDR_SET_CMD | (address & PT6964_ADDR_MSK))); // Set Address cmd
      
  _spi.write(_flip(data)); // data 
  
  wait_us(1);
  _cs=1;             
}

/** Write Display datablock to PT6964
  *  @param  DisplayData_t data Array of PT6964_DISPLAY_MEM (=14) bytes for displaydata (starting at address 0)
  *  @param  length number bytes to write (valide range 0..PT6964_DISPLAY_MEM (=14), starting at address 0)     
  *  @return none
  */ 
void PT6964::writeData(DisplayData_t data, int length) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6964_ADDR_SET_CMD | 0x00)); // Set Address at 0
      
// sanity check
  if (length < 0) {length = 0;}
  if (length > PT6964_DISPLAY_MEM) {length = PT6964_DISPLAY_MEM;}

//  for (int idx=0; idx<PT6964_DISPLAY_MEM; idx++) {  
  for (int idx=0; idx<length; idx++) {    
    _spi.write(_flip(data[idx])); // data 
  }
  
  wait_us(1);
  _cs=1;             
}


/** Read keydata block from PT6964
  *  @param  *keydata Ptr to Array of PT6964_KEY_MEM (=5) bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  *
  * Note: Due to the hardware configuration the PT6964 key matrix scanner will detect multiple keys pressed at same time,
  *       but this may also result in some spurious keys being set in keypress data array.
  *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.  
  */ 
bool PT6964::getKeys(KeyData_t *keydata) {
  int keypress = 0;
  char data;

  // Read keys
  _cs=0;
  wait_us(1);    
  
  // Enable Key Read mode
  _spi.write(_flip(PT6964_DATA_SET_CMD | PT6964_KEY_RD | PT6964_ADDR_INC | PT6964_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  for (int idx=0; idx < PT6964_KEY_MEM; idx++) {
    data = _flip(_spi.write(0xFF));    // read keys and correct bitorder

    data = data & PT6964_KEY_MSK; // Mask valid bits
    if (data != 0) {  // Check for any pressed key
      for (int bit=0; bit < 8; bit++) {
        if (data & (1 << bit)) {keypress++;} // Test all significant bits
      }
    }  

    (*keydata)[idx] = data;            // Store keydata after correcting bitorder
  }

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6964_DATA_SET_CMD, PT6964_DATA_WR | PT6964_ADDR_INC | PT6964_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
#if(1)
// Dismiss multiple keypresses at same time
  return (keypress == 1);    
#else
// Allow multiple keypress and accept possible spurious keys
  return (keypress > 0);
#endif  
}
    

/** Helper to reverse all command or databits. The PT6964 expects LSB first, whereas SPI is MSB first
  *  @param  char data
  *  @return bitreversed data
  */ 
char PT6964::_flip(char data) {
 char value=0;
  
 if (data & 0x01) {value |= 0x80;} ;  
 if (data & 0x02) {value |= 0x40;} ;
 if (data & 0x04) {value |= 0x20;} ;
 if (data & 0x08) {value |= 0x10;} ;
 if (data & 0x10) {value |= 0x08;} ;
 if (data & 0x20) {value |= 0x04;} ;
 if (data & 0x40) {value |= 0x02;} ;
 if (data & 0x80) {value |= 0x01;} ;
 return value;       
}


/** Write command and parameter to PT6964
  *  @param  int cmd Command byte
  *  &Param  int data Parameters for command
  *  @return none
  */  
void PT6964::_writeCmd(int cmd, int data){
    
  _cs=0;
  wait_us(1);    
//  _spi.write(_flip( (cmd & 0xF0) | (data & 0x0F)));  
  _spi.write(_flip( (cmd & PT6964_CMD_MSK) | (data & ~PT6964_CMD_MSK)));   
 
  wait_us(1);
  _cs=1;          
}  




/** Constructor for class for driving Princeton PT6964 controller as used in DVD538A
  *
  *  @brief Supports 4 Digits of 7 Segments, 1 Grid of 9 Icons. Also supports a scanned keyboard of 4.
  *   
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
PT6964_DVD538A::PT6964_DVD538A(PinName mosi, PinName miso, PinName sclk, PinName cs) : PT6964(mosi, miso, sclk, cs, Grid7_Seg10) {
  _column  = 0;
  _columns = DVD538A_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  */
void PT6964_DVD538A::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int PT6964_DVD538A::columns() {
    return _columns;
}


/** Clear the screen and locate to 0
  * @param bool clrAll Clear Icons also (default = false)
  */ 
void PT6964_DVD538A::cls(bool clrAll) {  

  if (clrAll) {
    //clear local buffer (including Icons)
    for (int idx=0; idx < (DVD538A_NR_GRIDS << 1); idx++) {  // * PT6964_BYTES_PER_GRID
      _displaybuffer[idx] = 0x00;  
    }
  }  
  else {
    //clear local buffer (preserving Icons)
    for (int idx=0; idx < DVD538A_NR_GRIDS; idx++) {
      _displaybuffer[(idx<<1)]     = _displaybuffer[(idx<<1)]     & MASK_ICON_GRID[idx][0];  
      _displaybuffer[(idx<<1) + 1] = _displaybuffer[(idx<<1) + 1] & MASK_ICON_GRID[idx][1];
    }  
  }

  writeData(_displaybuffer, (DVD538A_NR_GRIDS * PT6964_BYTES_PER_GRID));

  _column = 0;   
}     


/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void PT6964_DVD538A::setIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) << 1;     // * PT6964_BYTES_PER_GRID
    
  //Save char...and set bits for icon to write
  _displaybuffer[addr]   = _displaybuffer[addr]   | LO(icn);      
  _displaybuffer[addr+1] = _displaybuffer[addr+1] | HI(icn);      
  writeData(_displaybuffer, (DVD538A_NR_GRIDS * PT6964_BYTES_PER_GRID));
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 16 LSBs
  * @return none
  */
void PT6964_DVD538A::clrIcon(Icon icon) {
  int addr, icn;

  icn =        icon  & 0xFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) << 1;     // * PT6964_BYTES_PER_GRID

  //Save char...and clr bits for icon to write
  _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(icn);      
  _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~HI(icn);      
  writeData(_displaybuffer, (DVD538A_NR_GRIDS * PT6964_BYTES_PER_GRID));
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (16 bits)       
  */
void PT6964_DVD538A::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (DVD538A_NR_UDC-1)) {
    return;
  }
  // Mask out Icon bits?

  _UDC_7S[udc_idx] = udc_data;
}


/** Write a single character (Stream implementation)
  */
int PT6964_DVD538A::_putc(int value) {
    int addr;
    bool validChar = false;
    short pattern  = 0x0000;
    
    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    } 
    else if (value == '-') {
      //No character to write
      validChar = true;
      pattern = C7_MIN;         
    }
    else if ((value >= 0) && (value < DVD538A_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_7S[value];
    }  
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'a'];
    } //else

    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      //_column == 0 => Grid5 => addr = 8
      //_column == 1 => Grid4 => addr = 6
      //_column == 2 => Grid3 => addr = 4
      //_column == 3 => Grid4 => addr = 2            
      addr = (4 - _column) << 1;  // * PT6964_BYTES_PER_GRID

      //Save icons...and set bits for character to write
      _displaybuffer[addr]   = (_displaybuffer[addr]   & MASK_ICON_GRID[4 - _column][0]) | LO(pattern);
      _displaybuffer[addr+1] = (_displaybuffer[addr+1] & MASK_ICON_GRID[4 - _column][1]) | HI(pattern);

      writeData(_displaybuffer, (DVD538A_NR_GRIDS * PT6964_BYTES_PER_GRID));
                                
      //Update Cursor
      _column++;
      if (_column > (DVD538A_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}

// get a single character (Stream implementation)
int PT6964_DVD538A::_getc() {
    return -1;
}
