/* mbed PT6318 Library, for Princeton PT6318 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version, for KUH8300 code
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "PT6318.h"

/** Constructor for class for driving Princeton PT6318 VFD controller
 *
 *  @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
 *         SPI bus interface device. 
 *   
 *  @param PinName mosi, miso, sclk, cs SPI bus pins
 *  @param Mode selects either number of Digits and Segments
*/
PT6318::PT6318(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode) : _spi(mosi,miso,sclk), _cs(cs), _mode(mode) {

  _init();
}

/** Init the SPI interface and the controller
  * @param  none
  * @return none
  */ 
void PT6318::_init(){
  
//init SPI
  _cs=1;
  _spi.format(8,3); //PT6318 uses mode 3 (Clock High on Idle, Data latched on second (=rising) edge)
//  _spi.frequency(100000);   
  _spi.frequency(500000);     

//init controller  
  _writeCmd(PT6318_MODE_SET_CMD, _mode);                                               // Mode set command

  _display = PT6318_DSP_ON;
  _bright  = PT6318_BRT_DEF; 
  _writeCmd(PT6318_DSP_CTRL_CMD, _display | _bright );                                 // Display control cmd, display on/off, brightness   
  
  _writeCmd(PT6318_DATA_SET_CMD, PT6318_DATA_WR | PT6318_ADDR_INC | PT6318_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}   


/** Clear the screen and locate to 0
 */  
void PT6318::cls() {
  
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6318_ADDR_SET_CMD | 0x00)); // Address set cmd, 0
      
  for (int cnt=0; cnt<PT6318_DISPLAY_MEM; cnt++) {
    _spi.write(0x00); // data 
  }
  
  wait_us(1);
  _cs=1;      
  
}  

/** Set Brightness
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
  * @return none
  */
void PT6318::setBrightness(char brightness){

  _bright = brightness & PT6318_BRT_MSK; // mask invalid bits
  
  _writeCmd(PT6318_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness  

}

/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void PT6318::setDisplay(bool on) {
  
  if (on) {
    _display = PT6318_DSP_ON;
  }
  else {
   _display = PT6318_DSP_OFF;
  }
  
  _writeCmd(PT6318_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness   
}

/** Write databyte to PT6318
  *  @param  char data byte written at given address
  *  @param  int address display memory location to write byte  
  *  @return none
  */ 
void PT6318::writeData(char data, int address) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6318_ADDR_SET_CMD | (address & PT6318_ADDR_MSK))); // Set Address cmd
      
  _spi.write(_flip(data)); // data 
  
  wait_us(1);
  _cs=1;         
    
}

/** Write Display datablock to PT6318
  *  @param  DisplayData_t data Array of PT6318_DISPLAY_MEM (=48) bytes for displaydata (starting at address)
  *  @param  length number bytes to write (valid range 0..PT6318_DISPLAY_MEM (=48), starting at address)   
  *  @param  int address display memory location to write byte    
  *  @return none
  */    
void PT6318::writeData(DisplayData_t data, int length, int address) {
  _cs=0;
  wait_us(1);    
     
// sanity check
  address &= PT6318_ADDR_MSK;
  if (length < 0) {length = 0;}
//  if (length > PT6318_DISPLAY_MEM) {length = PT6318_DISPLAY_MEM;}
  if ((address + length) > PT6318_DISPLAY_MEM) {length = PT6318_DISPLAY_MEM - address;}
  
//  _spi.write(_flip(PT6318_ADDR_SET_CMD | 0x00)); // Set Address at 0
  _spi.write(_flip(PT6318_ADDR_SET_CMD | address)); // Set Address
    
//  for (int idx=0; idx<PT6318_DISPLAY_MEM; idx++) {  
  for (int idx=0; idx<length; idx++) {    
//    _spi.write(_flip(data[idx])); // data 
    _spi.write(_flip(data[address + idx])); // data    
  }
  
  wait_us(1);
  _cs=1;
}

/** Read keydata block from PT6318
  *  @param  *keydata Ptr to Array of PT6318_KEY_MEM (=6) bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  *
  * Note: Due to the hardware configuration the PT6318 key matrix scanner will detect multiple keys pressed at same time,
  *       but this may also result in some spurious keys being set in keypress data array.
  *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.  
  */ 
bool PT6318::getKeys(KeyData_t *keydata) {
  int keypress = 0;
  char data;

  // Read keys
  _cs=0;
  wait_us(1);    
  
  // Enable Key Read mode
  _spi.write(_flip(PT6318_DATA_SET_CMD | PT6318_KEY_RD | PT6318_ADDR_INC | PT6318_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  for (int idx=0; idx < PT6318_KEY_MEM; idx++) {
    data = _flip(_spi.write(0xFF));    // read keys and correct bitorder
    
    data = data & PT6318_KEY_MSK; // Mask valid bits
    if (data != 0) {  // Check for any pressed key
      for (int bit=0; bit < 8; bit++) {
        if (data & (1 << bit)) {keypress++;} // Test all significant bits
      }
    }  

    (*keydata)[idx] = data;            // Store keydata after correcting bitorder
  }

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6318_DATA_SET_CMD, PT6318_DATA_WR | PT6318_ADDR_INC | PT6318_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
#if(1)
// Dismiss multiple keypresses at same time
  return (keypress == 1);    
#else
// Allow multiple keypress and accept possible spurious keys
  return (keypress > 0);
#endif  
}

/** Read switches from PT6318
  *
  *  @param  none
  *  @return char for switch data (4 least significant bits) 
  *
  */   
char PT6318::getSwitches() {
  char data;

  // Read switches
  _cs=0;
  wait_us(1);    
  
  // Enable Switch Read mode
  _spi.write(_flip(PT6318_DATA_SET_CMD | PT6318_SW_RD | PT6318_ADDR_INC | PT6318_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  data = _flip(_spi.write(0xFF)) & PT6318_SW_MSK;   // read switches and correct bitorder

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6318_DATA_SET_CMD, PT6318_DATA_WR | PT6318_ADDR_INC | PT6318_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
  return data;       
}
   

/** Set LEDs
  *
  * @param  char leds (4 least significant bits)  
  * @return none
  */
void  PT6318::setLED (char leds) {

  // Set LEDs
  _cs=0;
  wait_us(1);    
  
  // Enable LED Write mode
  _spi.write(_flip(PT6318_DATA_SET_CMD | PT6318_LED_WR | PT6318_ADDR_INC | PT6318_MODE_NORM)); // Data set cmd, normal mode, auto incr, write data

  _spi.write(_flip(leds & PT6318_LED_MSK));    // write LEDs in correct bitorder

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6318_DATA_SET_CMD, PT6318_DATA_WR | PT6318_ADDR_INC | PT6318_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}



/** Helper to reverse all command or databits. The PT6318 expects LSB first, whereas SPI is MSB first
  *  @param  char data
  *  @return bitreversed data
  */ 
char PT6318::_flip(char data) {
 char value=0;
  
 if (data & 0x01) {value |= 0x80;} ;  
 if (data & 0x02) {value |= 0x40;} ;
 if (data & 0x04) {value |= 0x20;} ;
 if (data & 0x08) {value |= 0x10;} ;
 if (data & 0x10) {value |= 0x08;} ;
 if (data & 0x20) {value |= 0x04;} ;
 if (data & 0x40) {value |= 0x02;} ;
 if (data & 0x80) {value |= 0x01;} ;
 return value;       
}


/** Write command and parameter to PT6318
  *  @param  int cmd Command byte
  *  &Param  int data Parameters for command
  *  @return none
  */  
void PT6318::_writeCmd(int cmd, int data){
    
  _cs=0;
  wait_us(1);    
 
  _spi.write(_flip( (cmd & PT6318_CMD_MSK) | (data & ~PT6318_CMD_MSK)));   
 
  wait_us(1);
  _cs=1;      
    
};  



#if (KUH8300_TEST == 1)
/** Constructor for class for driving Princeton PT6318 VFD controller as used in KUH8300
  *
  *  @brief Supports 8 Grids of 20 Segments and Icons (8 1/2 digits of 14 Segments plus some icons).  
  *         Also supports a scanned keyboard of 12 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */  
PT6318_KUH8300::PT6318_KUH8300(PinName mosi, PinName miso, PinName sclk, PinName cs) : PT6318(mosi, miso, sclk, cs, Grid8_Seg20) {
  _column  = 0;
  _columns = KUH8300_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  */
void PT6318_KUH8300::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int PT6318_KUH8300::columns() {
    return _columns;
}

    
/** Clear the screen and locate to 0
  * @param bool clrAll Clear Icons also (default = false)
  */ 
void PT6318_KUH8300::cls(bool clrAll) {  
  int idx;
  
  if (clrAll) {
    //clear local buffer (including Icons)
    for (idx=0; idx < (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID); idx++) {
      _displaybuffer[idx] = 0x00;  
    }
  }  
  else {
    //clear local buffer (preserving Icons)
    for (int grd=0; grd < KUH8300_NR_GRIDS; grd++) {
      idx = grd * PT6318_BYTES_PER_GRID; // 3 bytes for every Grid
      _displaybuffer[idx    ] = _displaybuffer[idx    ] & MASK_ICON_GRID[grd][0];  
      _displaybuffer[idx + 1] = _displaybuffer[idx + 1] & MASK_ICON_GRID[grd][1];
      _displaybuffer[idx + 2] = _displaybuffer[idx + 2] & MASK_ICON_GRID[grd][2];      
    }  
  }

  writeData(_displaybuffer, (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID));

  _column = 0;   
}    

/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  * @return none
  */
void PT6318_KUH8300::setIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) * PT6318_BYTES_PER_GRID;  // 3 Bytes for every Grid
    
  //Save char...and set bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] | LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] | MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] | HI(icn);        
  writeData(_displaybuffer, (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID));
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  * @return none
  */
void PT6318_KUH8300::clrIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) * PT6318_BYTES_PER_GRID;   // 3 Bytes for every Grid
    
  //Save char...and clr bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] & ~LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] & ~MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] & ~HI(icn);        
  writeData(_displaybuffer, (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID));
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (16 bits)       
  */
void PT6318_KUH8300::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (KUH8300_NR_UDC-1)) {
    return;
  }

  // Mask out Icon bits?
  _UDC_7S[udc_idx] = udc_data & 0xFF;
}

/** Write a single character (Stream implementation)
  */
int PT6318_KUH8300::_putc(int value) {
    bool validChar = false;
    bool even = false;
    char pattern  = 0x00;
    int addr;
    
    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    }
    else if ((value >= 0) && (value < KUH8300_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_7S[value];
    }  
#if (SHOW_ASCII == 1)
    //display all ASCII characters
    else if ((value >= FONT_7S_START) && (value <= FONT_7S_END)) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - FONT_7S_START];
    } // else
#else
    //display only digits and hex characters
    else if (value == '-') {
      //Character to write
      validChar = true;
      pattern = C7_MIN;         
    }
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_7S[10 + value - (int) 'a'];
    } //else
#endif
    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      //Note that the KUH8300 has 8 7-Segment digits using 4 Grids. 
      //Some of these Grids also have icons that need to be preserved
      //_column ==  0 => Grid5 => addr = 12..14
      //_column ==  1 => Grid5 => addr = 12..14
      // ....
      //_column ==  6 => Grid2 => addr =  3..5            
      //_column ==  7 => Grid2 => addr =  3..5            
      addr = 12 - ((_column >> 1) * PT6318_BYTES_PER_GRID); // 3 Bytes for every Grid;
      even = ((_column & 0x01) == 0x00);                    // Odd or Even column
      
      // pattern is a placeholder, test pattern and set the actual segment depending on the odd or even _column for this Grid
      if (pattern & S7_A) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_A0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] | MD(S7_A1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_A0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~MD(S7_A1);
        }
      }

      if (pattern & S7_B) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_B0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] | MD(S7_B1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_B0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~MD(S7_B1);
        }
      }

      if (pattern & S7_C) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_C0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] | MD(S7_C1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_C0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~MD(S7_C1);
        }
      }

      if (pattern & S7_D) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_D0);
        }
        else {
          _displaybuffer[addr+2] = _displaybuffer[addr+2] | HI(S7_D1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_D0);
        }
        else {
          _displaybuffer[addr+2] = _displaybuffer[addr+2] & ~HI(S7_D1);
        }
      }

      if (pattern & S7_E) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_E0);
        }
        else {
          _displaybuffer[addr+2] = _displaybuffer[addr+2] | HI(S7_E1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_E0);
        }
        else {
          _displaybuffer[addr+2] = _displaybuffer[addr+2] & ~HI(S7_E1);
        }
      }

      if (pattern & S7_F) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_F0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] | MD(S7_F1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_F0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~MD(S7_F1);
        }
      }

      if (pattern & S7_G) {
        //Set Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   | LO(S7_G0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] | MD(S7_G1);
        }
      }
      else {
        //Clear Segment
        if (even) {
          _displaybuffer[addr]   = _displaybuffer[addr]   & ~LO(S7_G0);
        }
        else {
          _displaybuffer[addr+1] = _displaybuffer[addr+1] & ~MD(S7_G1);
        }
      }

//      //Save icons...and set bits for character to write
//      _displaybuffer[addr]   = (_displaybuffer[addr]   & MASK_ICON_GRID[_column][0]) | LO(pattern);
//      _displaybuffer[addr+1] = (_displaybuffer[addr+1] & MASK_ICON_GRID[_column][1]) | MD(pattern);
//      _displaybuffer[addr+2] = (_displaybuffer[addr+2] & MASK_ICON_GRID[_column][2]) | HI(pattern);

//      writeData(_displaybuffer, (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID));
      writeData(_displaybuffer, PT6318_BYTES_PER_GRID, addr);      
                                
      //Update Cursor
      _column++;
      if (_column > (KUH8300_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}

// get a single character (Stream implementation)
int PT6318_KUH8300::_getc() {
    return -1;
}
#endif
