/* mbed PT6318 Library, for Princeton PT6318 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version for KUH8300
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6318_H
#define PT6318_H

// Select one of the testboards for Princeton PT6318 VFD controller
#include "PT6318_Config.h"

/** An interface for driving Princeton PT6318 VFD controller
 *
 * @code
 *
 * #if (PT6318_TEST == 1)  
 * // Direct driving of PT6318 Test
 *
 * #include "mbed.h"
 * #include "PT6318.h" 
 * 
 * DisplayData_t size is 24 bytes (8 Grids @ 20 Segments) ... 48 bytes (16 Grids @ 12 Segments) 
 * DisplayData_t size default is 48 bytes (16 Grids @ 12 Segments) 
 * PT6318::DisplayData_t all_str  = {0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00,
 *                                   0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00};  
 *
 * // KeyData_t size is 6 bytes  
 * PT6318::KeyData_t keydata; 
 *
 * // PT6318 declaration, Default setting 16 Grids @ 12 Segments
 * PT6318 PT6318(p5,p6,p7, p8);
 *
 * int main() {
 *   PT6318.cls(); 
 *   PT6318.writeData(all_str);
 *   wait(4);
 *   PT6318.setBrightness(PT6318_BRT0);
 *   wait(1);
 *   PT6318.setBrightness(PT6318_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (PT6318.getKeys(&keydata)) {
 *      pc.printf("Keydata 0..5 = 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n", keydata[0], keydata[1], keydata[2], keydata[3], keydata[4], keydata[5]);
 *
 *      if (keydata[0] == 0x01) { //sw1   
 *        PT6318.cls(); 
 *        wait(1);
 *        PT6318.writeData(all_str);
 *      }  
 *    } 
 *   }   
 * }
 * #endif
 *
 * @endcode
 */

//PT6318 Display and Keymatrix data
#define PT6318_MAX_NR_GRIDS   16
#define PT6318_BYTES_PER_GRID  3
//Significant bits Keymatrix data
#define PT6318_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define PT6318_DISPLAY_MEM  (PT6318_MAX_NR_GRIDS * PT6318_BYTES_PER_GRID)
#define PT6318_KEY_MEM         6

//Reserved bits for commands
#define PT6318_CMD_MSK      0xC0

//Mode setting command
#define PT6318_MODE_SET_CMD 0x00
#define PT6318_GR8_SEG20    0x00
#define PT6318_GR9_SEG19    0x08
#define PT6318_GR10_SEG18   0x09
#define PT6318_GR11_SEG17   0x0A
#define PT6318_GR12_SEG16   0x0B
#define PT6318_GR13_SEG15   0x0C
#define PT6318_GR14_SEG14   0x0D
#define PT6318_GR15_SEG13   0x0E
#define PT6318_GR16_SEG12   0x0F //default

//Data setting commands
#define PT6318_DATA_SET_CMD 0x40
#define PT6318_DATA_WR      0x00
#define PT6318_LED_WR       0x01
#define PT6318_KEY_RD       0x02
#define PT6318_SW_RD        0x03
#define PT6318_ADDR_INC     0x00
#define PT6318_ADDR_FIXED   0x04
#define PT6318_MODE_NORM    0x00
#define PT6318_MODE_TEST    0x08

//LED settings data
#define PT6318_LED_MSK      0x1F
#define PT6318_LED1         0x01
#define PT6318_LED2         0x02
#define PT6318_LED3         0x04
#define PT6318_LED4         0x08
#define PT6318_LED5         0x10

//Switch settings data
#define PT6318_SW_MSK       0x0F
#define PT6318_SW1          0x01
#define PT6318_SW2          0x02
#define PT6318_SW3          0x04
#define PT6318_SW4          0x08

//Address setting commands
#define PT6318_ADDR_SET_CMD 0xC0
#define PT6318_ADDR_MSK     0x3F

//Display control commands
#define PT6318_DSP_CTRL_CMD 0x80
#define PT6318_BRT_MSK      0x07
#define PT6318_BRT0         0x00 //Pulsewidth 1/16, Default
#define PT6318_BRT1         0x01
#define PT6318_BRT2         0x02
#define PT6318_BRT3         0x03
#define PT6318_BRT4         0x04
#define PT6318_BRT5         0x05
#define PT6318_BRT6         0x06
#define PT6318_BRT7         0x07 //Pulsewidth 14/16

#define PT6318_BRT_DEF      PT6318_BRT3

#define PT6318_DSP_OFF      0x00 //Default
#define PT6318_DSP_ON       0x08


/** A class for driving Princeton PT6318 VFD controller
 *
 * @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
 *        SPI bus interface device. 
 */
class PT6318 {
 public:

  /** Enums for display mode */
  enum Mode {  
    Grid8_Seg20  = PT6318_GR8_SEG20,  /**<   8 Grids, 20 Segments */
    Grid9_Seg19  = PT6318_GR9_SEG19,  /**<   9 Grids, 19 Segments */
    Grid10_Seg18 = PT6318_GR10_SEG18, /**<  10 Grids, 18 Segments */
    Grid11_Seg17 = PT6318_GR11_SEG17, /**<  11 Grids, 17 Segments */
    Grid12_Seg16 = PT6318_GR12_SEG16, /**<  12 Grids, 16 Segments */
    Grid13_Seg15 = PT6318_GR13_SEG15, /**<  13 Grids, 15 Segments */
    Grid14_Seg14 = PT6318_GR14_SEG14, /**<  14 Grids, 14 Segments */
    Grid15_Seg13 = PT6318_GR15_SEG13, /**<  15 Grids, 13 Segments */
    Grid16_Seg12 = PT6318_GR16_SEG12  /**<  16 Grids, 12 Segments */
  };
 
  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[PT6318_DISPLAY_MEM];
  typedef char KeyData_t[PT6318_KEY_MEM];
    
 /** Constructor for class for driving Princeton PT6318 VFD controller
  *
  *  @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 32 keys, 4 switches and 5 LEDs.
  *         SPI bus interface device. 
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  *  @param  Mode selects either number of Grids and Segments (default 16 Grids, 12 Segments)
  */
  PT6318(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode=Grid16_Seg12);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to PT6318
   *  @param  char data byte written at given address
   *  @param  int address display memory location to write byte   
   *  @return none
   */ 
  void writeData(char data, int address); 
 
  /** Write Display datablock to PT6318
   *  @param  DisplayData_t data Array of PT6318_DISPLAY_MEM (=48) bytes for displaydata (starting at address)
   *  @param  length number bytes to write (valid range 0..PT6318_DISPLAY_MEM (=48), starting at address)   
   *  @param  int address display memory location to write byte (default = 0)
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = PT6318_DISPLAY_MEM, int address = 0);


  /** Read keydata block from PT6318
   *  @param  *keydata Ptr to Array of PT6318_KEY_MEM (=6) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the PT6318 key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);


  /** Read switches from PT6318
   *
   *  @param  none
   *  @return char for switch data (4 least significant bits)
   */   
  char getSwitches();

  /** Set LEDs
    *
    * @param  char leds (5 least significant bits)  
    * @return none
    */
  void setLED (char leds = 0);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = PT6318_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  char _display;
  char _bright; 
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The PT6318 expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write command and parameter to PT6318
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
  void _writeCmd(int cmd, int data);  
};



#if (KUH8300_TEST == 1)
// Derived class for KUH8300 front display unit
//   Grids 2-11 all display 14-Segment digits and several Icons.
//   Grid 1 and 12 display Icons. 

//
#include "Font_7Seg.h"

//KUH8300 Display data
#define KUH8300_NR_GRIDS   6
#define KUH8300_NR_DIGITS  8
#define KUH8300_NR_UDC     8

//KUH8300 Memory size in bytes for Display
#define KUH8300_DISPLAY_MEM (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID)


/** Constructor for class for driving Princeton PT6318 VFD controller as used in KUH8300
  *
  *  @brief Supports 8 Grids of 20 Segments and Icons (8 1/2 digits of 14 segments plus some icons).
  *         Also supports a scanned keyboard of 12 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class PT6318_KUH8300 : public PT6318, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  enum Icon {
    CAM  = (1<<24) | S7_CAM,
    DIG  = (1<<24) | S7_DIG,
    MONO = (1<<24) | S7_MONO,
    PRG  = (1<<24) | S7_PRG,    
    DTS  = (1<<24) | S7_DTS,
    MEM  = (1<<24) | S7_MEM,
    KHZ  = (1<<24) | S7_KHZ,
    STR  = (1<<24) | S7_STR,
    MHZ  = (1<<24) | S7_MHZ,

    KEY  = (2<<24) | S7_KEY,
    ALL  = (2<<24) | S7_ALL,
    DP6  = (2<<24) | S7_DP6,
    COL6 = (2<<24) | S7_COL6,    

    ARW  = (3<<24) | S7_ARW,
    A_   = (3<<24) | S7_A_,
    B_   = (3<<24) | S7_B_,
    CD4A = (3<<24) | S7_CD4A,    
    
    CHP  = (4<<24) | S7_CHP,
    N3   = (4<<24) | S7_N3,
    L3   = (4<<24) | S7_L3,
    HK3  = (4<<24) | S7_HK3,    

    TTL  = (5<<24) | S7_TTL,
    JM1  = (5<<24) | S7_JM1,    
    HK2  = (5<<24) | S7_HK2,    

    PSE  = (6<<24) | S7_PSE,
    PLY  = (6<<24) | S7_PLY,    
    PBC  = (6<<24) | S7_PBC,    
    DVD  = (6<<24) | S7_DVD,
    CD   = (6<<24) | S7_CD,    
    V    = (6<<24) | S7_V,    
    S    = (6<<24) | S7_S
  };
  
  typedef char UDCData_t[KUH8300_NR_UDC];


/** Constructor for class for driving Princeton PT6318 VFD controller as used in KUH8300
  *
  *  @brief Supports 8 Grids of 20 Segments and Icons (8 1/2 digits of 14 Segments plus some icons).
  *         Also supports a scanned keyboard of 12 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
  PT6318_KUH8300(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to PT6318
     *  @param  char data byte written at given address
     *  @param  int address display memory location to write byte     
     *  @return none
     */ 
    void writeData(int address, char data){
      PT6318::writeData(address, data);
    }        
 
   /** Write Display datablock to PT6318
    *  @param  DisplayData_t data Array of PT6318_DISPLAY_MEM (=36) bytes for displaydata (starting at address)
    *  @param  length number bytes to write (valid range 0..(KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID) == 24, starting at address)   
    *  @param  int address display memory location to write byte (default = 0)
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (KUH8300_NR_GRIDS * PT6318_BYTES_PER_GRID), int address = 0) {
      PT6318::writeData(data, length, address);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;                     // Current cursor location
    int _columns;                    // Max number of columns
    
    DisplayData_t _displaybuffer;    // Local mirror for all chars and icons
    UDCData_t _UDC_7S;               // User Defined Character pattterns (UDC)    
}; 
#endif

#endif
