/* mbed PT6315 Library, for Princeton PT6315 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version for DVDR3510
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef PT6315_H
#define PT6315_H

// Select one of the testboards for Princeton PT6315 VFD controller
#include "PT6315_Config.h"

/** An interface for driving Princeton PT6315 VFD controller
 *
 * @code
 *
 * #if (PT6315_TEST == 1)  
 * // Direct driving of PT6315 Test
 *
 * #include "mbed.h"
 * #include "PT6315.h" 
 * 
 * DisplayData_t size is 24 bytes (8 Grids @ 20 Segments) ... 48 bytes (16 Grids @ 12 Segments) 
 * DisplayData_t size default is 48 bytes (16 Grids @ 12 Segments) 
 * PT6315::DisplayData_t all_str  = {0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00,
 *                                   0xFF,0x0F,0x00 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00, 0xFF,0x0F,0x00};  
 *
 * // KeyData_t size is 6 bytes  
 * PT6315::KeyData_t keydata; 
 *
 * // PT6315 declaration, Default setting 16 Grids @ 12 Segments
 * PT6315 PT6315(p5,p6,p7, p8);
 *
 * int main() {
 *   PT6315.cls(); 
 *   PT6315.writeData(all_str);
 *   wait(4);
 *   PT6315.setBrightness(PT6315_BRT0);
 *   wait(1);
 *   PT6315.setBrightness(PT6315_BRT3);
 *
 *   while (1) {
 *    // Check and read keydata
 *    if (PT6315.getKeys(&keydata)) {
 *      pc.printf("Keydata 0..5 = 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x\r\n", keydata[0], keydata[1], keydata[2], keydata[3], keydata[4], keydata[5]);
 *
 *      if (keydata[0] == 0x10) { //sw2   
 *        PT6315.cls(); 
 *        wait(1);
 *        PT6315.writeData(all_str);
 *      }  
 *    } 
 *   }   
 * }
 * #endif
 *
 * @endcode
 */

//PT6315 Display and Keymatrix data
#define PT6315_MAX_NR_GRIDS   12
#define PT6315_BYTES_PER_GRID  3
//Significant bits Keymatrix data
#define PT6315_KEY_MSK      0xFF 

//Memory size in bytes for Display and Keymatrix
#define PT6315_DISPLAY_MEM  (PT6315_MAX_NR_GRIDS * PT6315_BYTES_PER_GRID)
#define PT6315_KEY_MEM         4

//Reserved bits for commands
#define PT6315_CMD_MSK      0xC0

//Mode setting command
#define PT6315_MODE_SET_CMD 0x00
#define PT6315_GR4_SEG24    0x00
#define PT6315_GR5_SEG23    0x01
#define PT6315_GR6_SEG22    0x02
#define PT6315_GR7_SEG21    0x03
#define PT6315_GR8_SEG20    0x04
#define PT6315_GR9_SEG19    0x05
#define PT6315_GR10_SEG18   0x06
#define PT6315_GR11_SEG17   0x07
#define PT6315_GR12_SEG16   0x08 //default

//Data setting commands
#define PT6315_DATA_SET_CMD 0x40
#define PT6315_DATA_WR      0x00
#define PT6315_LED_WR       0x01
#define PT6315_KEY_RD       0x02
#define PT6315_SW_RD        0x03
#define PT6315_ADDR_INC     0x00
#define PT6315_ADDR_FIXED   0x04
#define PT6315_MODE_NORM    0x00
#define PT6315_MODE_TEST    0x08

//LED settings data
#define PT6315_LED_MSK      0x0F
#define PT6315_LED1         0x01
#define PT6315_LED2         0x02
#define PT6315_LED3         0x04
#define PT6315_LED4         0x08

//Address setting commands
#define PT6315_ADDR_SET_CMD 0xC0
#define PT6315_ADDR_MSK     0x3F

//Display control commands
#define PT6315_DSP_CTRL_CMD 0x80
#define PT6315_BRT_MSK      0x07
#define PT6315_BRT0         0x00 //Pulsewidth 1/16, Default
#define PT6315_BRT1         0x01
#define PT6315_BRT2         0x02
#define PT6315_BRT3         0x03
#define PT6315_BRT4         0x04
#define PT6315_BRT5         0x05
#define PT6315_BRT6         0x06
#define PT6315_BRT7         0x07 //Pulsewidth 14/16

#define PT6315_BRT_DEF      PT6315_BRT3

#define PT6315_DSP_OFF      0x00 //Default
#define PT6315_DSP_ON       0x08


/** A class for driving Princeton PT6315 VFD controller
 *
 * @brief Supports 8 Grids of 20 Segments upto 16 Grids of 12 Segments. Also supports a scanned keyboard of upto 48 keys, 4 switches and 5 LEDs.
 *        SPI bus interface device. 
 */
class PT6315 {
 public:

  /** Enums for display mode */
  enum Mode {
    Grid4_Seg24  = PT6315_GR4_SEG24,
    Grid5_Seg23  = PT6315_GR5_SEG23,
    Grid6_Seg22  = PT6315_GR6_SEG22,
    Grid7_Seg21  = PT6315_GR7_SEG21,
    Grid8_Seg20  = PT6315_GR8_SEG20,
    Grid9_Seg19  = PT6315_GR9_SEG19,
    Grid10_Seg18 = PT6315_GR10_SEG18,
    Grid11_Seg17 = PT6315_GR11_SEG17,   
    Grid12_Seg16 = PT6315_GR12_SEG16   
  };
 
  /** Datatypes for display and keymatrix data */
  typedef char DisplayData_t[PT6315_DISPLAY_MEM];
  typedef char KeyData_t[PT6315_KEY_MEM];
    
 /** Constructor for class for driving Princeton PT6315 VFD controller
  *
  *  @brief Supports 4 Grids of 24 Segments upto 12 Grids of 16 Segments. Also supports a scanned keyboard of upto 32 keys and 4 LEDs.
  *         SPI bus interface device. 
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  *  @param  Mode selects either number of Grids and Segments (default 12 Grids, 16 Segments)
  */
  PT6315(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode=Grid12_Seg16);
      
  /** Clear the screen and locate to 0
   */ 
  void cls();  

  /** Write databyte to PT6315
   *  @param  int address display memory location to write byte
   *  @param  char data byte written at given address
   *  @return none
   */ 
  void writeData(int address, char data); 
 
  /** Write Display datablock to PT6315
   *  @param  DisplayData_t data Array of PT6315_DISPLAY_MEM (=36) bytes for displaydata (starting at address 0)
   *  @param  length number bytes to write (valid range 0..PT6315_DISPLAY_MEM (=36), starting at address 0)   
   *  @return none
   */   
  void writeData(DisplayData_t data, int length = PT6315_DISPLAY_MEM);


  /** Read keydata block from PT6315
   *  @param  *keydata Ptr to Array of PT6315_KEY_MEM (=4) bytes for keydata
   *  @return bool keypress True when at least one key was pressed
   *
   * Note: Due to the hardware configuration the PT6315 key matrix scanner will detect multiple keys pressed at same time,
   *       but this may result in some spurious keys also being set in keypress data array.
   *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.
   */   
  bool getKeys(KeyData_t *keydata);


  /** Set LEDs
    *
    * @param  char leds (4 least significant bits)  
    * @return none
    */
  void setLED (char leds = 0);

  /** Set Brightness
    *
    * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/16 dutycycle)  
    * @return none
    */
  void setBrightness(char brightness = PT6315_BRT_DEF);
  
  /** Set the Display mode On/off
    *
    * @param bool display mode
    */
  void setDisplay(bool on);
  
 private:  
  SPI _spi;
  DigitalOut _cs;
  Mode _mode;
  char _display;
  char _bright; 
  
  /** Init the SPI interface and the controller
    * @param  none
    * @return none
    */ 
  void _init();

  /** Helper to reverse all command or databits. The PT6315 expects LSB first, whereas SPI is MSB first
    *  @param  char data
    *  @return bitreversed data
    */ 
  char _flip(char data);

  /** Write command and parameter to PT6315
    *  @param  int cmd Command byte
    *  &Param  int data Parameters for command
    *  @return none
    */ 
  void _writeCmd(int cmd, int data);  
};



#if (DVDR3510_TEST == 1)
// Derived class for DVDR3510 front display unit
//   Grids 2-11 all display 14-Segment digits and several Icons.
//   Grid 1 and 12 display Icons. 

//
#include "Font_16Seg.h"

//DVDR3510 Display data
#define DVDR3510_NR_GRIDS  11
#define DVDR3510_NR_DIGITS  8
#define DVDR3510_NR_UDC     8

//DVDR3510 Memory size in bytes for Display
#define DVDR3510_DISPLAY_MEM (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID)


/** Constructor for class for driving Princeton PT6315 VFD controller as used in DVDR3510
  *
  *  @brief Supports 11 Grids of 17 Segments and Icons (8 digits of 14 segments plus some icons).
  *         Also supports a scanned keyboard of 11 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
class PT6315_DVDR3510 : public PT6315, public Stream {
 public:

  /** Enums for Icons */
  //  Grid encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  enum Icon {
    RCV = (1<<24) | S16_RCV,
    SAT = (8<<24) | S16_SAT,
    DRT = (9<<24) | S16_DRT,
    TV  = (9<<24) | S16_TV,    
    COL4 = (10<<24) | S16_COL4,
    TMR  = (10<<24) | S16_TMR,
    COL6 = (11<<24) | S16_COL6,
    DP6  = (11<<24) | S16_DP6,
    PRS  = (11<<24) | S16_PRS
  };
  
  typedef short UDCData_t[DVDR3510_NR_UDC];


/** Constructor for class for driving Princeton PT6315 VFD controller as used in DVDR3510
  *
  *  @brief Supports 8 Grids of 16 Segments and Icons (8 digits of 14 Segments plus some icons).
  *         Also supports a scanned keyboard of 11 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */
  PT6315_DVDR3510(PinName mosi, PinName miso, PinName sclk, PinName cs);

#if DOXYGEN_ONLY
    /** Write a character to the Display
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the Display
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif

     /** Locate cursor to a screen column
     *
     * @param column  The horizontal position from the left, indexed from 0
     */
    void locate(int column);
    
    /** Clear the screen and locate to 0
     * @param bool clrAll Clear Icons also (default = false)
     */
    void cls(bool clrAll = false);

    /** Set Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
     * @return none
     */
    void setIcon(Icon icon);

    /** Clr Icon
     *
     * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
     * @return none
     */
    void clrIcon(Icon icon);

   /** Set User Defined Characters (UDC)
     *
     * @param unsigned char udc_idx   The Index of the UDC (0..7)
     * @param int udc_data            The bitpattern for the UDC (16 bits)       
     */
    void setUDC(unsigned char udc_idx, int udc_data);


   /** Number of screen columns
    *
    * @param none
    * @return columns
    */
    int columns();   

   /** Write databyte to PT6315
     *  @param  int address display memory location to write byte
     *  @param  char data byte written at given address
     *  @return none
     */ 
    void writeData(int address, char data){
      PT6315::writeData(address, data);
    }        
 
   /** Write Display datablock to PT6315
    *  @param  DisplayData_t data Array of PT6315_DISPLAY_MEM (=36) bytes for displaydata (starting at address 0)
    *  @param  length number bytes to write (valid range 0..(DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID) == 24, starting at address 0)   
    *  @return none
    */   
    void writeData(DisplayData_t data, int length = (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID)) {
      PT6315::writeData(data, length);
    }  

protected:  
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

private:
    int _column;                     // Current cursor location
    int _columns;                    // Max number of columns
    
    DisplayData_t _displaybuffer;    // Local mirror for all chars and icons
    UDCData_t _UDC_16S;              // User Defined Character pattterns (UDC)    
};
#endif

#endif
