/* mbed PT6315 Library, for Princeton PT6315 VFD controller
 * Copyright (c) 2016, v01: WH, Initial version, for DVDR3510 code
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include "mbed.h" 
#include "PT6315.h"

/** Constructor for class for driving Princeton PT6315 VFD controller
 *
 *  @brief Supports 4 Grids of 24 Segments upto 12 Grids of 16 Segments. Also supports a scanned keyboard of upto 32 keys and 4 LEDs.
 *         SPI bus interface device. 
 *   
 *  @param PinName mosi, miso, sclk, cs SPI bus pins
 *  @param Mode selects either number of Digits and Segments
*/
PT6315::PT6315(PinName mosi, PinName miso, PinName sclk, PinName cs, Mode mode) : _spi(mosi,miso,sclk), _cs(cs), _mode(mode) {

  _init();
}

/** Init the SPI interface and the controller
  * @param  none
  * @return none
  */ 
void PT6315::_init(){
  
//init SPI
  _cs=1;
  _spi.format(8,3); //PT6315 uses mode 3 (Clock High on Idle, Data latched on second (=rising) edge)
//  _spi.frequency(100000);   
  _spi.frequency(500000);     

//init controller  
  _writeCmd(PT6315_MODE_SET_CMD, _mode);                                               // Mode set command

  _display = PT6315_DSP_ON;
  _bright  = PT6315_BRT_DEF; 
  _writeCmd(PT6315_DSP_CTRL_CMD, _display | _bright );                                 // Display control cmd, display on/off, brightness   
  
  _writeCmd(PT6315_DATA_SET_CMD, PT6315_DATA_WR | PT6315_ADDR_INC | PT6315_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}   


/** Clear the screen and locate to 0
 */  
void PT6315::cls() {
  
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6315_ADDR_SET_CMD | 0x00)); // Address set cmd, 0
      
  for (int cnt=0; cnt<PT6315_DISPLAY_MEM; cnt++) {
    _spi.write(0x00); // data 
  }
  
  wait_us(1);
  _cs=1;      
  
}  

/** Set Brightness
  *
  * @param  char brightness (3 significant bits, valid range 0..7 (1/16 .. 14/14 dutycycle)  
  * @return none
  */
void PT6315::setBrightness(char brightness){

  _bright = brightness & PT6315_BRT_MSK; // mask invalid bits
  
  _writeCmd(PT6315_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness  

}

/** Set the Display mode On/off
  *
  * @param bool display mode
  */
void PT6315::setDisplay(bool on) {
  
  if (on) {
    _display = PT6315_DSP_ON;
  }
  else {
   _display = PT6315_DSP_OFF;
  }
  
  _writeCmd(PT6315_DSP_CTRL_CMD, _display | _bright );  // Display control cmd, display on/off, brightness   
}

/** Write databyte to PT6315
  *  @param  int address display memory location to write byte
  *  @param  char data byte written at given address
  *  @return none
  */ 
void PT6315::writeData(int address, char data) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6315_ADDR_SET_CMD | (address & PT6315_ADDR_MSK))); // Set Address cmd
      
  _spi.write(_flip(data)); // data 
  
  wait_us(1);
  _cs=1;         
    
}

/** Write Display datablock to PT6315
  *  @param  DisplayData_t data Array of PT6315_DISPLAY_MEM (=48) bytes for displaydata (starting at address 0)
  *  @param  length number bytes to write (valid range 0..PT6315_DISPLAY_MEM (=48), starting at address 0)   
  *  @return none
  */    
void PT6315::writeData(DisplayData_t data, int length) {
  _cs=0;
  wait_us(1);    
  _spi.write(_flip(PT6315_ADDR_SET_CMD | 0x00)); // Set Address at 0
      
// sanity check
  if (length < 0) {length = 0;}
  if (length > PT6315_DISPLAY_MEM) {length = PT6315_DISPLAY_MEM;}

//  for (int idx=0; idx<PT6315_DISPLAY_MEM; idx++) {  
  for (int idx=0; idx<length; idx++) {    
    _spi.write(_flip(data[idx])); // data 
  }
  
  wait_us(1);
  _cs=1;         
    
}

/** Read keydata block from PT6315
  *  @param  *keydata Ptr to Array of PT6315_KEY_MEM (=4) bytes for keydata
  *  @return bool keypress True when at least one key was pressed
  *
  * Note: Due to the hardware configuration the PT6315 key matrix scanner will detect multiple keys pressed at same time,
  *       but this may also result in some spurious keys being set in keypress data array.
  *       It may be best to ignore all keys in those situations. That option is implemented in this method depending on #define setting.  
  */ 
bool PT6315::getKeys(KeyData_t *keydata) {
  int keypress = 0;
  char data;

  // Read keys
  _cs=0;
  wait_us(1);    
  
  // Enable Key Read mode
  _spi.write(_flip(PT6315_DATA_SET_CMD | PT6315_KEY_RD | PT6315_ADDR_INC | PT6315_MODE_NORM)); // Data set cmd, normal mode, auto incr, read data

  for (int idx=0; idx < PT6315_KEY_MEM; idx++) {
    data = _flip(_spi.write(0xFF));    // read keys and correct bitorder
    
    data = data & PT6315_KEY_MSK; // Mask valid bits
    if (data != 0) {  // Check for any pressed key
      for (int bit=0; bit < 8; bit++) {
        if (data & (1 << bit)) {keypress++;} // Test all significant bits
      }
    }  

    (*keydata)[idx] = data;            // Store keydata after correcting bitorder
  }

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6315_DATA_SET_CMD, PT6315_DATA_WR | PT6315_ADDR_INC | PT6315_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
      
#if(1)
// Dismiss multiple keypresses at same time
  return (keypress == 1);    
#else
// Allow multiple keypress and accept possible spurious keys
  return (keypress > 0);
#endif  
}

/** Set LEDs
  *
  * @param  char leds (4 least significant bits)  
  * @return none
  */
void  PT6315::setLED (char leds) {

  // Set LEDs
  _cs=0;
  wait_us(1);    
  
  // Enable LED Write mode
  _spi.write(_flip(PT6315_DATA_SET_CMD | PT6315_LED_WR | PT6315_ADDR_INC | PT6315_MODE_NORM)); // Data set cmd, normal mode, auto incr, write data

  _spi.write(_flip(leds & PT6315_LED_MSK));    // write LEDs in correct bitorder

  wait_us(1);
  _cs=1;    

  // Restore Data Write mode
  _writeCmd(PT6315_DATA_SET_CMD, PT6315_DATA_WR | PT6315_ADDR_INC | PT6315_MODE_NORM); // Data set cmd, normal mode, auto incr, write data  
}



/** Helper to reverse all command or databits. The PT6315 expects LSB first, whereas SPI is MSB first
  *  @param  char data
  *  @return bitreversed data
  */ 
char PT6315::_flip(char data) {
 char value=0;
  
 if (data & 0x01) {value |= 0x80;} ;  
 if (data & 0x02) {value |= 0x40;} ;
 if (data & 0x04) {value |= 0x20;} ;
 if (data & 0x08) {value |= 0x10;} ;
 if (data & 0x10) {value |= 0x08;} ;
 if (data & 0x20) {value |= 0x04;} ;
 if (data & 0x40) {value |= 0x02;} ;
 if (data & 0x80) {value |= 0x01;} ;
 return value;       
}


/** Write command and parameter to PT6315
  *  @param  int cmd Command byte
  *  &Param  int data Parameters for command
  *  @return none
  */  
void PT6315::_writeCmd(int cmd, int data){
    
  _cs=0;
  wait_us(1);    
 
  _spi.write(_flip( (cmd & PT6315_CMD_MSK) | (data & ~PT6315_CMD_MSK)));   
 
  wait_us(1);
  _cs=1;      
    
};  



#if (DVDR3510_TEST == 1)
/** Constructor for class for driving Princeton PT6315 VFD controller as used in DVDR3510
  *
  *  @brief Supports 11 Grids of 17 Segments and Icons (8 digits of 14 Segments plus some icons).  
  *         Also supports a scanned keyboard of 11 keys and 1 LED.
  *  
  *  @param  PinName mosi, miso, sclk, cs SPI bus pins
  */  
PT6315_DVDR3510::PT6315_DVDR3510(PinName mosi, PinName miso, PinName sclk, PinName cs) : PT6315(mosi, miso, sclk, cs, Grid11_Seg17) {
  _column  = 0;
  _columns = DVDR3510_NR_DIGITS;    
}  

#if(0)
#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formatted string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formatting the string.
     */
    int printf(const char* format, ...);   
#endif
#endif

/** Locate cursor to a screen column
  *
  * @param column  The horizontal position from the left, indexed from 0
  */
void PT6315_DVDR3510::locate(int column) {
  //sanity check
  if (column < 0) {column = 0;}
  if (column > (_columns - 1)) {column = _columns - 1;}  
  
  _column = column;       
}


/** Number of screen columns
  *
  * @param none
  * @return columns
  */
int PT6315_DVDR3510::columns() {
    return _columns;
}

    
/** Clear the screen and locate to 0
  * @param bool clrAll Clear Icons also (default = false)
  */ 
void PT6315_DVDR3510::cls(bool clrAll) {  
  int idx;
  
  if (clrAll) {
    //clear local buffer (including Icons)
    for (idx=0; idx < (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID); idx++) {
      _displaybuffer[idx] = 0x00;  
    }
  }  
  else {
    //clear local buffer (preserving Icons)
    for (int grd=0; grd < DVDR3510_NR_GRIDS; grd++) {
      idx = grd * PT6315_BYTES_PER_GRID; // 3 bytes for every Grid
      _displaybuffer[idx    ] = _displaybuffer[idx    ] & MASK_ICON_GRID[grd][0];  
      _displaybuffer[idx + 1] = _displaybuffer[idx + 1] & MASK_ICON_GRID[grd][1];
      _displaybuffer[idx + 2] = _displaybuffer[idx + 2] & MASK_ICON_GRID[grd][2];      
    }  
  }

  writeData(_displaybuffer, (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID));

  _column = 0;   
}    

/** Set Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  * @return none
  */
void PT6315_DVDR3510::setIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) * PT6315_BYTES_PER_GRID;  // 3 Bytes for every Grid
    
  //Save char...and set bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] | LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] | MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] | HI(icn);        
  writeData(_displaybuffer, (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID));
}

/** Clr Icon
  *
  * @param Icon icon Enums Icon has Grid position encoded in 8 MSBs, Icon pattern encoded in 24 LSBs
  * @return none
  */
void PT6315_DVDR3510::clrIcon(Icon icon) {
  int addr, icn;

   icn =        icon  & 0xFFFFFF;
  addr = (icon >> 24) & 0xFF; 
  addr = (addr - 1) * PT6315_BYTES_PER_GRID;   // 3 Bytes for every Grid
    
  //Save char...and clr bits for icon to write
  _displaybuffer[addr    ] = _displaybuffer[addr    ] & ~LO(icn);      
  _displaybuffer[addr + 1] = _displaybuffer[addr + 1] & ~MD(icn);      
  _displaybuffer[addr + 2] = _displaybuffer[addr + 2] & ~HI(icn);        
  writeData(_displaybuffer, (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID));
}


/** Set User Defined Characters (UDC)
  *
  * @param unsigned char udc_idx  The Index of the UDC (0..7)
  * @param int udc_data           The bitpattern for the UDC (16 bits)       
  */
void PT6315_DVDR3510::setUDC(unsigned char udc_idx, int udc_data) {

  //Sanity check
  if (udc_idx > (DVDR3510_NR_UDC-1)) {
    return;
  }

  // Mask out Icon bits?
  _UDC_16S[udc_idx] = udc_data & 0xFFFF;
}

/** Write a single character (Stream implementation)
  */
int PT6315_DVDR3510::_putc(int value) {
    bool validChar = false;
    short pattern  = 0x0000;
    int addr;
    
    if ((value == '\n') || (value == '\r')) {
      //No character to write
      validChar = false;
      
      //Update Cursor      
      _column = 0;
    }
    else if ((value >= 0) && (value < DVDR3510_NR_UDC)) {
      //Character to write
      validChar = true;
      pattern = _UDC_16S[value];
    }  
#if (SHOW_ASCII == 1)
    //display all ASCII characters
    else if ((value >= FONT_16S_START) && (value <= FONT_16S_END)) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[value - FONT_16S_START];
    } // else
#else
    //display only digits and hex characters
    else if (value == '-') {
      //Character to write
      validChar = true;
      pattern = C16_MIN;         
    }
    else if ((value >= (int)'0') && (value <= (int) '9')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[value - (int) '0'];
    }
    else if ((value >= (int) 'A') && (value <= (int) 'F')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[10 + value - (int) 'A'];
    }
    else if ((value >= (int) 'a') && (value <= (int) 'f')) {   
      //Character to write
      validChar = true;
      pattern = FONT_16S[10 + value - (int) 'a'];
    } //else
#endif
    if (validChar) {
      //Character to write
 
      //Translate between _column and displaybuffer entries
      //Note that the DVDR3510 has eight 14 Segment digits/grids. 
      //Some of these Grids also have icons that need to be preserved
      //_column ==  0 => Grid1 => addr = 0
      //_column ==  1 => Grid2 => addr = 3
      // ....
      //_column ==  7 => Grid8 => addr = 21            
      addr = (_column * PT6315_BYTES_PER_GRID); // 3 Bytes for every Grid;
      
      //Save icons...and set bits for character to write
      _displaybuffer[addr]   = (_displaybuffer[addr]   & MASK_ICON_GRID[_column][0]) | LO(pattern);
      _displaybuffer[addr+1] = (_displaybuffer[addr+1] & MASK_ICON_GRID[_column][1]) | MD(pattern);
      _displaybuffer[addr+2] = (_displaybuffer[addr+2] & MASK_ICON_GRID[_column][2]) | HI(pattern);

      writeData(_displaybuffer, (DVDR3510_NR_GRIDS * PT6315_BYTES_PER_GRID));
                                
      //Update Cursor
      _column++;
      if (_column > (DVDR3510_NR_DIGITS - 1)) {
        _column = 0;
      }

    } // if validChar           

    return value;
}

// get a single character (Stream implementation)
int PT6315_DVDR3510::_getc() {
    return -1;
}
#endif
